// ======================================================================
// efArgs.h - Arguments of Extension Functions.
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Changed a call to FStruc::isAtom()
//         to FStruc::isAtomValue().
// 090501: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions to supress warnings
//         under -pedantic -Wall.
// 071101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef EFARGS_HPP
#define EFARGS_HPP

#include "fsRegs.hpp"

#include <map>
#include <vector>
#include <bitset>

namespace UKernel {

// EFArgConvTable converts a argument symbol to its rightful position in EFArgs
// this way we can have arbitrary ordering of keyword arguments
// CAUTION: only insert() and clear() are redefined to update maxIdx; be
// careful to accidentally change maxIdx while invoking the methods of map<>.
class EFArgConvTable: public std::map<Symbol,FSRegisters::size_type> {
  typedef std::map<Symbol,FSRegisters::size_type> _Parent;

  FSRegisters::size_type maxIdx;

public:

  EFArgConvTable ():maxIdx(0) {}

  // the following two update maxIdx
  void insert (const Symbol &sym, FSRegisters::size_type idx);
  void clear ();

  FSRegisters::size_type readMaxIdx () const { return maxIdx; }
};

struct FSPath {
  FSRegisters::size_type fsIdx;
  Path path;

  FSPath () {}
  FSPath (FSRegisters::size_type fsIdx, const Path &path):fsIdx(fsIdx),path(path) {}
};

std::ostream &operator << (std::ostream &os, const FSPath &fsPath);

struct EFArg {
  enum EFArgFlag { tValue, tFS, tFSPath, tBool, vBool, isSet, numFlags };
  typedef std::bitset<numFlags> Flags;

  Flags flags;
  Symbol name;

  void *ptr;

  EFArg ():ptr(NULL) {}
};


class EFArgs: protected std::vector<EFArg> {
  typedef std::vector<EFArg> _Parent;

public:

  typedef _Parent::size_type size_type;

private:

  // copy 'arg' to the i-th arg
  void copy (size_type i, const EFArg &arg);
  void copy (const EFArgs &args);

public:

  size_type size () const { return _Parent::size(); }

  Value &newValue ();
  FStruc &newFS ();
  FSPath &newFSPath ();
  void newBool (bool t);

  Value &newValue (const Symbol &name);
  FStruc &newFS (const Symbol &name);
  FSPath &newFSPath (const Symbol &name);
  void newBool (const Symbol &name, bool t);

  const Symbol &readName (size_type i) const;

  EFArg::Flags &getFlags (size_type i);
  Value &getValue (size_type i);
  FStruc &getFS (size_type i);
  FSPath &getFSPath (size_type i);

  const EFArg::Flags &readFlags (size_type i) const;
  const Value &readValue (size_type i) const;
  const FStruc &readFS (size_type i) const;
  const FSPath &readFSPath (size_type i) const;
  bool readBool (size_type i) const;

  // return true and update sym, iff the i-th arg points to an atom (symbol)
  bool isAtom (size_type i, const FSRegisters &fsRegs, Symbol &sym) const;

  FSRegisters::size_type readMaxFSIdx () const;

  void clear ();
  void clear (size_type i);

  // ASSUMPTION: all entries must have names
  void convert (const EFArgConvTable &table);

  EFArgs () {}
  EFArgs (const EFArgs &args) { copy(args); }
  ~EFArgs () { clear(); }

  void operator = (const EFArgs &args) { clear(); copy(args); }
};

std::ostream &operator << (std::ostream &os, const EFArgs &args);

// ============================ inline functions ============================

inline void EFArgs::copy (const EFArgs &args) {
  size_type i,s=args.size();

  resize(s);
  for (i=0;i<s;++i) copy(i,args[i]);
}

inline void EFArgs::clear () {
  size_type i,s=_Parent::size();

  for (i=0;i<s;++i) clear(i);
  _Parent::clear();
}

inline const Symbol &EFArgs::readName (size_type i) const {
  assert(i>=0 && i<_Parent::size());
  return operator[](i).name;
}

inline EFArg::Flags &EFArgs::getFlags (size_type i) {
  assert(i>=0 && i<_Parent::size());
  return operator[](i).flags;
}

inline Value &EFArgs::getValue (size_type i) {
  assert(i>=0 && i<_Parent::size());

  EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tValue] && arg.ptr);

  return *((Value*)arg.ptr);
}

inline FStruc &EFArgs::getFS (size_type i) {
  assert(i>=0 && i<_Parent::size());

  EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tFS] && arg.ptr);

  return *((FStruc*)arg.ptr);
}

inline FSPath &EFArgs::getFSPath (size_type i) {
  assert(i>=0 && i<_Parent::size());

  EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tFSPath] && arg.ptr);

  return *((FSPath*)arg.ptr);
}

inline const EFArg::Flags &EFArgs::readFlags (size_type i) const {
  assert(i>=0 && i<_Parent::size());
  return operator[](i).flags;
}

inline const Value &EFArgs::readValue (size_type i) const {
  assert(i>=0 && i<_Parent::size());

  const EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tValue] && arg.ptr);

  return *((const Value*)arg.ptr);
}

inline const FStruc &EFArgs::readFS (size_type i) const {
  assert(i>=0 && i<_Parent::size());

  const EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tFS] && arg.ptr);

  return *((const FStruc*)arg.ptr);
}

inline const FSPath &EFArgs::readFSPath (size_type i) const {
  assert(i>=0 && i<_Parent::size());

  const EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tFSPath] && arg.ptr);

  return *((const FSPath*)arg.ptr);
}

inline bool EFArgs::readBool (size_type i) const {
  assert(i>=0 && i<_Parent::size());

  const EFArg &arg=operator[](i);

  assert(arg.flags[EFArg::isSet] && arg.flags[EFArg::tBool]);

  return arg.flags[EFArg::vBool];
}

inline bool EFArgs::isAtom (size_type i, const FSRegisters &fsRegs, Symbol &sym) const {
  assert(i>=0 && i<_Parent::size());

  const EFArg &arg=operator[](i);

  if (arg.flags[EFArg::isSet])
    if (arg.flags[EFArg::tValue]) {
      assert(arg.ptr);
      const Value &v=*((const Value*)arg.ptr);
      if (v.isAtom()) {
	sym=*v.begin();
	return true;
      }
    }
    else if (arg.flags[EFArg::tFSPath]) {
      assert(arg.ptr);
      const FSPath &fsPath=*((const FSPath*)arg.ptr);
      return fsRegs[fsPath.fsIdx].isAtomValue(fsPath.path,sym);
    }
  return false;
}

inline FSRegisters::size_type EFArgs::readMaxFSIdx () const {
  const_iterator i;
  FSRegisters::size_type fsIdx,maxFSIdx=0;

  for (i=begin();i!=end();++i)
    if (i->flags[EFArg::isSet] && i->flags[EFArg::tFSPath] &&
	(fsIdx=((const FSPath*)i->ptr)->fsIdx)>maxFSIdx) maxFSIdx=fsIdx;

  return maxFSIdx;
}

};

#endif
