// ======================================================================
// eBlock.hpp - Equation Block (*OR*, *EOR*, *CASE* and main blocks).
// 
// 022603: Benjamin Han <benhdj@cs.cmu.edu> Added _EBlockBase::containBlocks()
//         and EBlockMain::isSingleEqu(); to really make sure an EBlockMain is
//         empty, you have to call both; but for the other types of blocks
//         the first method should be sufficient.
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 110602: Benjamin Han <benhdj@cs.cmu.edu> EBlockMain constructor for type
//         4 equations now has an additional argument 'g' of type Grammar.
// 103002: Benjamin Han <benhdj@cs.cmu.edu> Bug fixed: since the introduction
//         of new operators =c', =t' and =i', the code of updating 
//         _EBlockBase::mIdxSet has not been updated until now; added a new 
//         EBlockMain constructor for type 5 equations, to deal with 
//         "(lFS lPath) op QFS" (QFS = quoted FS).
// 120601: Benjamin Han <benhdj@cs.cmu.edu> FSIdxSet is now a nested class
//         of FSRegisters; changed several int declarations into 
//         FSRegisters::size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added setTracing(), 
//         resetTracing(), setTracingOutput() and setTracingILevel() in
//         EBlockMain for printing out tracing info; added one new operator
//         Symbol::OP_ISO ("=i").
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed Symbol::OP_EQUAL to 
//         Symbol::OP_TEST.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Revised equation type 4 to be
//         the interface to an extension function (also see eFunc.*); removed
//         equation type 5-10: no more next-lex, and more flexible interfaces
//         for get-lex and get-lex-fs (these are now implemented via the
//         extension function mechanism).
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Removed _EBlockBase::eraseBlocks()
//         and EBlockMain::eraseEquations(); reset maxFSIdx and mIdxSet in
//         _EBlockBase properly in EBlockMain::addBlock(); fixed the bugs
//         in setting maxFSIdx incorrectly in the constructors of EBlockMain
//         for equation type 5-10.
// 063001: Benjamin Han <benhdj@cs.cmu.edu> Changed the behaviors of *EOR*, 
//         *OR* and *CASE* equation blocks: daughter blocks are independent
//         in that the execution result of one block won't affect the other -
//         this is done by running the daugher blocks on a local copy of the
//         global registers.
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Changed the way an FS is addressed -
//         now at compile time only the index of an FS in FSRegisters is kept,
//         and when running one needs to suppy a copy of FSRegisters.
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Added EBlockMain constructors
//         for type 9 and 10 equations.
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Added all lexical equations
//         (type 4 to 8); changed type 3 EBlockMain constructor prototype
//         - the ending argument is changed from const Symbol& to
//         Symbol::SymCode.
// 052801: Benjamin Han <benhdj@cs.cmu.edu> Now only EBlockMain can
//         be an equation-only block.
// 052001: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef EBLOCK_HPP
#define EBLOCK_HPP

#include "equation.hpp"

#include <iostream>

namespace UKernel {

// INTERNAL USE: The base class of all equation blocks.
class _EBlockBase {
public:

  typedef std::list<_EBlockBase*> _EBList;

protected:

  static bool tracingFlag;  // true iff tracing info is needed
  static int iLevel;        // indentation level for printing tracing info
  static std::ostream *outPtr;

  FSRegisters::size_type maxFSIdx;
  FSRegisters::FSIdxSet mIdxSet;  // mIdxSet keeps the set of indices of 
                                  // the FSs which are potentially modified 
                                  // by the equations

  _EBList *eblPtr;

public:

  _EBlockBase ():maxFSIdx(0),eblPtr(NULL) {}
  virtual ~_EBlockBase ();

  void clearBlocks ();  // only delete individual _EBlockBase pointers

  FSRegisters::size_type readMaxFSIdx () const { return maxFSIdx; }
  const FSRegisters::FSIdxSet &readMIdxSet () const { return mIdxSet; }

  bool containBlocks () const { return eblPtr!=NULL; }
  
  virtual bool run (FSRegisters &fsRegs) const=0;
  virtual void print (std::ostream &os) const=0;
};

// INTERNAL USE: An entry in EBlockCase::_CaseList
struct _CaseEntry {
  Value v;
  _EBlockBase::_EBList::const_iterator bi;

  _CaseEntry (const Value &v, _EBlockBase::_EBList::const_iterator bi)
    : v(v),bi(bi) {}
};

// A *CASE* equation block: if multiple cases having the same values,
// only the 1st one gets executed
class EBlockCase: public _EBlockBase {
  typedef std::list<_CaseEntry> _CaseList;
  _CaseList cl;

  FSRegisters::size_type fsIdx;
  Path path;

public:

  // don't worry about eblPtr - it's destructed in _EBlockBase
  EBlockCase (FSRegisters::size_type fsIdx, const Path &path):fsIdx(fsIdx),path(path)
  { eblPtr=new _EBList(); maxFSIdx=fsIdx; }

  void addCaseBlock (const Value &v, _EBlockBase *ebPtr);

  bool run (FSRegisters &fsRegs) const;
  void print (std::ostream &os) const;
};

// A main EBlock executes its constituents in a short-circuit-and fashion;
// the constituent is either an equation or a list of EBlocks, i.e.
// either equPtr is non-NULL or eblPtr is not-NULL but not both.
// NOTE: An empty main EBlock is legal!
// WARNING: DON'T TRY TO COPY IT
class EBlockMain: public _EBlockBase {
protected:

  _Equation *equPtr;

public:

  // tracing-related settings
  static void setTracing () { tracingFlag=true; }
  static void resetTracing () { tracingFlag=false; }
  static void setTracingOutput (std::ostream &out) { outPtr=&out; }
  static void setTracingILevel (int iLevel) { _EBlockBase::iLevel=iLevel; }

  // ============================================================
  //    Constructor for a block containing some other blocks
  // ------------------------------------------------------------
  EBlockMain ():equPtr(NULL) {}
  // ============================================================

  // ============================================================
  //    Constructors for a block containing a single equation
  // ------------------------------------------------------------
  EBlockMain (FSRegisters::size_type lFSIdx, const Path &lPath, 
	      FSRegisters::size_type rFSIdx, const Path &rPath, Symbol::SymCode opCode);
  EBlockMain (FSRegisters::size_type lFSIdx, const Path &lPath, const Value &v, 
	      Symbol::SymCode opCode);
  EBlockMain (FSRegisters::size_type lFSIdx, const Path &lPath, Symbol::SymCode symCode);
  EBlockMain (FSRegisters::size_type lFSIdx, const Path &lPath, const Symbol &funcName,
	      const EFArgs &req, const EFArgs &opt, Symbol::SymCode opCode, Grammar &g);
  EBlockMain (FSRegisters::size_type lFSIdx, const Path &lPath, const FStruc &fs, 
	      Symbol::SymCode opCode);
  // ============================================================
	      
  ~EBlockMain ();

  bool isSingleEqu () const { return equPtr!=NULL; }

  void addBlock (_EBlockBase *ebPtr);

  bool run (FSRegisters &fsRegs) const;  // run in a short-circuit-and fashion
  void print (std::ostream &os) const;
};

// An *OR* equation block
// ASSUMPTION: must contain at least one block for run() and print()
class EBlockOr: public EBlockMain {
public:
  bool run (FSRegisters &fsRegs) const;  // run in a non-short-circuit-or fashion
  void print (std::ostream &os) const;
};

// An *EOR* equation block
// ASSUMPTION: must contain at least one block for run() and print()
class EBlockEor: public EBlockMain {
public:
  bool run (FSRegisters &fsRegs) const;  // run in a short-circuit-or fashion
  void print (std::ostream &os) const;
};

// ============================ inline functions ============================

// Indented printing: you must take care of the first line (see equTst.cpp) for
// examples
inline std::ostream &operator << (std::ostream &os, const _EBlockBase &eb) {
  eb.print(os);
  return os;
}

inline void EBlockCase::addCaseBlock (const Value &v, _EBlockBase *ebPtr) {
  if (ebPtr->readMaxFSIdx()>maxFSIdx) maxFSIdx=ebPtr->readMaxFSIdx();
  mIdxSet+=ebPtr->readMIdxSet();
  eblPtr->push_back(ebPtr);
  cl.push_back(_CaseEntry(v,--(eblPtr->end())));
}

// add equation type 1
inline EBlockMain::EBlockMain (FSRegisters::size_type lFSIdx,const Path &lPath, 
			       FSRegisters::size_type rFSIdx, const Path &rPath, 
			       Symbol::SymCode opCode) {
  if (lFSIdx>maxFSIdx) maxFSIdx=lFSIdx;
  if (rFSIdx>maxFSIdx) maxFSIdx=rFSIdx;

  if (opCode!=Symbol::OP_CONSTRAIN_NOFILTER && opCode!=Symbol::OP_TEST_NOFILTER &&
      opCode!=Symbol::OP_ISO_NOFILTER)
    mIdxSet.insert(lFSIdx);
  if (opCode==Symbol::OP_REMOVE_ASSIGN || opCode==Symbol::OP_POP)
    mIdxSet.insert(rFSIdx);

  equPtr=new _EquType1(lFSIdx,lPath,rFSIdx,rPath,opCode);
}

// add equation type 2
inline EBlockMain::EBlockMain (FSRegisters::size_type lFSIdx,const Path &lPath, const Value &v, 
			       Symbol::SymCode opCode) {
  if (lFSIdx>maxFSIdx) maxFSIdx=lFSIdx;
  if (opCode!=Symbol::OP_CONSTRAIN_NOFILTER && opCode!=Symbol::OP_TEST_NOFILTER &&
      opCode!=Symbol::OP_ISO_NOFILTER)
    mIdxSet.insert(lFSIdx);
  equPtr=new _EquType2(lFSIdx,lPath,v,opCode);
}

// add equation type 3 - no need to set opCode
inline EBlockMain::EBlockMain (FSRegisters::size_type lFSIdx,const Path &lPath, Symbol::SymCode symCode) {
  if (lFSIdx>maxFSIdx) maxFSIdx=lFSIdx;
  mIdxSet.insert(lFSIdx);  // all operators act like filters
  equPtr=new _EquType3(lFSIdx,lPath,symCode);
}

// add equation type 4
inline EBlockMain::EBlockMain (FSRegisters::size_type lFSIdx, const Path &lPath, const Symbol &funcName,
			       const EFArgs &req, const EFArgs &opt, 
			       Symbol::SymCode opCode, Grammar &g) {
  FSRegisters::size_type i;

  if (lFSIdx>maxFSIdx) maxFSIdx=lFSIdx;
  if ((i=req.readMaxFSIdx())>maxFSIdx) maxFSIdx=i;
  if ((i=opt.readMaxFSIdx())>maxFSIdx) maxFSIdx=i;
  if (opCode!=Symbol::OP_CONSTRAIN_NOFILTER && opCode!=Symbol::OP_TEST_NOFILTER &&
      opCode!=Symbol::OP_ISO_NOFILTER)
    mIdxSet.insert(lFSIdx);
  equPtr=new _EquType4(lFSIdx,lPath,funcName,req,opt,opCode,g);
}

inline void EBlockMain::addBlock (_EBlockBase *ebPtr) {
  if (!eblPtr) {
    if (equPtr) {
      delete equPtr;
      equPtr=NULL;
      maxFSIdx=0;
      mIdxSet.clear();
    }
    
    // don't worry about eblPtr - it's destructed in _EBlockBase
    eblPtr=new _EBList();
  }
  
  if (ebPtr->readMaxFSIdx()>maxFSIdx) maxFSIdx=ebPtr->readMaxFSIdx();
  mIdxSet+=ebPtr->readMIdxSet();
  eblPtr->push_back(ebPtr);
}

// add equation type 5
inline EBlockMain::EBlockMain (FSRegisters::size_type lFSIdx,const Path &lPath, const FStruc &fs, 
			       Symbol::SymCode opCode) {
  if (lFSIdx>maxFSIdx) maxFSIdx=lFSIdx;
  if (opCode!=Symbol::OP_CONSTRAIN_NOFILTER && opCode!=Symbol::OP_TEST_NOFILTER &&
      opCode!=Symbol::OP_ISO_NOFILTER)
    mIdxSet.insert(lFSIdx);
  equPtr=new _EquType5(lFSIdx,lPath,fs,opCode);
}

};

#endif
