// ======================================================================
// eBlock.cpp - Equation Block (*OR*, *EOR*, *CASE* and main blocks).
// 
// 011403: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 092802: Benjamin Han <benhdj@cs.cmu.edu> Changed the call FStruc::test() to
//         FStruc::testNoFilter() in EBlockCase::run().
// 091602: Benjamin Han <benhdj@cs.cmu.edu> Minor printing correction for
//         all equation blocks: print out "()" when the content is none.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> FSIdxSet is now a nested class
//         of FSRegisters; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added code for showing the 
//         tracing info of running equations.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed calls to FStruc::equal()
//         to FStruc::test().
// 090501: Benjamin Han <benhdj@cs.cmu.edu> Fixed a bug in EBlockOr::run():
//         now it reports 'true' correctly; revised EBlockOr::run() and
//         EBlockEor::run() and they now run more efficiently by ensuring 
//         that copying FSs only happens when absolutely necessary.
// 082801: Benjamin Han <benhdj@cs.cmu.edu> Use set<>::rbegin() whenever 
//         possible.
// 080901: Benjamin Han <benhdj@cs.cmu.edu> Fixed a potential bug in
//         _EBlockOr::run() - when propagating back local regs we might point
//         before mIdxSet.begin() if mIdxSet is empty.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions after revising
//         eqaution.*.
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Removed _EBlockBase::eraseBlocks()
//         and EBlockMain::eraseEquations(); reset maxFSIdx and mIdxSet in
//         _EBlockBase properly in _EBlockBase::clearBlocks().
// 063001: Benjamin Han <benhdj@cs.cmu.edu> Changed the behaviors of *EOR*, 
//         *OR* and *CASE* equation blocks: daughter blocks are independent
//         in that the execution result of one block won't affect the other -
//         this is done by running the daugher blocks on a local copy of the
//         global registers.
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Changed the way an FS is addressed -
//         now at compile time only the index of an FS in FSRegisters is kept,
//         and when running one needs to suppy a copy of FSRegisters.
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Added EBlockMain constructors
//         for type 9 and 10 equations.
// 061801: Benjamin Han <benhdj@cs.cmu.edu> Removed extra space between right
//         parentheses when printing a *CASE*, *OR*, and *EOR* block; fixed
//         printing problem of *CASE* block.
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Added all lexical equations
//         (type 4 to 8).
// 052801: Benjamin Han <benhdj@cs.cmu.edu> Now we have indented printing
//         of an equation block thanks to indent.* in Toolbox.
// 052001: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "eBlock.hpp"

#include <sstream>

namespace UKernel {

bool _EBlockBase::tracingFlag=false;
int _EBlockBase::iLevel=0;
std::ostream *_EBlockBase::outPtr=NULL;

_EBlockBase::~_EBlockBase () {
  if (eblPtr) {
    _EBList::const_iterator ei;      
    for (ei=eblPtr->begin();ei!=eblPtr->end();++ei) delete *ei;
    delete eblPtr;
    eblPtr=NULL;
  }
}

void _EBlockBase::clearBlocks () {
  if (eblPtr) {
    _EBList::const_iterator ei;      
    for (ei=eblPtr->begin();ei!=eblPtr->end();++ei) delete *ei;
    eblPtr->clear();
    maxFSIdx=0;
    mIdxSet.clear();
  }
}

bool EBlockCase::run (FSRegisters &fsRegs) const {
  _CaseList::const_iterator ci;

  // find the first matched case
  for (ci=cl.begin();ci!=cl.end() && !fsRegs[fsIdx].testNoFilter(path,ci->v);++ci);
  if (ci!=cl.end()) return (*(ci->bi))->run(fsRegs);
  else return false;
}

void EBlockCase::print (std::ostream &os) const {
  assert(cl.size());

  _CaseList::const_iterator ci;
  std::ostringstream oss;
  int iLevel;
  
  os<<"(*CASE* ";
  if (path.size()) os<<"(X"<<fsIdx<<' '<<path<<')';
  else os<<'X'<<fsIdx;
  os<<indent(2);

  for (ci=cl.begin();ci!=cl.end();++ci,oss.str("")) {
    os<<std::endl;
    indent::print(os)<<"  ";
    oss<<'('<<ci->v<<' ';
    iLevel=oss.str().length();
    os<<oss.str()<<indent(iLevel)<<**(ci->bi)<<indent(-iLevel)<<')';
  }
  
  os<<')'<<indent(-2);
}

EBlockMain::~EBlockMain () {
  if (equPtr) {
    delete equPtr;
    equPtr=NULL;
  }
}

bool EBlockMain::run (FSRegisters &fsRegs) const {
  bool result=true;

  if (equPtr)
    if (tracingFlag) {
      result=equPtr->runTracing(fsRegs,*outPtr,iLevel);
      (*outPtr)<<std::endl;
    }
    else result=equPtr->run(fsRegs);
  else if (eblPtr) {
    _EBList::const_iterator ei;
    for (ei=eblPtr->begin();ei!=eblPtr->end() && result;++ei)
      result=(*ei)->run(fsRegs);
  }
  return result;
}

void EBlockMain::print (std::ostream &os) const {
  if (equPtr) os<<'('<<*equPtr<<')';
  else if (eblPtr) {
    _EBList::const_iterator ei;
    bool first=true;
    
    os<<'('<<indent(1);
    for (ei=eblPtr->begin();ei!=eblPtr->end();++ei)
      if (first) {
	os<<**ei;
	first=false;
      }
      else {
	os<<std::endl;
	indent::print(os)<<**ei;
      }
    
    os<<')'<<indent(-1);
  }
  else os<<"()";
}

struct _LocalFSRegsEntry {
  _EBlockBase::_EBList::const_iterator ei;
  FSRegisters fsRegs;  
};

typedef std::list<_LocalFSRegsEntry> _LocalFSRegs;

bool EBlockOr::run (FSRegisters &fsRegs) const {
  assert(eblPtr);

  _EBList::const_iterator ei;
  if ((ei=eblPtr->begin())!=eblPtr->end()) {
    bool result=false,newLocalCopy=true; // only new a local copy when it's necessary  
    _LocalFSRegs localFSRegs;
    _LocalFSRegs::iterator li=localFSRegs.begin();
    _LocalFSRegsEntry emptyEntry;
    
    // === execute all blocks on a local copy of FSs if necessary ===
    do {
      const _EBlockBase &eb=**ei;
      const FSRegisters::FSIdxSet &subMIdxSet=eb.readMIdxSet();
      
      if (subMIdxSet.size()) {
	if (newLocalCopy) {
	  localFSRegs.push_back(emptyEntry);
	  ++li;  // ASSUMPTION: list is circular!
	}
	li->ei=ei;
	li->fsRegs.copy(fsRegs,eb.readMaxFSIdx()+1);
	
	if (eb.run(li->fsRegs)) {
	  result=true;
	  if ((++ei)!=eblPtr->end()) newLocalCopy=true;
	  else break;
	}
	
	// if the sub-block fails, restore the current local copy for 
	// the next sub-block, if any
	else if ((++ei)!=eblPtr->end()) {
	  li->fsRegs.copy(fsRegs,subMIdxSet);
	  newLocalCopy=false;
	}

	else {
	  localFSRegs.erase(li);
	  break;
	}
      }
      else {
	// if no FS can be modified we run the sub-block directly on the
	// global copy
	if (eb.run(fsRegs)) result=true;
	if ((++ei)==eblPtr->end()) break;
      }
    } while (true);
    
    // === combine all successful local FS registers and propagate them back ===
    if (localFSRegs.size()) {
      FSRegisters::size_type i,mIdx,maxMIdx=*(mIdxSet.rbegin());
      std::vector<bool> v(maxMIdx+1);   // record if a particular reg is a value
      std::vector<bool> f(maxMIdx+1);   // record which reg is updated
      FSRegisters::FSIdxSet::const_iterator fi;

      // check if there's any type inconsistency between same modified
      // reg
      for (li=localFSRegs.begin();li!=localFSRegs.end();++li) {      
	const FSRegisters::FSIdxSet &eMIdSet=(*(li->ei))->readMIdxSet();

	for (fi=eMIdSet.begin();fi!=eMIdSet.end();++fi) {
	  mIdx=*fi;
	  if (!f[mIdx]) {
	    f[mIdx]=true;
	    v[mIdx]=li->fsRegs[mIdx].isValue();
	  }
	  else if (li->fsRegs[mIdx].isValue()!=v[mIdx]) return false;
	}
      }
      
      for (i=0;i<=maxMIdx;++i) f[i]=false;
      
      // begin the *OR* construction
      for (li=localFSRegs.begin();li!=localFSRegs.end();++li) {
	const FSRegisters::FSIdxSet &eMIdSet=(*(li->ei))->readMIdxSet();

	for (fi=eMIdSet.begin();fi!=eMIdSet.end();++fi) {
	  mIdx=*fi;
	  
	  assert(mIdx<=maxMIdx);
	  if (!f[mIdx]) {
	    f[mIdx]=true;
	    fsRegs[mIdx]=li->fsRegs[mIdx];
	  }
	  else fsRegs[mIdx].mergeOr(li->fsRegs[mIdx]);
	}	      
      }
    }

    return result;
  }
  else return true;
}

void EBlockOr::print (std::ostream &os) const {
  assert(eblPtr);

  _EBList::const_iterator ei;
  bool first=true;
  
  os<<"(*OR* "<<indent(6);
  for (ei=eblPtr->begin();ei!=eblPtr->end();++ei)
    if (first) {
      os<<**ei;
      first=false;
    }
    else {
      os<<std::endl;
      indent::print(os)<<**ei;
    }
  
  os<<')'<<indent(-6);
}

bool EBlockEor::run (FSRegisters &fsRegs) const {
  assert(eblPtr);

  _EBList::const_iterator ei;
  if ((ei=eblPtr->begin())!=eblPtr->end()) {
    FSRegisters localFSRegs;
    
    localFSRegs.copy(fsRegs,maxFSIdx+1);
    do {
      const _EBlockBase &eb=**ei;
      const FSRegisters::FSIdxSet &subMIdxSet=eb.readMIdxSet();
      
      if (subMIdxSet.size())
	if (eb.run(localFSRegs)) {
	  // send back the modifications
	  fsRegs.copy(localFSRegs,subMIdxSet);
	  return true;
	}
	else if ((++ei)!=eblPtr->end())
	  // restore the modified FSs
	  localFSRegs.copy(fsRegs,subMIdxSet);
	else return false;
      
      // no modification is possible - run it on the global copy directly
      else
	if (eb.run(fsRegs)) return true;
	else if ((++ei)==eblPtr->end()) return false;
      
    } while (true);
  }
  else return true;
}

void EBlockEor::print (std::ostream &os) const {
  assert(eblPtr);

  _EBList::const_iterator ei;
  bool first=true;
  
  os<<"(*EOR* "<<indent(7);
  for (ei=eblPtr->begin();ei!=eblPtr->end();++ei)
    if (first) {
      os<<**ei;
      first=false;
    }
    else {
      os<<std::endl;
      indent::print(os)<<**ei;
    }
  
  os<<')'<<indent(-7);
}

};
