#include "genkit.hpp"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/wait.h>
#include <signal.h>

#include <string.h>
#include <fstream.h>
#include <iostream.h>
#include <sstream>

#define BACKLOG 10     // how many pending connections queue will hold


int GenKit::loadGrammarFile(string filename) {
  ifstream s;
  s.open(filename.c_str());
  //cout << "parsing file";

  // Createn a scanner that reads from the input stream passed to us
  GenKitLexer lexer(s);
    
  // Create a parser that reads from the scanner
  GenKitParser parser(lexer);
  parser.setGrammar(g);
  parser.setDebug(debug);
    
  // start parsing at the rules rule
  parser.rules();
  // print out the toy grammar
  //cout<<"* The toy grammar (outside):"<<endl;
  //cout<<g<<endl<<endl;

  return 1;
}


// Can be called multiple times, order is important
int GenKit::loadLexiconFile(string filename, Symbol language) {
  ifstream s;
  s.open(filename.c_str());
  //cout << "parsing file";

  // Createn a scanner that reads from the input stream passed to us
  LexiconLexer lexer(s);
    
  // Create a parser that reads from the scanner
  LexiconParser parser(lexer);
  parser.setLexicons(lexicons, language);
  parser.setDebug(debug);
    
  // start parsing at the rules rule
  parser.lexentries();
  // print out the toy grammar
  //cout<<"* The toy grammar (outside):"<<endl;
  //cout<<g<<endl<<endl;

  s.close();
  return 1;
  

}


int GenKit::generateFromFStructure(string fstruct, vector<string> &genstrings, string tracing) {
  // Create a scanner that reads from the input stream passed to us
  string fstructstring = fstruct;
  stringstream lexstream(fstructstring);

  if (debug) cout << "\nAttempting Generation on " << fstruct <<"\n";

  FStructLexer lexer(lexstream);
  // Create a parser that reads from the scanner
  FStructParser parser(lexer);
  if (debug) cout << "Setting grammar in generate from fstructs\n"; 
  parser.setGrammar(g);

  if (debug) cout << "Set return string\n";
  parser.setReturn(genstrings);
  if (debug) cout << "Running fstructs\n";
  parser.setTracing(tracing);
  parser.setMode(modetype);
  parser.setDebug(true);
  parser.fstructs();

  if (genstrings.size() == 0) {
    cerr << "Nothing generated\n";
  }

  return genstrings.size();
}

// Takes a char array of the fstruct as input, returns
// the generated language string
int GenKit::generateFromFStructure(string fstruct, vector<string> &genstrings) {
  // Create a scanner that reads from the input stream passed to us
  string fstructstring = fstruct;
  stringstream lexstream(fstructstring);

  cout << "Attempting Generation on " << fstruct <<"\n";

  FStructLexer lexer(lexstream);
  // Create a parser that reads from the scanner
  FStructParser parser(lexer);
  cout << "Setting grammar in generate from fstructs\n"; 
  parser.setGrammar(g);

  cout << "Set return string\n";
  parser.setReturn(genstrings);
  parser.setMode(modetype);
  cout << "Running fstructs\n";
  parser.fstructs();

  return genstrings.size();
}
  

// Allows program to generate from a list of f-structures in a file,
// writing the output sentences to another file
int GenKit::generateFromFStructureFile(string filename, vector<string> &genstrings, string tracing) {
  ifstream s;
  s.open(filename.c_str());

  // Create a scanner that reads from the input stream passed to us
  if (debug) cout << "Attempting Generation on file " << filename <<"\n";

  FStructLexer lexer(s);
  // Create a parser that reads from the scanner
  FStructParser parser(lexer);
  if (debug) cout << "Setting grammar in generate from fstructs\n"; 
  parser.setGrammar(g);
  parser.setTracing(tracing);
  parser.setMode(modetype);

  if (debug) cout << "Set return string vector\n";
  parser.setReturn(genstrings);
  if (debug) cout << "Running fstructs\n";
  parser.fstructs();

  return genstrings.size();
}

void GenKit::setGrammarLoadDebug(bool mydebug) {
  debug = mydebug;
}

void GenKit::initFromFile(string initfile) {
  ifstream s;
  string line;
  char linebuffer[1024];
  string argument, file;

  s.open(initfile.c_str());

  do {
      //s.getline(linebuffer, 1024);
    s >> line;

    if (line[0] == ';') {
      // Skip
    } else if (line.find("genlang") == 0) {
	s >> argument;
	//cout << "genlang arg " << argument << endl;
	setGenerateLanguage(argument);
	cout << "Setting generation langauge to " << argument << endl;
	
    } else if (line.find("loadlex") == 0) {
	s >> argument;
	//cout << "loadlex arg " << argument << argument.find(".lex") << endl;
	file = "";
	if (argument.find(".lex") > -1) {
	    file = argument;
	} else {
	    file = argument + ".lex";
	}
	cout << "Loading " << file << endl;
	loadLexiconFile(file, genlang);
	
    } else if (line.find("compgra") == 0 ||
	       line.find("load") == 0) {
	s >> argument;
	//cout << "compgra arg " << argument << endl;
	file = "";
	if (argument.find(".gra") > -1) {
	    //cout << "has gra ending!" << endl;
	    file = argument;
	} else {
	    file = argument + ".gra";
	}
	cout << "Loading " << file << endl;
	loadGrammarFile(file);

    } else if (line.find("testgra") == 0) {
	s >> argument;
	file = "";
	if (argument.find(".gra") > -1) {
	    //cout << "has gra ending!" << endl;
	    file = argument;
	} else {
	    file = argument + ".gra";
	}
	cout << "Testing " << file << endl;
	loadGrammarFile(file);
	cout<<"* The current grammar:"<<endl;
	cout<<*g<<endl<<endl;

    } else if (line.find("tracing") == 0) {
      s >> argument;
      if (argument == "eq") {
	tracetype = "eq";
      } else if (argument == "rule") {
	tracetype = "rule";
      } else if (argument == "fs") {
	tracetype = "fs";
      } else if (argument == "off") {
	tracetype = "off";
      } else {
	tracetype = "off";
      }

    } else if (line.find("mode") == 0) {
      s >> argument;
      if (argument == "plain") {
	  modetype = "plain";
      } else if (argument == "fork") {
	  modetype = "fork";
      }
    }

    //cout << line << endl;
    line = "";
  } while (!s.eof());
  
}

int GenKit::runBatch(string infile, string outfile) {
  ofstream out;
  vector<string> genstrings;
  int i;

  generateFromFStructureFile(infile, genstrings, "off");

  out.open(outfile.c_str());
  for (i = 0; i < genstrings.size(); i++) {
    out << genstrings[i] << endl;
  }

  out.close();

}

void sigchld_handler(int s)
   {
       while(wait(NULL) > 0);
   }

void GenKit::runServer(int serverport) {
   int sockfd, new_fd;  // listen on sock_fd, new connection on new_fd
   struct sockaddr_in my_addr;    // my address information
   struct sockaddr_in their_addr; // connector's address information
   int sin_size;
   struct sigaction sa;
   int yes=1;
   vector<string> genstrings;
   string clientmsg, errormsg, resultmsg, fstruct;
   char buffer[1025];
   bool msgend;
   int bytesread, i;
   
   cout << "Running server on port " << serverport << "... " << endl;

   if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
     perror("socket");
     exit(1);
   }
   
   if (setsockopt(sockfd,SOL_SOCKET,SO_REUSEADDR,&yes,sizeof(int)) == -1) {
     perror("setsockopt");
     exit(1);
   }
   
   my_addr.sin_family = AF_INET;         // host byte order
   my_addr.sin_port = htons(serverport);     // short, network byte order
   my_addr.sin_addr.s_addr = INADDR_ANY; // automatically fill with my IP
   memset(&(my_addr.sin_zero), '\0', 8); // zero the rest of the struct
   
   if (bind(sockfd, (struct sockaddr *)&my_addr, sizeof(struct sockaddr))
                                                                       == -1) {
     perror("bind");
     exit(1);
   }
   
   if (listen(sockfd, BACKLOG) == -1) {
     perror("listen");
     exit(1);
   }
   
   sa.sa_handler = sigchld_handler; // reap all dead processes
   sigemptyset(&sa.sa_mask);
   sa.sa_flags = SA_RESTART;
   if (sigaction(SIGCHLD, &sa, NULL) == -1) {
     perror("sigaction");
     exit(1);
   }
   
   while(1) {  // main accept() loop
     sin_size = sizeof(struct sockaddr_in);
     if ((new_fd = accept(sockfd, (struct sockaddr *)&their_addr, (socklen_t *)&sin_size)) == -1) {
       perror("accept");
       continue;
     }
     printf("server: got connection from %s\n", inet_ntoa(their_addr.sin_addr));
     if (!fork()) { // this is the child process
       close(sockfd);
       // Read the f-structure
       cout << "In child\n";
       msgend = false;
       bytesread = recv(new_fd, buffer, 1024, 0);
       while (msgend == false && bytesread != 0) {
	 buffer[bytesread] = '\0';
	 cout << "Buffer " << buffer << endl;
	 clientmsg += buffer;
	 for (i = 0; i < bytesread; i++) {
	   if (buffer[i] == '\n') {
	     cout << "Found newline.  Should be done now" << endl;
	     msgend = true;
	     break;
	   }
	 }
	 if (msgend == true) break;
	 bytesread = recv(new_fd, buffer, 1024, 0);
       }

       // Generate from an F-structure
       if (clientmsg.find("genf\t") == 0) {
	 fstruct = clientmsg.substr(5, clientmsg.length() - 5);
	 // process the fstructure(s)
	 generateFromFStructure(fstruct, genstrings);
	 // Return results
	 if (genstrings[0].length() > 0) {
	   resultmsg = "res\t" + genstrings[0] + "\n";
	   if (send(new_fd, resultmsg.c_str(), resultmsg.length(), 0) == -1)
	     perror("send");
	 } else {
	   errormsg = "err\tNothing generated.\n";
	   if (send(new_fd, errormsg.c_str(), errormsg.length(), 0) == -1)
	     perror("send");
	 }
       }

       close(new_fd);
       exit(0);
     }
     close(new_fd);  // parent doesn't need this
   }

}


void GenKit::runInteractive() {
  bool finish = false;
  string argument, file, fstruct, output, word, cat;
  string command;
  vector<string> genstrings;
  int i, parcount;
  Lexicon *lexiconPtr = &lexicons.findLexicon(Symbol(genlang));
  FStruc result(Symbol("RESULT"));


  while (finish == false) {
    cout << "> ";
    cin >> command;
    if (command == "quit" ||
	       command == "q" ||
	       command == "exit") {
      finish = true;
    } else if (command == "help") {
      cout << "Possible Commands:\n";
      cout << "compgra grammarfile: Load a generation grammar in grammarfile (sans .gra ending)" << endl;
      cout << "load grammarfile   : Same as compgra" << endl;
      cout << "loadlex lexiconfile: Load a lexicon file (without .lex ending)" << endl;
      cout << "looklex word       : Look up a work in the currently loaded lexicon" << endl;
      cout << "tracing (rule|eq|fs|off)  : Tracing just rules, rules&f-structs, or turn off" << endl;
      cout << "mode (plain|fork)  : Possible generation modes" << endl;
      cout << "debug (on|off)     : If grammar not loading, will show details of loading" << endl;
      cout << "generate fstruct   : Generate text from an f-structure" << endl;
      cout << "generatef filename : Generate text from f-structure(s) in filename" << endl;
      cout << "exit|quit|q        : Quit the program" << endl;
    } else if (command == "debug") {
      cin >> argument;
      if (argument == "on") {
	debug = true;
      } else {
	debug = false;
      }
    } else if (command == "tracing") {
      cin >> argument;
      if (argument == "eq") {
	tracetype = "eq";
      } else if (argument == "rule") {
	tracetype = "rule";
      } else if (argument == "fs") {
	tracetype = "fs";
      } else if (argument == "off") {
	tracetype = "off";
      } else {
	tracetype = "off";
      }

    } else if (command.find("mode") == 0) {
      cin >> argument;
      if (argument == "plain") {
	  modetype = "plain";
      } else if (argument == "fork") {
	  modetype = "fork";
      }

    } else if (command.find("loadlex") == 0) {
      cin >> argument;
      file = "";
      if (argument.find(".lex") > -1) {
	file = argument;
      } else {
	file = argument + ".lex";
      }
      cout << "Loading " << file << endl;
      loadLexiconFile(file, Symbol("ENG"));

    } else if (command.find("looklex") == 0) {
      cin >> word;
      for (i = 0; i < word.length(); i++) {
	word[i] = toupper(word[i]);
      }  
      
      cin >> cat;
      for (i = 0; i < cat.length(); i++) {
	cat[i] = toupper(cat[i]);
      }  
      cout << "Lexicon Entry: " << endl << lexiconPtr->findLex(Symbol(word), Symbol(cat), result) << endl;

    } else if (command.find("compgra") == 0 ||
	       command.find("load") == 0) {
      cin >> argument;
      file = "";
      if (argument.find(".gra") > -1) {
	file = argument;
      } else {
	file = argument + ".gra";
      }
      cout << "Loading " << file << endl;
      loadGrammarFile(file);

    } else if (command.find("generatef") == 0) {
      cin >> file;
      genstrings.clear();
      generateFromFStructureFile(file, genstrings, tracetype);
      for (i = 0; i < genstrings.size(); i++) {
	cout << genstrings[i] << endl << endl;
      }
      
    } else if (command.find("generate") == 0) {
      fstruct.erase();
      genstrings.clear();
      parcount = 0;
      do {
	cin >> argument;
	fstruct += argument + " ";
	for (i = 0; i < argument.length(); i++) {
	    if (argument[i] == '(') {
		parcount++;
	    } else if (argument[i] == ')') {
		parcount--;
	    }
	}
      } while (parcount > 0);
      genstrings.clear();
      generateFromFStructure(fstruct, genstrings, tracetype);
      for (i = 0; i < genstrings.size(); i++) {
	cout << genstrings[i] << endl;
      }
    } else {
      cout << "Command not recognized.  Type \"help\" for commands\n";
    }

  }

}
