// This source file defines methods of the class
//    CParam
//

#include <iostream>
#include <fstream>
#include <iomanip>
#include <sstream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include <Param.hh>

//using namespace std;

//_________________________________________________________
//
// CParam
//_________________________________________________________
  /* Return values for Interpret                */
  const int CParam::OK                              =   1;
  const int CParam::NOT_OK                          =   0;
 
  /* Return values for Interpret_returnError    */
  const int CParam::PROBLEM_UNDEFINED_PROBLEM       =   0; 
  const int CParam::PROBLEM_MISSING_HYPHEN          =  -1;
  const int CParam::PROBLEM_UNKNOWN_KEYWORD         =  -2;
  const int CParam::PROBLEM_UNKNOWN_FLAG            =  -3;
  const int CParam::PROBLEM_INVALID_VALUE           =  -4;
  const int CParam::PROBLEM_MISSING_VALUE           =  -5;
  const int CParam::PROBLEM_MISSING_PARAMETER       =  -6;
  const int CParam::PROBLEM_CANNOT_OPEN_FILE        =  -7;
  const int CParam::PROBLEM_MISSING_VALUE_IN_FILE   =  -8;
  const int CParam::PROBLEM_UNKNOWN_KEYWORD_IN_FILE =  -9;
  const int CParam::PROBLEM_GENERAL_PROBLEM_IN_FILE = -10;
  const int CParam::PROBLEM_PARAM_FILE_LOOP         = -11;


CParam::CParam ()
{
  NParamDef = 0;
  ParamList = 0;
  Tail      = 0;
  argv0     = 0;
  no_of_kwdless_par = 0;
  ParamFileList = NULL;
}
//_________________________________________________________

CParam::~CParam ()
{
  ParamDefElem *tmp;

  for (ParamDefElem *elem = ParamList;
       elem != 0;)
    {
      delete (elem->definition);
      tmp = elem;
      elem = elem->next;
      delete tmp;
    }

  // clean up ParamFileList
  PParamFileElem deletable = ParamFileList;
  while (deletable != NULL) {
      ParamFileList = deletable->next;
      delete[] deletable->name;
      delete deletable;
      deletable = ParamFileList;
  } // endwhile: delete param file list
}
//_________________________________________________________
// already instantiated CParamDef

int CParam::AddParamDef( CParamDef *def )
{
  char c, checkC, *cString, *checkCString;

  ParamDefElem *dummy;

  // Check for reserved flags 
  c = '\0';
  if ( def->understands ('h') )
    c = 'h';
  else  if ( def->understands ('f'))
    c = 'f';
 
  // Check for reserved keywords
  cString = 0;
  if ( def->understands("help") )
    cString = "help";
  else if ( def->understands("ParameterFile") )
    cString = "ParameterFile";
  
  // Check for already used flags or keywords
  checkC = def->getFlag();
  checkCString = strdup(def->getKeyword());
  for (dummy = ParamList; dummy != 0; dummy = dummy->next)
    {
      if (dummy->definition->understands(checkC))
	{
	  c = checkC;
	}
      if (dummy->definition->understands(checkCString))
	{
	  cString = checkCString;
	}
    }
  // warn if check was positive
  if (c)
    {	 
	  //      std::cerr << std::endl << rcsid_param_cc << std::endl;
      std::cerr << std::endl << "CParam.AddParamDef :";
      std::cerr << std::endl << " Internal warning: the flag '" << c;
      std::cerr << "' is reserved for special purposes or has";
      std::cerr << std::endl << " been used for another parameter.";
      std::cerr << std::endl << " It should not be redefined.";
      std::cerr << std::endl << "          BEWARE OF UNPREDICTIBLE BEHAVIOUR!" << std::endl << std::endl;
#ifdef CPARAM_EXIT_ON_WARNINGS
      exit(-1);
#endif
      return 0;
    }
  if (cString)
    {
	  //      std::cerr << std::endl << rcsid_param_cc << std::endl;
      std::cerr << std::endl << "CParam.AddParamDef :";
      std::cerr << std::endl << " Internal warning: the keyword '" << cString;
      std::cerr << "' is reserved for special purposes or has";
      std::cerr << std::endl << " been used for another parameter.";
      std::cerr << std::endl << " It should not be redefined.";
      std::cerr << std::endl << "          BEWARE OF UNPREDICTIBLE BEHAVIOUR!" << std::endl << std::endl;
#ifdef CPARAM_EXIT_ON_WARNINGS
      exit(-1);
#endif
      return 0;
    }

  free(checkCString);  // it was created via strdup()
                       // DO NOT TRY TO USE cString AFTER THIS - 
                       // IT HAS BEEN free()'ed BY THIS TOO !

  if (! (dummy = new ParamDefElem) )
    return 0;
  
  dummy->next = 0;
  dummy->definition = def;

  if (NParamDef == 0) 
    {
      ParamList = dummy;
      Tail = ParamList;
    }
  else
    {
      Tail->next = dummy;
      Tail = Tail->next;
    }
  NParamDef++;
  return 1;
}
//_________________________________________________________
// general

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 types partypeInit,
			 char *descriptionInit,
			 char *errordescInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CParamDef *ParamDef;

  if (! (ParamDef = 
	 new CParamDef (keywordInit, 
			flagInit, 
			partypeInit,
			partypenameInit,
			descriptionInit, 
			errordescInit,
			optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// char

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 char  *varInit,
			 char  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CCharParamDef *ParamDef;

  if (! (ParamDef = 
	 new CCharParamDef (keywordInit, 
			    flagInit, 
			    varInit,
			    defltInit,
			    partypenameInit,
			    descriptionInit, 
			    optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// int

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 int  *varInit,
			 int  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CIntParamDef *ParamDef;

  if (! (ParamDef = 
	 new CIntParamDef (keywordInit, 
			   flagInit, 
			   varInit,
			   defltInit,
			   partypenameInit,
			   descriptionInit, 
			   optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// unsigned int

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 unsigned int  *varInit,
			 unsigned int  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CUnsignedintParamDef *ParamDef;

  if (! (ParamDef = 
	 new CUnsignedintParamDef (keywordInit, 
				   flagInit, 
				   varInit,
				   defltInit,
				   partypenameInit,
				   descriptionInit, 
				   optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// short int

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 short int  *varInit,
			 short int  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CShortintParamDef *ParamDef;

  if (! (ParamDef = 
	 new CShortintParamDef (keywordInit, 
				flagInit, 
				varInit,
				defltInit,
				partypenameInit,
				descriptionInit, 
				optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// unsigned short int

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 unsigned short int  *varInit,
			 unsigned short int  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CUnsignedshortParamDef *ParamDef;

  if (! (ParamDef = 
	 new CUnsignedshortParamDef (keywordInit, 
				     flagInit, 
				     varInit,
				     defltInit,
				     partypenameInit,
				     descriptionInit, 
				     optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// long int

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 long int  *varInit,
			 long int  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CLongintParamDef *ParamDef;

  if (! (ParamDef = 
	 new CLongintParamDef (keywordInit, 
			       flagInit, 
			       varInit,
			       defltInit,
			       partypenameInit,
			       descriptionInit, 
			       optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// unsigned long int

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 unsigned long int  *varInit,
			 unsigned long int  defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CUnsignedlongParamDef *ParamDef;

  if (! (ParamDef = 
	 new CUnsignedlongParamDef (keywordInit, 
				    flagInit, 
				    varInit,
				    defltInit,
				    partypenameInit,
				    descriptionInit, 
				    optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// string

int CParam::AddParamDef (char *keywordInit,  
			 char flagInit,
			 char  **varInit,
			 char  *defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CStringParamDef *ParamDef;

  if (! (ParamDef = 
	 new CStringParamDef (keywordInit, 
			      flagInit, 
			      varInit,
			      defltInit,
			      partypenameInit,
			      descriptionInit, 
			      optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// double

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 double  *varInit,
			 double   defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CDoubleParamDef *ParamDef;

  if (! (ParamDef = 
	 new CDoubleParamDef (keywordInit, 
			      flagInit, 
			      varInit,
			      defltInit,
			      partypenameInit,
			      descriptionInit, 
			      optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// float

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 float  *varInit,
			 float   defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CFloatParamDef *ParamDef;

  if (! (ParamDef = 
	 new CFloatParamDef (keywordInit, 
			     flagInit, 
			     varInit,
			     defltInit,
			     partypenameInit,
			     descriptionInit, 
			     optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________
// bool

int CParam::AddParamDef (char *keywordInit,    
			 char flagInit,
			 bool  *varInit,
			 bool   defltInit,
			 char *descriptionInit,
			 bool optInit, char *partypenameInit,
			 int acceptWithoutKeywordOrder)
{
  CBoolParamDef *ParamDef;

  if (! (ParamDef = 
	 new CBoolParamDef (keywordInit, 
			    flagInit, 
			    varInit,
			    defltInit,
			    partypenameInit,
			    descriptionInit, 
			    optInit, acceptWithoutKeywordOrder)) )
    return 0;
  else
    return AddParamDef ( ParamDef );
}
//_________________________________________________________


int CParam::Interpret_returnError (char **argv, int argc)
{
  bool
    stopInterpretation = false,
    skipAnotherArgument = false,
    recognized = false,
    noMoreFlags = false; // '--' was given in the command line -> dont interpret the rest
                         // of the line as flags/keyword
  
  char c0,c1;        // First and second char of the present string

  char 
    *Keyword=0,
    *Value=0,
    *WholeParameter=0,
    *FirstEqualityInParameter=0;
   
  TParamDefPtr par;
  ParamDefElem *elem;

  int
    arg=1,
    pos=0,
    errNo=0,
    paramNo=0; // Nr of present non-kwd parameter
   


  CheckParameters();
  argv0 = argv[0];
  while ((arg < argc) && (!stopInterpretation))
    {
      WholeParameter = strdup(argv[arg]);
      // As we change it, we should use a copy
      assert(WholeParameter);
      c0 = WholeParameter[0];
      if ( (c0 != '-') || noMoreFlags)
	{
	  // Case One: parameter doesn't start with "-" -> 
	  // Find corresponding param. with a suitable number
	  paramNo++;
	  recognized = false;
	  for (elem = ParamList; ( (elem != 0) && (!recognized) );
	       elem = elem->next)
	    {
	      par = elem->definition;
	      if (par->myKWless_param(paramNo) == true)
		{
		  // got it
	          if (!par->dealWith(WholeParameter))
		    {
		      errNo = showError(PROBLEM_INVALID_VALUE, 
					par->getPartypeName(), 
					WholeParameter);
		      return errNo;
		    }
		  else
		    {
		      recognized = true;
		    }
		}
	    }
	  if (!recognized)
	    {
	      // couldnt use parameter or couldnt find corresponding
	      // CParamDef instance -> error.
	      errNo = showError(PROBLEM_MISSING_HYPHEN, WholeParameter, "");
	      free(WholeParameter);
	      return errNo;
	    }
	  else
	    {
	      arg++;
	    }
	}
      else
	{
	  // starting "-" is given, o.k.
	  pos = 1;
	  c1 = WholeParameter[pos];
	  if (c1 == '-')
	    {
	      // Case Two: GNU-Style Parameter: "--"
	      // Find '=' if present
	      FirstEqualityInParameter = strchr(WholeParameter, '=');
	      if (FirstEqualityInParameter != 0)
		{
		  Keyword = &(WholeParameter[2]);
		  Value = &(FirstEqualityInParameter[1]); // After '=' is value
		  *FirstEqualityInParameter = '\0';     // Thus ending Keyword
		}
	      else
		{
		  Keyword = &(WholeParameter[2]);
		  Value = argv[arg + 1];
		}
	      recognized = false;
	      // find ParamDef instance with keywd
	      for (elem = ParamList; ( (elem != 0) && (!stopInterpretation) );
		   elem = elem->next)
		{
		  par = elem->definition;
		  if (par->understands(Keyword) && !par->no_keyword())
		    {
		      skipAnotherArgument = par->skipAfter();
		      recognized = true;
		      if (!par->dealWith(Value))
			{
			  char WrongParameter[5];
			  sprintf(WrongParameter, "--%s",Keyword);
			  errNo = showError(PROBLEM_INVALID_VALUE, WrongParameter, Value);
			  free(WholeParameter);
			  return errNo;
			}
		    }
		}
	      if (!recognized)
		{
		  if (!strcmp(Keyword, "help"))
		    {
		      usageInfo(argv0);
		      skipAnotherArgument = false;
		    }
		  else
		    if (!strcmp(Keyword, "ParameterFile"))
		      {
			errNo = read_parameter_file(Value);
			if (errNo < 0)
			  {
			    free(WholeParameter);
			    return errNo;
			  }
			skipAnotherArgument = true;
		      }
		    else
		      {
			if (Keyword[0] == '\0')
			  {
			    // '--' -> interpret the rest as kwdless parameters
			    skipAnotherArgument = false;
			    noMoreFlags = true;
			  }
			else
			  {
			    errNo = showError(PROBLEM_UNKNOWN_KEYWORD, Keyword, Value);
			    free(WholeParameter); 
			    // Keyword and Value are parts of WholeParameter
			    return errNo;
			  }
		      }
		}
	      if (skipAnotherArgument && (FirstEqualityInParameter == 0))
		arg += 2; // non-bool option and no '=' in Param -> skip twice
	      else
		arg += 1;
	    }
	  else
	    {
	      // Case Three: short options
	      // Before: ARE there any options?
	      // No: Show error
	      if (c1 == 0)
		{
		  errNo = showError(PROBLEM_UNKNOWN_FLAG,"","");
		  // This should be self-explaining... if not, add
		  // another error message.
		  free(WholeParameter);
		  return errNo;	
		}
	      // First: evaluate all bool options if present;
	      recognized = true; // to start while-part
	      while (recognized && (c1 != '\0') )
		{
		  recognized = false; // we haven't found any yet
		  for (elem = ParamList; (elem != 0) && (!recognized); elem = elem ->next)
		    {
		      par = elem -> definition;
		      if ( par->understands(c1) && !par->skipAfter() )
			{
			  if (!par->dealWith(""))
			    {
			      std::cerr << "\nThis shoultn't have been happened...\n"
				   << "An empty value of a bool flag wasn't accepted!";
			      exit(-1);
			    }
			  else
			    {
			      recognized = true;
			    }
			}
		    }
		  if (recognized)
		    {
		      // get next char
		      pos++;
		      c1 = WholeParameter[pos];
		    }
		}
	      // Check for -h
	      if (c1 == 'h')
		{
		  usageInfo(argv[0]);
		}
	      // Next: evaluate the remaining non-Bool option (if there are any)
	      // (only one in each argv[arg])
	      if (c1 != '\0')
		{
		  Value = &(WholeParameter[pos+1]);
		  if (Value[0] == '\0')
		    {
		      Value = argv[arg+1];
		      pos = 0;
		      skipAnotherArgument = true;
		    }
		  else
		    {
		      skipAnotherArgument = false;
		    }
		  for (elem = ParamList; 
		       ( (elem != 0) && (!recognized) ); elem = elem ->next)
		    {
		      par = elem->definition;
		      if (par->understands(c1) && !par->no_keyword() && par->skipAfter())
			// actually, the last condition should always be true
			{
			  if (!par->dealWith(Value))
			    {
			      char WrongParameter[5]; // init
			      sprintf(WrongParameter,"-%c", c1);
			      errNo = showError(PROBLEM_INVALID_VALUE, WrongParameter, Value);
			      free(WholeParameter);
			      return errNo;
			    }
			  else
			    {
			      recognized = true;
			    }
			}
		    }
		  // Check for '-f'
		  if (c1 == 'f')
		    {
		      errNo = read_parameter_file(Value);
		      if (errNo < 0)
			{
			  free(WholeParameter);
			  return errNo;
			}
		      recognized = true;
		    }
		  // If still any options present, set "error"
		  if (!recognized)
		    {
		      char WrongParameter[5]; // init
		      sprintf(WrongParameter, "-%c", c1);
		      errNo = showError(PROBLEM_UNKNOWN_FLAG, WrongParameter, "");
		      free(WholeParameter);
		      return errNo;
		    }
		  else
		    {
		      if (skipAnotherArgument)
			{
			  arg+=2;
			}
		      else
			{
			  arg+=1;
			}
		    }
		}
	      else
		{
		  arg++;
		}
	    }
	} // if c0!='-'
      free(WholeParameter);
    } // while arg<argc
  for ( elem = ParamList; (elem != 0); elem = elem->next)
    {
      par = elem->definition;
      if (!par->isSatisfied())
	{
	  if (!par->no_keyword())    // defined by keyword -> show it
	    errNo = showError(PROBLEM_MISSING_PARAMETER, par->getKeyword(), "");  
	  else                            // defined by parameter type name -> show it
	    errNo = showError(PROBLEM_MISSING_PARAMETER, par->getPartypeName(), "");
	  return errNo;
	}
    }
  return 1; // if it hasn't been ok, there would have been an "exit(...)"
}

// _______________________________________________________
// Interpret: returns CParam.OK (1) / CPARAM.NOT_OK (0)
int CParam::Interpret (char **argv,
		 int argc)
{
  int result;
  result = Interpret_returnError(argv, argc);
  if (result > 0) return OK;      /* 1 */
  else return NOT_OK;             /* 0 */
}


// _______________________________________________________
// Interpret: returns CParam.OK (1) / CPARAM.NOT_OK (0)
int CParam::Interpret_exitOnError (char **argv,
		 int argc)
{
  int result;
  result = Interpret_returnError(argv, argc);
  if (result > 0) return OK;      /* 1 */
  else {
    std::cerr << "...exiting." << std::endl;
    exit(-1);                         /* exit */
  }
}


/*
  Read parameters from parameter file. Returns negative value on error.

  Format:

  <paramFile> ::= (<line>\n)*
  <line>      ::= <WS>*<key>(<WS>|"="|"\0")<WS>*<value>
  <WS>        ::= ("\t" | " " | "\n")
*/
int CParam::read_parameter_file (const char *Filename)
{
    const int BufferSize = 1024;
    char Line[ BufferSize ];        // buffers for reading file
    char Buffer[ BufferSize ];      // buffer for altering (i.e. marking the sub-strings)
    char *strKeyword, *strValue;    // pointer on the key / value sub-strings
    
    int recognized;                 // flag set to 1 as soon as matching param def found
    
    TParamDefPtr par;               // a parameter definition struct
    ParamDefElem *elem;             // element of the parameter list
    
    PParamFileElem iFile;         // dummy for insertion into param file list
    
    
    // open parameter file
    std::ifstream ParFile ( Filename, std::ios::in );
    
    // Check error while opening file
    if ( !ParFile ) {
	showError(PROBLEM_CANNOT_OPEN_FILE, Filename, "");
	return PROBLEM_CANNOT_OPEN_FILE;
    } else {
	// file successfully opened
	
	// insert file into parameter file list
	iFile = new TParamFileElem;
	iFile->next = ParamFileList;
	ParamFileList = iFile;
	// memorize name
	iFile->name = new char[strlen(Filename)+1];
	strcpy(iFile->name, Filename);
	    
	// parse lines
	while ( ParFile.getline ( Line, BufferSize ) ) {
	    // skip comments
	    if (Line[0] == '#')
		continue;
	    
	    strcpy( Buffer, Line );
	    recognized = 0;          // reset flag
	    
	    // Pointer on keyword and value substring
	    strKeyword = Buffer;
	    strValue = Buffer;
	    // TODO: Why search for "\n" ? We already read only 1 line!
	    // Find first non-whitechar, the beginning of the keyword
	    while ((*strKeyword == ' ') ||
		   (*strKeyword == '\t') ||
		   (*strKeyword == '\n'))  strKeyword++;
	    strValue = strKeyword + 1;




	    // Find end of keyword (use strValue, unused yet)
	    while ( (*strValue != ' ') &&
		    (*strValue != '\0') &&
		    (*strValue != '\t') &&
		    (*strValue != '\n') &&
		    (*strValue != '=')) strValue++;
	    
	    // terminate keyword
	    *strValue = '\0';
	    strValue++;
	    
	    // find beginning of value
	    while ( (*strValue == ' ') ||
		    (*strValue == '\t') ||
		    (*strValue == '\n') ||
		    (*strValue == '=')) strValue++;

	    // find end of value (to handle end-of-line whitespace)
	    char *strValueEnd = strValue;
	    while ( (*strValueEnd != ' ') &&
		    (*strValueEnd != '\0') &&
		    (*strValueEnd != '\t') &&
		    (*strValueEnd != '\n')) strValueEnd++;
	    *strValueEnd = '\0';
	    
	    if (!strcmp(strKeyword, "")) {
		// Skip empty keyword
		continue;
	    } else {
		// Keyword not empty
		
		// search for corresponding keyword in parameter list
		for ( elem = ParamList; elem != 0; elem = elem->next) {
		    par = elem->definition;
		    if ( par->understands (strKeyword) ) {
			// corresponding param list entry found!
			recognized = 1;  // set flag
			if ( (strValue == NULL) && (par->skipAfter()) ) { 
			    // error: empty value
			    showError(PROBLEM_MISSING_VALUE_IN_FILE, Filename, Line);
			    return PROBLEM_MISSING_VALUE_IN_FILE;
			} else {
			    if ( !(par->dealWith( strValue , iFile->name ) ) ) {
				// error while parsing value
				showError(PROBLEM_GENERAL_PROBLEM_IN_FILE, Filename, Line);
				return PROBLEM_GENERAL_PROBLEM_IN_FILE;
			    } else {
				// SUCCESS!!!
			    } // endif: error
			} // endif: empty value
			break; // only one parameter definition can match
		    } // endif: parameter def matches
		} // endfor: search matching param def
		
		// parameter not handled yet?
		// Always true, due to "break" above
		if ( !recognized ) {
		    // Check whether another include file is given.
		    if (!strcmp(strKeyword,"ParameterFile")) {
			// OK, nested parameter file found. Check if new param file
			// already exists in our param file list
			PParamFileElem oldFile = ParamFileList;
			for ( ; (oldFile != NULL) && (strcmp(oldFile->name,strValue) != 0);
			      oldFile = oldFile->next) ;
			if (oldFile != NULL) {
			    // param file not yet seen
			    int rc; // return code
			    rc = read_parameter_file(strValue);
			    // only report errors, otherwise generate own rc later
			    if (rc <= 0) {
				return rc;
			    } // endif: error in nested param file call
			} else {
			    // print error msg and go on
			    showError(PROBLEM_PARAM_FILE_LOOP, Filename, Line);
			} // endif: nested param file already seen?
		    } else {
			// param key neither matched nor references other param file -> error
			showError(PROBLEM_UNKNOWN_KEYWORD_IN_FILE, Filename, Line);
			return PROBLEM_UNKNOWN_KEYWORD_IN_FILE;
		    } // endif: nested param file?
		} // endif: parameter handled?
	    } // endif: empty key
	} // endwhile: consume param file lines
	return CParam::OK;
    } // endif: error opening file?
} // CParam::read_parameter_file
//_________________________________________________________


int CParam::showError(int errorNo, const char *FirstString, char *SecondString)
{
  //  std::cerr << std::endl << rcsid_param_cc << std::endl << std::endl;
  switch (errorNo) {
  case PROBLEM_UNDEFINED_PROBLEM: 
    std::cerr << "Undefined error (0), please check program source code";
    break;
  case PROBLEM_MISSING_HYPHEN:
    std::cerr << "Invalid argument '" << FirstString << "'."; 
    std::cerr << std::endl <<"Each parameter must start with '-' or '--' resp.!";
    break;
  case PROBLEM_UNKNOWN_KEYWORD:
    std::cerr << "Unknown keyword '" << FirstString << "'."; 
    break;
  case PROBLEM_UNKNOWN_FLAG:
    std::cerr << "Unknown flag '" << FirstString << "'."; 
    break;
  case PROBLEM_INVALID_VALUE:
    std::cerr << "Invalid value '" << SecondString << "' at parameter '";
    std::cerr << FirstString << "'."; 
    break;
  case PROBLEM_MISSING_VALUE:
    std::cerr << "Missing value at keyword '";
    std::cerr << FirstString << "'."; 
    break;
  case PROBLEM_MISSING_PARAMETER:
    std::cerr << "Parameter '";
    std::cerr << FirstString << "' required."; 
    break;
  case PROBLEM_CANNOT_OPEN_FILE:
    std::cerr << "Cannot open file '";
    std::cerr << FirstString << "'."; 
    break;
  case PROBLEM_MISSING_VALUE_IN_FILE:
    std::cerr << "Missing value in file '";
    std::cerr << FirstString << "' at line:\n'" << SecondString << "'."; 
    break;
  case PROBLEM_UNKNOWN_KEYWORD_IN_FILE:
    std::cerr << "Unknown keyword in file '";
    std::cerr << FirstString << "' at line:\n'" << SecondString << "'."; 
    break;  
  case PROBLEM_GENERAL_PROBLEM_IN_FILE:
    std::cerr << "General problem in file '";
    std::cerr << FirstString << "' at line:\n'" << SecondString << "'."; 
    break;  
  case PROBLEM_PARAM_FILE_LOOP:
    std::cerr << "Loop detected in nested parameter file call in file '";
    std::cerr << FirstString << "' at line:\n'" << SecondString << "'.";
#ifndef EXIT_ON_WARNINGS
    std::cerr << "\nSkipping parameter file call.";
#endif 
    break;  
  default: 
    std::cerr << "Unknown error (" << errorNo << "), please check program source code";
    break;
  }
  std::cerr << std::endl << "Get help with '" << argv0 << " --help'" << std::endl << std::endl;
#ifdef EXIT_ON_WARNINGS
  exit(-1);
#endif
  return errorNo;
}

// _____________________________________________________________

bool CParam::CheckParameters()
{
  bool is_ok=true;
  bool optional_kwdless=false;
  TParamDefPtr par;
  ParamDefElem *elem;

  no_of_kwdless_par = 0;
  int i,j;
  // count kwdless parameters
  for (elem = ParamList; elem != 0; elem = elem->next)
    {
      par = elem->definition;
      if (par->no_keyword()) no_of_kwdless_par++;
    }

  // check if each pos. for kwdless parameters is filled
  for (i=1; i<=no_of_kwdless_par; i++)
    {
      j=0;
      for (elem = ParamList; elem != 0; elem = elem->next)
	{
	  par = elem->definition;
	  if (par->myKWless_param(i)) {
	    j++;
	    if (par->optional())
	      {
		optional_kwdless = true;
	      }
	    else 
	      {
		// Behind the first optional kwdless parameter
		// MUST NOT come a obligatory one -> check this.
		if (optional_kwdless)
		  {
			//		    std::cerr << std::endl << rcsid_param_cc << std::endl;
		    std::cerr << std::endl << "CParam::CheckParameters():";
		    std::cerr << std::endl << "Internal warning: An obligatory kwdless parameter (";
		    std::cerr << par->getPartypeName() << ") is located  ";
		    std::cerr << std::endl << "behind an optional one.";
		    std::cerr << std::endl << "          BEWARE OF UNPREDICTIBLE BEHAVIOUR!" << std::endl << std::endl;
		    is_ok=false;
#ifdef EXIT_ON_WARNINGS
		    exit(-1);
#endif
		  }
	      }
	  }
	}
      if (j!=1)
	{
	  //	  std::cerr << std::endl << rcsid_param_cc << std::endl;
	  std::cerr << std::endl << "CParam::CheckParameters():";
	  std::cerr << std::endl << "Internal Warning: Position nr. " << i << " for keyword-less parameters";
	  if (j==0)
	    std::cerr << std::endl << "is not used by any parameter.";
	  else
	    std::cerr << std::endl << "is used by " << j << " parameters." << std::endl;
	  std::cerr << std::endl << "          BEWARE OF UNPREDICTIBLE BEHAVIOUR!" << std::endl << std::endl;
	  is_ok=false;
#ifdef EXIT_ON_WARNINGS
	  exit(-1);
#endif

	}
    }
  return is_ok;
}

  


// _____________________________________________________________
void CParam::usageInfo(char *myName)   /* particular procedure for better readability */
{
  ParamDefElem *elem;
  TParamDefPtr par;
  int boolHyphenAlreadyPrinted=0; // print the - only if there is at least
                                  // one - bool parameter
  int i;

  // print version info
#ifdef PACKAGE_STRING
  cerr << PACKAGE_STRING << endl;
#ifdef PACKAGE_BUGREPORT
  cerr << "Send bug reports to <" << PACKAGE_BUGREPORT << ">" << endl;
#endif

#endif // PACKAGE_STRING defined by autoconf

  std::cerr << "\nusage: " << myName; 

  // First print all required parameters with keywords:  
  for ( elem = ParamList; elem != 0; elem = elem->next)
    {
      par = elem->definition;
      if ((! par->optional()) && (par->myKWless_param(-1))) 
	{
	  par->usageInfo();
	}
    }
 // Then print all boolean options:  
  for ( elem = ParamList; elem != 0; elem = elem->next)
    {
      par = elem->definition;
      if ( par->optional() && (! par->skipAfter()) )
	{
	  if (!boolHyphenAlreadyPrinted)
	    {
	      // this is the first bool param.: print "-["
	      std::cerr << " [-";
	      boolHyphenAlreadyPrinted = -1;
	    }
	  par->usageInfo();
	}
    }
  if (boolHyphenAlreadyPrinted)
    {
      // and end this with the closing "]"
      std::cerr << "]";
    }

  // After this, print all non-bool options with keyword:
  for ( elem = ParamList; elem != 0; elem = elem->next)
    {
      par = elem->definition;
      if ( par->optional() && par->skipAfter() && (par->myKWless_param(-1)) )
	{
	  par->usageInfo();
	}
    }
 
  // Now print the standard options:
  std::cerr << " [-h] [-f file] [--]";

  // After this, print all options without keyword, using the correct order:
  for ( i = 1; i <= no_of_kwdless_par; i++)
    {
      for ( elem = ParamList; elem != 0; elem = elem->next)
	{
	  par = elem->definition;
	  if (par->myKWless_param(i)) 
	    {
	      par->usageInfo();
	    }
	}
    }
  
  std::cerr << std::endl;
  
  // Now write the description:
  // Again, first the obligatory ones:
  for ( elem = ParamList; elem != 0; elem = elem->next)
    {
      par = elem->definition;
      if (!par->optional())
	par->writeDesc();
    }
  // Then the optional ones:
  for ( elem = ParamList; elem != 0; elem = elem->next)
    {
      par = elem->definition;
      if (par->optional())
	par->writeDesc();
    }
  std::cerr << "\n   -h, --help:\n\t\tPrint this help information";
  std::cerr << "\n   -f file, --ParameterFile=file:\n\t\tUse the parameters given in file";
  std::cerr << "\n   --:\n\t\tDelimiter: '-' won't be interpreted as flag/\n";
  std::cerr << "\t\tkeyword trigger behind this.\n";
  exit(-1);
}  

//_________________________________________________________


/**
   Find out which parameter file defined the value of a parameter, or whether
   the parameter was specified on the command line.

   Search the defined params for matching key, returns NULL if not found.
   Then, if the parameter has already been seen in a parameter file,
   CParamDef::paramFileName should contain a pointer to the file name (which
   is returned, then) otherwise (that is, parameter not seen yet or specified
   on command-line) CParamDef::paramFileName contains NULL, which is also
   returned, then.
 */
const char* CParam::getSourceFromKey(const char *key) {
    // search param defs
    for (ParamDefElem *parDef = ParamList;
	 parDef != NULL;
	 parDef = parDef->next) {
	if (strcmp(parDef->definition->getKeyword(),key) == 0) {
	    // matching keys found
	    return parDef->definition->getSource();
	} // endif: matching keys found
    } // endfor: search param defs

    // not found? return NULL
    return NULL;
} // CParam::getSourceFromKey()


void * CParam::getValueFromKey( const char *key )
{
  //cout << "get value for " << key << endl;
  // search param defs
  for (ParamDefElem *parDef = ParamList;
	   parDef != NULL;
	   parDef = parDef->next) {
	if (strcmp(parDef->definition->getKeyword(),key) == 0) {
	  // matching keys found
	  return parDef->definition->getValue();
	} // endif: matching keys found
  } // endfor: search param defs
  
    // not found? return NULL
  return NULL;
} // CParam::getSourceFromKey()


/**
   @brief Print out parameters to stream.

   Parameters are printed in a format that could be parsed by read_parameter_file().
*/
void CParam::Display(std::ostream& os) const {
  // traverse list of parameter definitions
  ParamDefElem *currentListElement = ParamList;
  while (currentListElement != NULL) {
    // the param def the list element points to must not be NULL
    assert(currentListElement->definition);

    // display parameter
    currentListElement->definition->display(os);

    // next parameter def
    currentListElement = currentListElement->next;
  } // endwhile: traverse list of parameter defs
} // void CParam::Display(std::ostream& os) const {
