
;;
;; PRINT-JIS-STRING.
;;

(defun print-jis-string (struct stream depth)

  "This version prints out the dispatching macro chars #@, followed by
   the characters in the structure and a #\Space. Re-readable as a
   JIS string."
  
  (declare (ignore depth))
  (write-char #\# stream)
  (write-char #\@ stream)
  (dolist (character (jis-string-characters struct)
		     (write-char #\Space stream))
    (write-char character stream)))

;;
;; JIS-STRING STRUCTURE.
;;

; a simple defstruct that uses just one slot, characters, which is a list of
; characters, and uses a :print-function to get the "right" thing printed
; out when the structure is printed.

(defstruct (jis-string (:print-function print-jis-string))
  characters)

;;
;; DISPATCH INPUT FUNCTION.
;;

(defun read-jis-string (stream subchar args)

  "Read any and all characters from the stream, up to but not including the
   next #\Space (which is thrown away). Builds a jis-string out of the
   characters that are read in."
  
  (declare (ignore subchar args))
  (let (char chars)
    (loop
      (setf char (read-char stream))
      (if (char-equal char #\Space)
	  (return)
	  (setf chars (nconc chars (list char)))))
    (make-jis-string
     :characters chars)))

;;
;; DISPATCH MACRO CHARACTERS.
;;

(set-dispatch-macro-character #\# #\@ #'read-jis-string)

;
; To use this facility in a grammar, prefix any jis codes with #@ and
; be sure the jis string is followed by a space. E.g.,
;
;(<n> --> (%)
;     (((x1 value) = #@$@%"%@%W%?!<(J ) ; note space before right paren!
;      ((x0 root) = adaputaa)
;      ((x0 cat) = n)))
;

;;
;; JIS-TO-STRING.
;;

(defun jis-to-string (jis-string)

  "Converts a jis-string to a simple-string. The simple-string may not
   be re-readable, i.e., it may contain double-quotes inside it."

  (let* ((chars (jis-string-characters jis-string))
	 (n (length chars))
	 (new (make-string n)))
    (dotimes (i n)
      (setf (char new i) (pop chars)))
    new))

; patch GenKit function.

(defun make-it-string (x)
  (if (typep x 'jis-string)
      (jis-to-string x)
      (if (symbolp x)
	  (symbol-name x)
	  (if (numberp x)
	      (prin1-to-string x) x))))
