#ifndef _LOGPROBH_
#define _LOGPROBH_

// Routines to perform integer exponential arithmetic.
// A number x is represented as n, where x = b**n
// It is assumed that b > 1, something like b = 1.001

#include <iostream.h>
#include <math.h>

inline int round(double x)
{ return (x >= 0) ? (int)(x + 0.5) : (int)(x - 0.5); }

class LogProb {
public:
  // Constructors
  LogProb() : logr(zeron) {}
  LogProb(const LogProb &obj) : logr(obj.logr) {}
  LogProb(double x) : logr(x == 0.0 ? zeron : round(log(x)/logb2)) {}
  // destructor
  ~LogProb() {}                  // default destructor

  operator double() const    // converts logr to (double) b**logr
    {
      if (logr < nmin) return ntof[0];
      if (logr > nmax) return ntof[nmax-nmin];
      return ntof[logr-nmin];
    }
  LogProb &operator=(const LogProb &obj) { logr = obj.logr; return *this; }
  int operator!() const { return logr == zeron; }
  
  // arithmetic operators 
  LogProb &operator+=(const LogProb &);   // logr2 = logb ( b**logr2 + b**logr1 )
  LogProb &operator-=(const LogProb &);   // logr2 = logb ( b**logr2 + b**logr1 )
  LogProb operator*(const LogProb &mul) const  // logr3 = logr2 + logr1
    {
      LogProb result;		// start out with result == 0
      if ((logr != zeron) && (mul.logr != zeron))
	result.logr = (logr+mul.logr > zeron) ? logr+mul.logr : zeron;
      return result;
    }
  LogProb &operator*=(const LogProb &mul) // logr2 += logr1
    { 
      if ((logr == zeron) || (mul.logr == zeron))
	logr = zeron;
      else
	logr = (logr+mul.logr > zeron) ? logr+mul.logr : zeron;
      return *this;
    }
  LogProb operator/(const LogProb &div) const  // logr3 = logr2 -logr1
    {
      LogProb result;
      if (logr != zeron)
	result.logr = (logr - div.logr > zeron) ? logr - div.logr : zeron;
      return result;
    }
  LogProb &operator/=(const LogProb &div) // logr2 -= logr1
    {
      if (logr != zeron)
	logr = (logr - div.logr > zeron) ? logr - div.logr : zeron;
      return *this;
    }
  LogProb operator+(const LogProb &l) const // logr3 = logb ( b**logr2 + b**logr1 )
    { LogProb result(*this); result += l; return result; }
  LogProb operator-(const LogProb &l) const // logr3 = logb ( b**logr2 - b**logr1 )
    { LogProb result(*this); result -= l; return result; }
  LogProb power(const int n) const // logr2 = logr1 * int
    { LogProb result(*this); result.logr *= n; return result; }
  
  // Conditional operators
  int operator<(const LogProb &obj)  const { return logr <  obj.logr; }
  int operator<=(const LogProb &obj) const { return logr <= obj.logr; }
  int operator>(const LogProb &obj)  const { return logr >  obj.logr; }
  int operator>=(const LogProb &obj) const { return logr >= obj.logr; }
  int operator==(const LogProb &obj) const { return logr == obj.logr; }
  int operator!=(const LogProb &obj) const { return logr != obj.logr; }
  
  int operator<(double d)  const { return ((double)*this) < d;  }
  int operator<=(double d) const { return ((double)*this) <= d; }
  int operator>(double d)  const { return ((double)*this) >  d; }
  int operator>=(double d) const { return ((double)*this) >= d; }
  int operator==(double d) const { return ((double)*this) == d; }
  int operator!=(double d) const { return ((double)*this) != d; }

  LogProb &SetZero() { logr = zeron; return *this; } // representation of 0,
  LogProb &SetOne() { logr = onen; return *this; }   // 1, and
  LogProb &SetInf() { logr = infn; return *this; }   // inf in logarithm domain 

private:
  int logr;			// a representation of logarithm

  // static constants
  static int    initialized; // initialization flag
  static const  double b;
  static const  double logb2;
  static const  int    nmin, nmax;
  static const  int    tblbnd;
  static const  int    zeron, onen, infn;  // zero, one, and inf in log domain
  static const  int    max_2byte_integer, min_2byte_integer;
  
  // Arithmetic computation Tables
  static double *ntof;
  static int   *addtbl;
  static int   *subtbl;

  static int Initialize();

public:
  static void FreeTables();
  // constants for initializing triples to 0 or 1
  static const LogProb zero;
  static const LogProb one;
};

#endif
