//  Routines to perform integer exponential arithmetic.
//  A number x is represented as n, where x = b**n.
//  It is assumed that b > 1, something like b = 1.001;

#pragma implementation

#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include "LogProb.H"

// static type declaration
double *LogProb::ntof  = NULL; // Tables will be initialized
int   *LogProb::addtbl = NULL; // in Initialize function.
int   *LogProb::subtbl = NULL; //

const int    LogProb::max_2byte_integer =  32767;
const int    LogProb::min_2byte_integer = -32768;
const double LogProb::b      = 1.001;   // a logarithm basis
const double LogProb::logb2  = log(b);
const int    LogProb::nmax   = round(78.0E0 * log(1.0E1) / logb2);
const int    LogProb::nmin   = -nmax;
const int    LogProb::tblbnd = round(log((b-1.0E0)/2.0E0)/logb2);
const int    LogProb::zeron  = round(pow(-2, 23));
const int    LogProb::onen   = 0;
const int    LogProb::infn   = onen - zeron;

int LogProb::initialized = LogProb::Initialize();
const LogProb LogProb::zero(0);
const LogProb LogProb::one(1);

// static table initialization function
int LogProb::Initialize()
{
  int nbytes = sizeof(double)*(nmax-nmin+1) + sizeof(int)*(0-tblbnd+1);
  cerr << nbytes << " bytes used for LogProb tables\n";
  ntof   = new double[nmax-nmin+1];
  addtbl = new int[-tblbnd+1];
  subtbl = new int[-tblbnd+1];
  int i;
  
  for (i=nmin; i<=nmax; ++i) {
    double x = i;
    ntof[i-nmin] = exp(x*logb2);
  }
  for (i=tblbnd; i<=0; ++i) {
    double x = 1.0 + pow(b, i);
    addtbl[i-tblbnd] = round(log(x)/logb2);
  }
  double sqrtb = exp(0.5*logb2);
  for (i=0; i<=-tblbnd; ++i) {
    double x = sqrtb * pow(b,i) - 1.0;
    subtbl[i] = round(log(x)/logb2);
  }
  return 1;
}

void LogProb::FreeTables()
{
   delete [] addtbl;
   delete [] subtbl;
   delete [] ntof;
}

//---------------------------------------------------------------------------
//            Aritmetic operators
//---------------------------------------------------------------------------

// Add two numbers represented as logarithms. Use the following method:
//   b**n + b**m = b**n(1 + b**(m-n)), assuming n >= m.
LogProb &LogProb::operator+=(const LogProb &add)
{
  if (add.logr == zeron)
    return *this;
  if (logr == zeron)
    {
      logr = add.logr;
      return *this;
    }
  int a = add.logr - logr;
  if (a > 0)
    {
      a = -a;
      logr = add.logr;
    }
  if (a < tblbnd)
    return *this;
  logr += addtbl[a-tblbnd];
  return *this;
}

// Subtract two logarithm numbers. Use the following method:
// b**n - b**m = b**m( b**(n-m) - 1 ), assuming n >= m. 
LogProb& LogProb::operator-=(const LogProb &subs) 
{
  if (subs.logr == zeron)
    return *this;
  int a = logr - subs.logr;
  if (a <= 0)
    {
      if (a < 0)
	{
	  cerr << "Invalid arguments to nsub" << endl;
	  exit(-1);
	}
      logr = zeron;
      return *this;
    }
  if (a > -tblbnd)
    return *this;
  logr = subs.logr + subtbl[a];
  return *this;
}
