/**
 * 
 */
package cmonson.morphologyInduction.initialization;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cmonson.morphologyInduction.Corpus;
import cmonson.morphologyInduction.MorphologyInducer;

class SetCorpusStartingPositionByTokenCommand extends AbstractInitFileCommand {

	public final static String COMMAND_STRING = "startReadingCorpusAfterTokenNumber";
	
	/** 
	 * @return A String error message specifying the format for a 'startReadingCorpusAfterTokenNumber'
	 *         initializations file command 
	 */
	public static String getCommandUsage() {
		String usage =
			String.format(COMMAND_STRING + " <integer>%n" +
						  "  Where <integer> is the exact number of tokens to skip before starting to%n" +
						  "  read from the corpus.  An <integer> must consist only of digits.");
		return usage;
	}

	/**
	 * Attempt to parse the line <code>line</code> as a 'startReadingCorpusAfterTokenNumber' command.  
	 * If the 'startReadingCorpusAfterTokenNumber' command is successfully parsed then 
	 * morphologyInducer is told to start reading tokens immediately after the specified number of tokens.
	 * 
	 * @param morphologyInducer If the 'startReadingCorpusAfterTokenNumber' command is 
	 * 						    successfully parsed then morphologyInducer is told to start 
	 * 							reading tokens immediately after the specified number of tokens.
	 * @param line The line from the initializations file we belive is a 'startReadingCorpusAfterTokenNumber' 
	 *               command.
	 * @param pathToInitFile The path to the initializations file where the 
	 *                       'startReadingCorpusAfterTokenNumber' command occured. 
	 * @param lineNum The line number from the initializations file where the 
	 *                'startReadingCorpusAfterTokenNumber' command occured.
	 * @return <code>true&nbsp;</code> if parse succeeds
	 *     <br><code>false</code>      if parse fails
	 */
	@Override
	public boolean parseCommand(MorphologyInducer morphologyInducer,
								String line, 
								String pathToInitFile, 
								int lineNum) {
		Pattern commandPattern =
			Pattern.compile("^\\s*" + COMMAND_STRING + "\\s+(\\d+)\\s*$",
							Pattern.CASE_INSENSITIVE);
		Matcher commandMatcher = commandPattern.matcher(line);
		boolean isValidCommand = commandMatcher.matches();
		
		if (isValidCommand) {
			Corpus corpus = morphologyInducer.getCorpus();
			corpus.setTokensToSkip(Integer.valueOf(commandMatcher.group(1)));
		} else {
			String errorMsg =
				String.format("Badly formatted '" + COMMAND_STRING + "' command%n%n" +
							  getCommandUsage() + "%n");
			handleParseError(errorMsg, pathToInitFile, lineNum);
			return false;
		}
		
		return true;
	}
}