package cmonson.morphologyChallengeUtilities;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cmonson.util.FileUtils;

public class MorphologyChallengeBatchEvaluator {

	private static class MorphoChallengeEvaluationResult {
		private double precision;
		private double precisionStdev;
		private double precisionNonAffixes;
		private double precisionNonAffixesStdev;
		private double precisionAffixes;
		private double precisionAffixesStdev;
		
		private double recall;
		private double recallStdev;
		private double recallNonAffixes;
		private double recallNonAffixesStdev;
		private double recallAffixes;
		private double recallAffixesStdev;
		
		private double f1;
		private double f1Stdev;
		private double f1NonAffixes;
		private double f1NonAffixesStdev;
		private double f1Affixes;
		private double f1AffixesStdev;
		
		MorphoChallengeEvaluationResult(
				double precision,           double precisionStdev, 
				double precisionNonAffixes, double precisionNonAffixesStdev, 
				double precisionAffixes,    double precisionAffixesStdev, 
				double recall,              double recallStdev, 
				double recallNonAffixes,    double recallNonAffixesStdev, 
				double recallAffixes,       double recallAffixesStdev, 
				double f1,                  double f1Stdev, 
				double f1NonAffixes,        double f1NonAffixesStdev, 
				double f1Affixes,           double f1AffixesStdev) {

			addPrecisions(
					precision,           precisionStdev, 
					precisionNonAffixes, precisionNonAffixesStdev, 
					precisionAffixes,    precisionAffixesStdev);
			addRecalls(
					recall,           recallStdev, 
					recallNonAffixes, recallNonAffixesStdev, 
					recallAffixes,    recallAffixesStdev);
			addF1s(
					f1,           f1Stdev, 
					f1NonAffixes, f1NonAffixesStdev, 
					f1Affixes,    f1AffixesStdev);
		}

		public MorphoChallengeEvaluationResult() {
			// TODO Auto-generated constructor stub
		}

		private void addPrecisions(double precision, double precisionStdev, double precisionNonAffixes, double precisionNonAffixesStdev, double precisionAffixes, double precisionAffixesStdev) {
			this.precision                = precision;
			this.precisionStdev           = precisionStdev;
			this.precisionNonAffixes      = precisionNonAffixes;
			this.precisionNonAffixesStdev = precisionNonAffixesStdev;
			this.precisionAffixes         = precisionAffixes;
			this.precisionAffixesStdev    = precisionAffixesStdev;
		}
		
		private void addRecalls(double recall, double recallStdev, double recallNonAffixes, double recallNonAffixesStdev, double recallAffixes, double recallAffixesStdev) {
			this.recall                = recall;
			this.recallStdev           = recallStdev;
			this.recallNonAffixes      = recallNonAffixes;
			this.recallNonAffixesStdev = recallNonAffixesStdev;
			this.recallAffixes         = recallAffixes;
			this.recallAffixesStdev    = recallAffixesStdev;
		}


		private void addF1s(double f1, double f1Stdev, double f1NonAffixes, double f1NonAffixesStdev, double f1Affixes, double f1AffixesStdev) {
			this.f1                = f1;
			this.f1Stdev           = f1Stdev;
			this.f1NonAffixes      = f1NonAffixes;
			this.f1NonAffixesStdev = f1NonAffixesStdev;
			this.f1Affixes         = f1Affixes;
			this.f1AffixesStdev    = f1AffixesStdev;
		}
		
		static String getCommaSeparatedColumnHeader() {
			String columnHeader = "";
			columnHeader += "P, ";
			columnHeader += "P Stdev, ";
			columnHeader += "R, ";
			columnHeader += "R Stdev, ";
			columnHeader += "F1, ";
			columnHeader += "F1 Stdev, ";
			
			columnHeader += "P Non-Affixes, ";
			columnHeader += "P Non-Affixes Stdev, ";
			columnHeader += "R Non-Affixes, ";
			columnHeader += "R Non-Affixes Stdev, ";
			columnHeader += "F1 Non-Affixes, ";
			columnHeader += "F1 Non-Affixes Stdev, ";
			
			columnHeader += "P Affixes, ";
			columnHeader += "P Affixes Stdev, ";
			columnHeader += "R Affixes, ";
			columnHeader += "R Affixes Stdev, ";
			columnHeader += "F1 Affixes, ";
			columnHeader += "F1 Affixes Stdev, ";
			
			
			return columnHeader;
		}
		
		String getCommaSeparatedString() {
			String commaSeparatedEvaluationResult = "";

			commaSeparatedEvaluationResult += String.format("%.4f, ", precision);
			commaSeparatedEvaluationResult += String.format("%.4f, ", precisionStdev);
			commaSeparatedEvaluationResult += String.format("%.4f, ", recall);
			commaSeparatedEvaluationResult += String.format("%.4f, ", recallStdev);
			commaSeparatedEvaluationResult += String.format("%.4f, ", f1);
			commaSeparatedEvaluationResult += String.format("%.4f, ", f1Stdev);

			commaSeparatedEvaluationResult += String.format("%.4f, ", precisionNonAffixes);
			commaSeparatedEvaluationResult += String.format("%.4f, ", precisionNonAffixesStdev);
			commaSeparatedEvaluationResult += String.format("%.4f, ", recallNonAffixes);
			commaSeparatedEvaluationResult += String.format("%.4f, ", recallNonAffixesStdev);
			commaSeparatedEvaluationResult += String.format("%.4f, ", f1NonAffixes);
			commaSeparatedEvaluationResult += String.format("%.4f, ", f1NonAffixesStdev);

			commaSeparatedEvaluationResult += String.format("%.4f, ", precisionAffixes);
			commaSeparatedEvaluationResult += String.format("%.4f, ", precisionAffixesStdev);
			commaSeparatedEvaluationResult += String.format("%.4f, ", recallAffixes);
			commaSeparatedEvaluationResult += String.format("%.4f, ", recallAffixesStdev);
			commaSeparatedEvaluationResult += String.format("%.4f, ", f1Affixes);
			commaSeparatedEvaluationResult += String.format("%.4f", f1AffixesStdev);

			return commaSeparatedEvaluationResult;
		}
		
		
	}
	
	/**
	 * @param args
	 * @throws IOException 
	 * @throws InterruptedException 
	 */
	public static void main(String[] args) throws IOException, InterruptedException {
		
		parseCommandLine(args);
		
		MorphologyChallengeBatchEvaluator batchEvaluator = 
			new MorphologyChallengeBatchEvaluator(
					args[0],
					Integer.parseInt(args[2]),
					Integer.parseInt(args[4]),
					new File(args[5]), 
					new File(args[6]),
					new File(args[7]),
					new File(args[8]),
					new File(args[9]));
		
		batchEvaluator.evaluate(batchEvaluator.topDirectoryContainingSegmentationFiles);
	}

	private static void parseCommandLine(String[] args) {
		if (args.length != 10) {
			usage();
		}
		
		if ( ! args[1].equals("-npairs")) {
			System.err.println();
			System.err.println("The second argument must be the string '-npairs'");
			usage();
		}
		
		if ( ! args[3].equals("-npartitions")) {
			System.err.println();
			System.err.println("The fourth argument must be the string '-npartitions'");
			usage();
		}
	}

	private enum AnalyzerType {
		MORFESSOR, 
		PARAMOR
	}

	AnalyzerType analyzerType;
	int numberOfPairs;
	int numberOfPartitions;
	File topDirectoryContainingSegmentationFiles; 
	File sampleWordPairsPerlScriptFile;
	File evalMorphemesPerlScriptFile;
	File goldStandardWordPairsFile;
	File goldStandardFile;
	private File morphologyChallengeStyleSegmentedFile;
	private Map<File, MorphoChallengeEvaluationResult> evaluationsResultsByFile =
		new TreeMap<File, MorphoChallengeEvaluationResult>();;


	public 
	MorphologyChallengeBatchEvaluator(
			String morphologyAnalyzerType, 
			int numberOfPairs, 
			int numberOfPartitions, 
			File topDirectoryContainingSegmentationFiles, 
			File sampleWordPairsPerlScriptFile,
			File evalMorphemesPerlScriptFile,
			File goldStandardWordPairsFile, 
			File goldStandardFile) throws IOException {
		
		morphologyAnalyzerType = morphologyAnalyzerType.toLowerCase();
		
		if (morphologyAnalyzerType.equals("-morfessor")) {
			analyzerType = AnalyzerType.MORFESSOR;
			
		} else if (morphologyAnalyzerType.equals("-paramor")) {
			analyzerType = AnalyzerType.PARAMOR;
		
		} else {
			System.err.println();
			System.err.println("  -morfessor or -paramor?");
			System.err.println();
			usage();
		}
		
		this.numberOfPairs = numberOfPairs;
		this.numberOfPartitions = numberOfPartitions;
		
		if ( ! topDirectoryContainingSegmentationFiles.isDirectory()) {
			System.err.println();
			System.err.println(
					"  " + topDirectoryContainingSegmentationFiles.getCanonicalPath() +
					" is not a directory");
			usage();
		}
		this.topDirectoryContainingSegmentationFiles = topDirectoryContainingSegmentationFiles;
		
		this.sampleWordPairsPerlScriptFile = sampleWordPairsPerlScriptFile;
		this.evalMorphemesPerlScriptFile = evalMorphemesPerlScriptFile;
		this.goldStandardWordPairsFile = goldStandardWordPairsFile;
		this.goldStandardFile = goldStandardFile;
	}

	private static void usage() {
		System.err.println();
		System.err.println("The command line must look like:");
		System.err.println(
		 "    java MorphologyChallengeBatchEvaluator " + String.format("%n") +
		 "        <-morfessor|-paramor>" + String.format("%n") +
		 "        -npairs <number-of-pairs-to-select>" + String.format("%n") +
		 "        -npartitions <number-of-partitions-over-which-to-compute-precision/recall/F1>" + String.format("%n") +
		 "        <path-to-top-directory-containing-segmentation-files> " + String.format("%n") +
		 "        <path-to-perl-script-that-selects-word-pairs" + String.format("%n") +
		 "        <path-to-perl-script-that-evaluates-a-Morphology-Challenge-style-segmented-file" + String.format("%n") +
		 "        <path-to-Morphology-Challenge-GoldStandard-wordPairs-File>" + String.format("%n") +
		 "        <path-to-Morphology-Challenge-GoldStandard-File>" + String.format("%n%n") +
		 "    -morfessor to handle a segmentations file in Morfessor format" + String.format("%n%n") +
		 "    -paramor to handle a segmentations file already in MorphoChallenge format" + String.format("%n%n"));
		System.err.println("    Exiting...");
		System.err.println();
		System.err.println();
		System.exit(0);
	}
	
	private void evaluate(File aDirectoryContainingSegmentationFilesAndDirectories) throws IOException, InterruptedException {
		
		System.err.println("Recursively Evaluating Files in the following directory:");
		System.err.println("  " + aDirectoryContainingSegmentationFilesAndDirectories.getCanonicalPath());
		System.err.println();
		
		File[] filesAndSubDirs = aDirectoryContainingSegmentationFilesAndDirectories.listFiles();
		
		for (File fileOrSubDir : filesAndSubDirs) {
			
			// Recursively evaluate segmentation files in this directory
			if (fileOrSubDir.isDirectory()) {
				evaluate(fileOrSubDir);
			}
			
			if (isASegmentationFile(fileOrSubDir)) {
				
				System.err.println(
						"  Evaluating the segmentation file: " + 
						fileOrSubDir.getCanonicalPath());
				
				File pairsFile = createPairsFile(fileOrSubDir);
				File evaluatedFile = evaluateFile(fileOrSubDir, pairsFile);
				extractAndSaveEvaluationResults(evaluatedFile);
			}
		}
		
		writeOutEvaluationResultsSummary();
	}

	private boolean isASegmentationFile(File fileOrSubDir) {
		if ( ! fileOrSubDir.isFile()) {
			return false;
		}
		
		String shortFilename = fileOrSubDir.getName();
		if (shortFilename.matches("^.*segmentation.*$")) {
			return true;
		}
		
		return false;
	}

	// Returns the File instance representing 
	private File createPairsFile(File segmentationsFile) throws IOException, InterruptedException {
		
		// Since you can't return two things easily from a method in Java, I set
		// this magic global variable
		morphologyChallengeStyleSegmentedFile = segmentationsFile;
		
		// If we are reading Morfessor style analyses, then convert the
		// Morfessor segmentation file into Morphology Challenge style format
		if (analyzerType == AnalyzerType.MORFESSOR) {
			morphologyChallengeStyleSegmentedFile = 
				convertMorfessorSegmentedFileToMorphologyChallengeStyleSegmentedFile(
						segmentationsFile);
		}

		String pairsFilename = morphologyChallengeStyleSegmentedFile.getCanonicalPath() + ".wordPairs";

		String sampleWordPairsCommand = 
			sampleWordPairsPerlScriptFile.getCanonicalPath() + 
			" -n " + numberOfPairs +
			" -refwords " + goldStandardFile +
			" -segmentations " + morphologyChallengeStyleSegmentedFile.getCanonicalPath() +
			" -wordpairs " + pairsFilename;
		
		System.err.println("    Creating the Word Pairs File with the command:");
		System.err.println("      " + sampleWordPairsCommand);
		
		Runtime runtime = Runtime.getRuntime();
		Process sampleWordPairsProcess = runtime.exec(sampleWordPairsCommand);
		
		sampleWordPairsProcess.waitFor();
		
		return new File(pairsFilename);
	}

	private File 
	convertMorfessorSegmentedFileToMorphologyChallengeStyleSegmentedFile(
			File segmentationsFile) throws IOException {
		
		System.err.println(
				"    Converting from Morfessor segmentations to " +
				"Morphology Challenge style segmentations");
		
		File parentDir = segmentationsFile.getParentFile();
		
		String shortFilename = segmentationsFile.getName();
		
		String morphologyChallengeStyleSegmentationsFilename =
			shortFilename + ".morphologyChallengeStyle";
		
		String fullMorphologyChallengeStyleSegmentationsFilename =
			parentDir.getCanonicalPath() + "/" + morphologyChallengeStyleSegmentationsFilename;
		
		File morphologyChallengeStyleSegmentationsFile =
			new File(fullMorphologyChallengeStyleSegmentationsFilename);
		
		MorfessorToMorphoChallengeAnalysis morfessorToMorphoChallengeAnalysis =
			new MorfessorToMorphoChallengeAnalysis(
					segmentationsFile, 
					morphologyChallengeStyleSegmentationsFile);
		
		morfessorToMorphoChallengeAnalysis.convert();
		
		return morphologyChallengeStyleSegmentationsFile;
	}

	private File evaluateFile(File fileOrSubDir, File pairsFile) throws IOException, InterruptedException {
		// morphologyChallengeStyleSegmentedFile is a magic global variable that
		// is set in createPairsFile().
		String evaluatedFilename = 
			morphologyChallengeStyleSegmentedFile.getCanonicalPath() + ".evaluated";
		File evaluatedFile = new File(evaluatedFilename);
		
		String evalMorphemesCommand =
			evalMorphemesPerlScriptFile.getCanonicalPath() +
			" -partitions " + numberOfPartitions + 
			" -trace " +
			goldStandardWordPairsFile.getCanonicalPath() +
			" " + pairsFile.getCanonicalPath() +
			" " + goldStandardFile.getCanonicalPath() +
			" " + morphologyChallengeStyleSegmentedFile.getCanonicalPath() +
			" " + evaluatedFilename;
		
		System.err.println("    Evaluating the segmentations with the command:");
		System.err.println("      " + evalMorphemesCommand);
		
		Runtime runtime = Runtime.getRuntime();
		Process evalMorphemesProcess = runtime.exec(evalMorphemesCommand);  // we don't need to wait for this command to finish
		evalMorphemesProcess.waitFor();
		
		return evaluatedFile;
	}

	// Matches and extracts the meat from, a line like:  
	//
	//   62.62% ( 1.38%)      non-affixes: 85.66% ( 1.18%)      affixes: 34.22% ( 2.06%)
	//
	// These numbers could be precisions, recalls, or f1s (or anything else)
	//
	
	// Matches and extracts the numbers from a String like:
	//
	//  63.63% ( 1.38%)
	//
	private static String resultAndStdevPatternString = 
		"\\s*([1234567890.]+)(%?)\\s*\\(\\s*([1234567890.]+)(%?)\\)\\s*";
	private static Pattern resultsPattern =
		Pattern.compile(
				resultAndStdevPatternString + "non-affixes:" +
				resultAndStdevPatternString + "affixes:" +
				resultAndStdevPatternString);
		
	private void extractAndSaveEvaluationResults(File evaluatedFile) throws IOException {
		
		int DEBUG = 1;
		if (DEBUG > 0) {
			System.err.println("Extracting Evaluation Results from the File: ");
			System.err.println("  " + evaluatedFile.getAbsolutePath());
		}
		
		BufferedReader evaluationResultsReader =
			FileUtils.openFileForReading(evaluatedFile, "ISO-8859-1");
		
		MorphoChallengeEvaluationResult thisFilesResults =
			new MorphoChallengeEvaluationResult();
		
		String lineFromEvaluatedFile;
		while ((lineFromEvaluatedFile = evaluationResultsReader.readLine()) != null) {
			if ( ! lineFromEvaluatedFile.matches("^AVE.*$")) {
				continue;
			}
			
			if (DEBUG > 0) {
				System.err.println("Extracting information from the line: ");
				System.err.println("  " + lineFromEvaluatedFile);
			}
			
			// Precisions Line
			if (lineFromEvaluatedFile.matches("^\\s*AVE P \\(stdev\\):.*$")) {
				
				if (DEBUG > 0) {
					System.err.println("Found a Precision Line");
				}
				
				String precisionResultsString = 
					lineFromEvaluatedFile.replaceAll(
							"^\\s*AVE P \\(stdev\\):", 
							"");
				Matcher precisionResultsMatcher = 
					resultsPattern.matcher(precisionResultsString);
				boolean resultsAreWellFormatted =
					precisionResultsMatcher.matches();
				if ( ! resultsAreWellFormatted) {
					System.err.println();
					System.err.println("ERROR: Precision Results are not well formatted. Line:");
					System.err.println("  " + lineFromEvaluatedFile);
					System.err.println("  From the file:");
					System.err.println("  " + evaluatedFile.getCanonicalPath());
				}
				
				String precisionString         = precisionResultsMatcher.group(1);
				String precisionAsPercent      = precisionResultsMatcher.group(2);
				String precisionStdevString    = precisionResultsMatcher.group(3);
				String precisionStdevAsPercent = precisionResultsMatcher.group(4);
				
				String precisionNonAffixesString         = precisionResultsMatcher.group(5);
				String precisionNonAffixesAsPercent      = precisionResultsMatcher.group(6);
				String precisionNonAffixesStdevString    = precisionResultsMatcher.group(7);
				String precisionNonAffixesStdevAsPercent = precisionResultsMatcher.group(8);

				String precisionAffixesString         = precisionResultsMatcher.group(9);
				String precisionAffixesAsPercent      = precisionResultsMatcher.group(10);
				String precisionAffixesStdevString    = precisionResultsMatcher.group(11);
				String precisionAffixesStdevAsPercent = precisionResultsMatcher.group(12);

				double precision = Double.valueOf(precisionString);
				if ( ! precisionAsPercent.equals("")) {
					precision /= 100.0;
				}
				double precisionStdev = Double.valueOf(precisionStdevString);
				if ( ! precisionStdevAsPercent.equals("")) {
					precisionStdev /= 100.0;
				}
				double precisionNonAffixes = Double.valueOf(precisionNonAffixesString);
				if ( ! precisionNonAffixesAsPercent.equals("")) {
					precisionNonAffixes /= 100.0;
				}
				double precisionNonAffixesStdev = Double.valueOf(precisionNonAffixesStdevString);
				if ( ! precisionNonAffixesStdevAsPercent.equals("")) {
					precisionNonAffixesStdev /= 100.0;
				}
				double precisionAffixes = Double.valueOf(precisionAffixesString);
				if ( ! precisionAffixesAsPercent.equals("")) {
					precisionAffixes /= 100.0;
				}
				double precisionAffixesStdev = Double.valueOf(precisionAffixesStdevString);
				if ( ! precisionAffixesStdevAsPercent.equals("")) {
					precisionAffixesStdev /= 100.0;
				}
				
				thisFilesResults.addPrecisions(
						precision,           precisionStdev, 
						precisionNonAffixes, precisionNonAffixesStdev, 
						precisionAffixes,    precisionAffixesStdev);
			}
			
			// Recalls Line
			if (lineFromEvaluatedFile.matches("^\\s*AVE R \\(stdev\\):.*$")) {
				String recallResultsString = 
					lineFromEvaluatedFile.replaceAll(
							"^\\s*AVE R \\(stdev\\):", 
							"");
				Matcher recallResultsMatcher = 
					resultsPattern.matcher(recallResultsString);
				boolean resultsAreWellFormatted =
					recallResultsMatcher.matches();
				if ( ! resultsAreWellFormatted) {
					System.err.println();
					System.err.println("ERROR: Recall Results are not well formatted. Line:");
					System.err.println("  " + lineFromEvaluatedFile);
					System.err.println("  From the file:");
					System.err.println("  " + evaluatedFile.getCanonicalPath());
				}
				
				String recallString         = recallResultsMatcher.group(1);
				String recallAsPercent      = recallResultsMatcher.group(2);
				String recallStdevString    = recallResultsMatcher.group(3);
				String recallStdevAsPercent = recallResultsMatcher.group(4);
				
				String recallNonAffixesString         = recallResultsMatcher.group(5);
				String recallNonAffixesAsPercent      = recallResultsMatcher.group(6);
				String recallNonAffixesStdevString    = recallResultsMatcher.group(7);
				String recallNonAffixesStdevAsPercent = recallResultsMatcher.group(8);

				String recallAffixesString         = recallResultsMatcher.group(9);
				String recallAffixesAsPercent      = recallResultsMatcher.group(10);
				String recallAffixesStdevString    = recallResultsMatcher.group(11);
				String recallAffixesStdevAsPercent = recallResultsMatcher.group(12);

				double recall = Double.valueOf(recallString);
				if ( ! recallAsPercent.equals("")) {
					recall /= 100.0;
				}
				double recallStdev = Double.valueOf(recallStdevString);
				if ( ! recallStdevAsPercent.equals("")) {
					recallStdev /= 100.0;
				}
				double recallNonAffixes = Double.valueOf(recallNonAffixesString);
				if ( ! recallNonAffixesAsPercent.equals("")) {
					recallNonAffixes /= 100.0;
				}
				double recallNonAffixesStdev = Double.valueOf(recallNonAffixesStdevString);
				if ( ! recallNonAffixesStdevAsPercent.equals("")) {
					recallNonAffixesStdev /= 100.0;
				}
				double recallAffixes = Double.valueOf(recallAffixesString);
				if ( ! recallAffixesAsPercent.equals("")) {
					recallAffixes /= 100.0;
				}
				double recallAffixesStdev = Double.valueOf(recallAffixesStdevString);
				if ( ! recallAffixesStdevAsPercent.equals("")) {
					recallAffixesStdev /= 100.0;
				}
				
				thisFilesResults.addRecalls(
						recall,           recallStdev, 
						recallNonAffixes, recallNonAffixesStdev, 
						recallAffixes,    recallAffixesStdev);
			}
			
			// F1s Line
			if (lineFromEvaluatedFile.matches("^\\s*AVE F1\\(stdev\\):.*$")) {
				String f1ResultsString = 
					lineFromEvaluatedFile.replaceAll(
							"^\\s*AVE F1\\(stdev\\):", 
							"");
				Matcher f1ResultsMatcher = 
					resultsPattern.matcher(f1ResultsString);
				boolean resultsAreWellFormatted =
					f1ResultsMatcher.matches();
				if ( ! resultsAreWellFormatted) {
					System.err.println();
					System.err.println("ERROR: f1 Results are not well formatted. Line:");
					System.err.println("  " + lineFromEvaluatedFile);
					System.err.println("  From the file:");
					System.err.println("  " + evaluatedFile.getCanonicalPath());
				}
				
				String f1String         = f1ResultsMatcher.group(1);
				String f1AsPercent      = f1ResultsMatcher.group(2);
				String f1StdevString    = f1ResultsMatcher.group(3);
				String f1StdevAsPercent = f1ResultsMatcher.group(4);
				
				String f1NonAffixesString         = f1ResultsMatcher.group(5);
				String f1NonAffixesAsPercent      = f1ResultsMatcher.group(6);
				String f1NonAffixesStdevString    = f1ResultsMatcher.group(7);
				String f1NonAffixesStdevAsPercent = f1ResultsMatcher.group(8);

				String f1AffixesString         = f1ResultsMatcher.group(9);
				String f1AffixesAsPercent      = f1ResultsMatcher.group(10);
				String f1AffixesStdevString    = f1ResultsMatcher.group(11);
				String f1AffixesStdevAsPercent = f1ResultsMatcher.group(12);

				double f1 = Double.valueOf(f1String);
				if ( ! f1AsPercent.equals("")) {
					f1 /= 100.0;
				}
				double f1Stdev = Double.valueOf(f1StdevString);
				if ( ! f1StdevAsPercent.equals("")) {
					f1Stdev /= 100.0;
				}
				double f1NonAffixes = Double.valueOf(f1NonAffixesString);
				if ( ! f1NonAffixesAsPercent.equals("")) {
					f1NonAffixes /= 100.0;
				}
				double f1NonAffixesStdev = Double.valueOf(f1NonAffixesStdevString);
				if ( ! f1NonAffixesStdevAsPercent.equals("")) {
					f1NonAffixesStdev /= 100.0;
				}
				double f1Affixes = Double.valueOf(f1AffixesString);
				if ( ! f1AffixesAsPercent.equals("")) {
					f1Affixes /= 100.0;
				}
				double f1AffixesStdev = Double.valueOf(f1AffixesStdevString);
				if ( ! f1AffixesStdevAsPercent.equals("")) {
					f1AffixesStdev /= 100.0;
				}
				
				thisFilesResults.addF1s(
						f1,           f1Stdev, 
						f1NonAffixes, f1NonAffixesStdev, 
						f1Affixes,    f1AffixesStdev);
			}
		}

		evaluationsResultsByFile.put(evaluatedFile, thisFilesResults);
	}
	
	private void writeOutEvaluationResultsSummary() {
		File evaluationsSummaryFile = 
			new File(
					topDirectoryContainingSegmentationFiles,
					"evaluationSummary.csv");
		
		PrintWriter evaluationSummaryWriter = 
			FileUtils.openFileForWriting(evaluationsSummaryFile, "ISO-8859-1");
		
		evaluationSummaryWriter.println(
				"Filename, " + 
				MorphoChallengeEvaluationResult.getCommaSeparatedColumnHeader());
		
		for (File resultsFile : evaluationsResultsByFile.keySet()) {
			MorphoChallengeEvaluationResult evaluationResultsForThisFile =
				evaluationsResultsByFile.get(resultsFile);
			
			evaluationSummaryWriter.println(
					resultsFile.getAbsolutePath() + ", " + 
					evaluationResultsForThisFile.getCommaSeparatedString());
		}
	}
}
