package info.jonclark.stat;

import info.jonclark.util.FormatUtils;
import info.jonclark.util.StringUtils;
import info.jonclark.util.TimeLength;

import java.io.PrintStream;
import java.util.concurrent.atomic.AtomicBoolean;

public class TextProgressBar implements Runnable, TaskListener {

	private final PrintStream out;
	private final Thread thread = new Thread(this);
	private final RemainingTimeEstimator est;
	private final SecondTimer timer;
	private final AtomicBoolean dirty = new AtomicBoolean(true);
	private final int screenWidth;
	private final String singularEventName;

	private int nEventsCompleted;
	private int nEventsToComplete = 100;
	private String completeBar;
	private String incompleteBar;

	public TextProgressBar(PrintStream out, String singularEventName, int nWindowSize,
			int screenWidthInColumns) {

		this.out = out;
		this.est = new RemainingTimeEstimator(nWindowSize);
		this.timer = new SecondTimer(true, false);
		this.singularEventName = singularEventName;
		this.screenWidth = screenWidthInColumns;

		// don't allow this thread to keep the JVM alive
		this.thread.setDaemon(true);
	}

	public void interrupt() {
		thread.interrupt();
	}

	public void recordEventCompletion() {
		synchronized (thread) {
			this.nEventsCompleted++;
			if (this.nEventsCompleted == this.nEventsToComplete) {
				this.timer.pause();
			}
			est.recordEvent();
		}
		dirty.set(true);
	}

	public void beginTask(int n) {
		this.timer.go();
		this.nEventsToComplete = n;
		dirty.set(true);
		if (screenWidth > 0) {
			this.thread.start();
		}
	}

	public void endTask() {
		this.thread.interrupt();
	}

	private void update() {

		if (dirty.get()) {
			float percentComplete = (float) nEventsCompleted / (float) nEventsToComplete;
			String strPercent =
					nEventsCompleted + "/" + nEventsToComplete + " "
							+ FormatUtils.formatDouble2(percentComplete * 100) + "%";
			String text = " " + strPercent;

			if (est != null) {
				int nEventsRemaining = nEventsToComplete - nEventsCompleted;

				String etc = "";
				String elapsed = "";
				String remainingTime = "";
				String rate;
				synchronized (thread) {
					TimeLength remainingLenth = est.getRemainingTime(nEventsRemaining);
					if (remainingLenth.getInMillis() > 500) {
						etc =
								", ETC: "
										+ est.getEstimatedCompetionTimeFormatted(nEventsRemaining);
						remainingTime = remainingLenth.toStringSingleUnit() + " left, ";
					}
					elapsed = timer.getElapsedTime().toStringSingleUnit() + " elapsed";

					if (est.getEventsPerSecond() < 1.0) {
						rate = ", " + est.getSecondsPerEventFormatted() + " sec/" + singularEventName;
					} else {
						rate =
								", " + est.getEventsPerSecondFormatted() + " " + singularEventName
										+ "s/sec";
					}
				}

				text += " (" + remainingTime + elapsed + etc + rate + ")";
			}

			int barWidth = screenWidth - text.length();
			barWidth -= 3; // space and two buffer bookend chars
			int completeWidth = (int) (barWidth * percentComplete);
			if (completeWidth == 0) {
				completeWidth = 1; // always save room for the arrow char
			} else if (completeWidth > barWidth) {
				completeWidth = barWidth;
			}

			int incompleteWidth = barWidth - completeWidth;
			synchronized (thread) {
				// change the bar appearance if we're done
				if (incompleteWidth == 0) {
					this.completeBar =
							"|" + StringUtils.duplicateCharacter('*', barWidth) + "| " + text;
					this.incompleteBar = "";
				} else {
					this.completeBar =
							"|" + StringUtils.duplicateCharacter('=', completeWidth - 1) + ">";
					this.incompleteBar =
							StringUtils.duplicateCharacter('-', incompleteWidth) + "|" + text;
				}
			}
		}
	}

	public void run() {

		char[] spinChars = new char[] { '|', '/', '-', '\\' };
		int i = 0;
		int j = 0;
		long spinTimeoutMills = 100;
		long textUpdateMillis = 2000;
		int textUpdateIterations = (int) (textUpdateMillis / spinTimeoutMills);

		// make sure we don't have null strings
		update();
		
		// spin unless told otherwise
		String prevString = "";
		boolean done = false;
		while (thread.isInterrupted() == false && done == false) {

			j++;
			if (j % textUpdateIterations == 0)
				update();

			out.print(StringUtils.duplicateCharacter('\b', prevString.length()));

			if (nEventsCompleted < nEventsToComplete) {
				StringBuilder builder = new StringBuilder(1000);
				synchronized (thread) {
					builder.append(completeBar);
					builder.append(spinChars[i]);
					builder.append(incompleteBar);
				}
				prevString = builder.toString();
				out.print(prevString);
			} else {
				done = true;
			}

			i++;
			i %= spinChars.length;
			try {
				Thread.sleep(spinTimeoutMills);
			} catch (InterruptedException e) {
				;
			}
		}

		synchronized (thread) {
			this.nEventsCompleted = this.nEventsToComplete;
			this.dirty.set(true);
			update();
			out.print(StringUtils.duplicateCharacter('\b', prevString.length()));
			out.println(completeBar);
		}
	}

	public static void main(String[] args) throws Exception {

		int nWindow = 500;
		TaskListener bar = new TextProgressBar(System.out, "nothing", nWindow, 100);

		int nEvents = 100;
		bar.beginTask(nEvents);
		for (int i = 0; i < nEvents; i++) {
			bar.recordEventCompletion();
			Thread.sleep(100);
		}

		bar = new TextProgressBar(System.out, "something", nWindow, 100);

		nEvents = 3;
		bar.beginTask(nEvents);
		for (int i = 0; i < nEvents; i++) {
			bar.recordEventCompletion();
			Thread.sleep(2500);
		}
	}
}
