package info.jonclark.stat;

import info.jonclark.util.FormatUtils;

import java.util.ArrayList;

/**
 * A timer meant for profiling execution timings
 */
public class ProfilerTimer {

	private long accumulatedTime = 0;
	private long startDate = 0;

	private final String name;
	private final boolean catchOverlappingGo;
	private ArrayList<ProfilerTimer> children = null;

	/**
	 * @param catchOverlappingGo
	 *            If true, calling go() twice without calling pause() will throw
	 *            an exception
	 */
	private ProfilerTimer(String name, boolean catchOverlappingGo, boolean goNow) {
		this.name = name;
		this.catchOverlappingGo = catchOverlappingGo;
		if (goNow) {
			go();
		}
	}

	public static ProfilerTimer newTimer(String name, ProfilerTimer parent,
			boolean catchOverlappingGo, boolean goNow) {
		
		ProfilerTimer timer = new ProfilerTimer(name, catchOverlappingGo, goNow);
		if (parent != null) {
			parent.registerChild(timer);
		}
		return timer;
	}

	private void registerChild(ProfilerTimer child) {
		if (children == null) {
			// keep list as small as possible
			children = new ArrayList<ProfilerTimer>(1);
		}
		children.add(child);
	}

	/**
	 * Calculates the number of events that occured per second, on average,
	 * during the time this timer was accumulating.
	 * 
	 * @param nEvents
	 *            The number of events that occurred while this SecondTimer has
	 *            been accumulating time.
	 * @return
	 */
	public String getEventsPerSecond(long nEvents) {
		if (getSeconds() > 0)
			return FormatUtils.FORMAT_2DECIMALS.format((double) nEvents / getSeconds());
		else
			return "Undefined";
	}

	/**
	 * Calculates the number of seconds elapsed per event on average, during the
	 * time this timer was accumulating.
	 * 
	 * @param nEvents
	 *            The number of events that occurred while this SecondTimer has
	 *            been accumulating time.
	 * @return
	 */
	public String getSecondsPerEvent(long nEvents) {
		if (nEvents > 0)
			return FormatUtils.FORMAT_2DECIMALS.format(getSeconds() / (double) nEvents);
		else
			return "Undefined";
	}

	/**
	 * Gets the number of seconds accumulated by this SecondTimer so far
	 * 
	 * @return
	 */
	public double getSeconds() {
		return (double) getMilliseconds() / 1000.0;
	}

	/**
	 * Gets the number of seconds accumulated by this SecondTimer so far to 2
	 * decimals accuracy.
	 * 
	 * @return
	 */
	public String getSecondsFormatted() {
		return FormatUtils.FORMAT_2DECIMALS.format(getSeconds());
	}

	/**
	 * Gets the number of milliseconds accumulated by this second timer so far
	 * 
	 * @return
	 */
	public long getMilliseconds() {
		if (startDate != 0)
			return accumulatedTime + System.currentTimeMillis() - startDate;
		else
			return accumulatedTime;
	}

	/**
	 * If the timer
	 * 
	 * @throws RuntimeException
	 *             if catchOverlappingGo was set to true and this is the second
	 *             call to go() without a previous call to pause()
	 */
	public void go() {
		if (catchOverlappingGo && startDate != 0)
			throw new RuntimeException("Overlapping go detected.");
		else if (startDate == 0)
			startDate = System.currentTimeMillis();
	}

	/**
	 * If the timer is currenting "going" then this stops the accumulation of
	 * time. Otherwise, it has no effect.
	 */
	public void pause() {
		accumulatedTime += System.currentTimeMillis() - startDate;
		startDate = 0;
	}

	/**
	 * Resets the amount of time accumulated by this timer to zero.
	 */
	public void reset() {
		accumulatedTime = 0;
	}

	public String getTimingReport(boolean recursive) {
		StringBuilder builder = new StringBuilder();
		getTimingReport(builder, recursive, 0);
		return builder.toString();
	}

	private void getTimingReport(StringBuilder builder, boolean recursive, int level) {
		for (int i = 0; i < level; i++) {
			builder.append("+ ");
		}

		builder.append(name + "\t" + getSecondsFormatted() + "s\n");

		if (recursive) {
			if (children != null) {
				for (ProfilerTimer child : children) {
					child.getTimingReport(builder, recursive, level + 1);
				}
			}
		}
	}
}
