/*
 * Copyright (c) 2006, Jonathan Clark <jon_DOT_h_DOT_clark_AT_gmail_DOT_com> 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright notice,
 *       this list of conditions and the following disclaimer in the documentation
 *       and/or other materials provided with the distribution.
 *     * Neither the name of my affiliates nor the names of thier contributors
 *       may be used to endorse or promote products derived from this software
 *       without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIEDWARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */
package info.jonclark.lang;

/**
 * Similar to a standard {@link StringBuilder} except that it supports
 * <code>prepend()</code> instead of <code>append()</code>.
 */
public class DoubleEndedStringBuilder {

	private final StringBuilder appender;
	private final PrependStringBuilder prepender;
	private String cachedString = null;

	public DoubleEndedStringBuilder() {
		prepender = new PrependStringBuilder();
		appender = new StringBuilder();
	}

	public DoubleEndedStringBuilder(DoubleEndedStringBuilder other) {
		prepender = new PrependStringBuilder(other.prepender);
		appender = new StringBuilder(other.appender);
	}

	/**
	 * Constructs a string builder initialized to the contents of the specified
	 * string. The initial capacity of the string builder is <code>16</code>
	 * plus the length of the string argument.
	 * 
	 * @param str
	 *            the initial contents of the buffer.
	 * @throws NullPointerException
	 *             if <code>str</code> is <code>null</code>
	 */
	public DoubleEndedStringBuilder(String str) {
		prepender = new PrependStringBuilder();
		appender = new StringBuilder(str);
	}

	public DoubleEndedStringBuilder append(String str) {
		cachedString = null;
		appender.append(str);
		return this;
	}

	public DoubleEndedStringBuilder prepend(String str) {
		cachedString = null;
		prepender.prepend(str);
		return this;
	}

	public int length() {
		return prepender.length() + appender.length();
	}

	public String toString() {
		if (cachedString == null) {
			cachedString = prepender.toString() + appender.toString();
		}
		return cachedString;
	}
}
