/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.trees.smart;

import info.jonclark.util.ArrayUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.TreeMap;

import edu.cmu.cs.lti.avenue.trees.cfg.CfgRule;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelDisplay;

public class TreeNode implements Cloneable {

	private final ArrayList<TreeNode> children = new ArrayList<TreeNode>();
	protected final ArrayList<String> values = new ArrayList<String>();
	private TreeMap<String, Object> metadata = null;
	protected SmartTree parentTree;
	private TreeNode parentNode;
	protected int treeLabelingIndex;
	protected int absoluteTreeIndex;
	protected int terminalIndex;
	private String treeName;

	private TreeNode(int nTerminalIndex, String name) {
		this.parentTree = null;
		this.parentNode = null;
		this.treeLabelingIndex = -1;
		this.absoluteTreeIndex = 0;
		this.terminalIndex = nTerminalIndex;
		this.treeName = name;
	}

	protected TreeNode(SmartTree parentTree, TreeNode parentNode, int treeLabelingIndex,
			int absoluteTreeIndex) {
		this.parentTree = parentTree;
		this.parentNode = parentNode;
		this.treeLabelingIndex = treeLabelingIndex;
		this.terminalIndex = -1;
		this.treeName = parentTree.getName();
		this.absoluteTreeIndex = absoluteTreeIndex;
	}

	/**
	 * Creates a TreeNode not associated with any SmartTree. This method is
	 * recommended only for special (hackish) cases in which no real source tree
	 * is possible.
	 * 
	 * @nTerminalIndex The ONE-BASED index of the terminal
	 */
	public static TreeNode createOrphanNode(int nTerminalIndex, String name) {
		TreeNode root = new TreeNode(nTerminalIndex, name);
		root.terminalIndex = nTerminalIndex - 1;
		return root;
	}

	protected void addChild(TreeNode child) {
		assert child != this : "cycle detected";
		assert child != parentNode : "cycle detected";
		assert parentTree == null || child != parentTree.getRootNode() : "cycle detected";

		children.add(child);
		if (parentTree != null)
			this.parentTree.fireTreeStructureChanged();
	}

	/**
	 * Adds a child while removing it from the previous tree.
	 * 
	 * @param child
	 */
	public void assimilateChild(TreeNode child) {
		children.add(child);
		if (parentTree != null)
			this.parentTree.fireTreeStructureChanged();
	}

	public TreeNode addChild() {
		TreeNode child = new TreeNode(this.parentTree, this.parentNode, -1, -1);

		children.add(child);
		if (parentTree != null)
			this.parentTree.fireTreeStructureChanged();

		return child;
	}

	/**
	 * Returns all nodes in this subtree that have a label (index). Order is
	 * bottom-up (post-order traversal).
	 */
	protected void getLabeledNodes(ArrayList<TreeNode> nodes) {

		if (!this.isTerminal()) {
			for (final TreeNode child : children) {
				child.getLabeledNodes(nodes);
			}
		}

		if (this.hasTreeLabelingIndex()) {
			nodes.add(this);
		}
	}

	protected void getUnlabeledNodes(ArrayList<TreeNode> nodes) {

		if (!this.isTerminal()) {
			for (final TreeNode child : children) {
				child.getUnlabeledNodes(nodes);
			}
		}

		if (!this.hasTreeLabelingIndex()) {
			nodes.add(this);
		}
	}

	protected void getAllNodes(ArrayList<TreeNode> nodes) {

		if (!this.isTerminal()) {
			for (final TreeNode child : children) {
				child.getAllNodes(nodes);
			}
		}
		nodes.add(this);
	}

	public ArrayList<TreeNode> getChildren() {
		return children;
	}

	/**
	 * Returns all nodes in this subtree that have a label (index). Order is not
	 * guaranteed.
	 */
	public ArrayList<TreeNode> getLabeledNodes() {
		ArrayList<TreeNode> nodes = new ArrayList<TreeNode>();
		getLabeledNodes(nodes);
		return nodes;
	}

	public ArrayList<TreeNode> getUnlabeledNodes() {
		ArrayList<TreeNode> nodes = new ArrayList<TreeNode>();
		getUnlabeledNodes(nodes);
		return nodes;
	}

	public ArrayList<TreeNode> getAllNodes() {
		ArrayList<TreeNode> nodes = new ArrayList<TreeNode>();
		getAllNodes(nodes);
		return nodes;
	}

	public void addValue(String value) {
		assert value == value.intern() : "value not interned.";
		this.values.add(value);
	}

	public ArrayList<String> getValues() {
		return values;
	}

	public String getValuesAsString() {
		StringBuilder builder = new StringBuilder();
		appendValuesAsString(builder);
		return builder.toString();
	}

	public void appendValuesAsString(StringBuilder builder) {
		for (String value : values) {
			if (value.equals("(")) {
				builder.append("-LRP- ");
			} else if (value.equals(")")) {
				builder.append("-RRP- ");
			} else {
				builder.append(value.toString() + " ");
			}
		}

		if (values.size() > 0)
			builder.deleteCharAt(builder.length() - 1);
	}

	public final boolean isTerminal() {
		return (children.size() == 0);
	}

	/**
	 * Gets the one-based tree index. May be -1 if this is not a labeled node.
	 * 
	 * @return
	 */
	public int getTreeLabelingIndex() {
		return treeLabelingIndex;
	}

	public boolean hasTreeLabelingIndex() {
		return (treeLabelingIndex != -1);
	}

	/**
	 * Gets the zero-based absolute tree index. Every node in the tree will have
	 * this index and each index is guaranteed to be unique.
	 * 
	 * @return
	 */
	public int getAbsoluteTreeIndex() {
		if (parentTree != null)
			parentTree.updateTreeStructure();
		return absoluteTreeIndex;
	}

	/**
	 * Gets the 1-based indices of this node relative to the sequence of
	 * terminals. This is useful for tasks such as going from a constituent
	 * index to a lexical index.
	 * 
	 * @return
	 */
	public ArrayList<Integer> getTerminalIndices() {

		// this could be a loner node
		if (parentTree != null)
			this.parentTree.updateTreeStructure();

		if (this.isTerminal()) {
			ArrayList<Integer> list = new ArrayList<Integer>(1);
			list.add(terminalIndex + 1);
			return list;
		} else {
			ArrayList<Integer> list = new ArrayList<Integer>();
			for (TreeNode child : this.children) {
				child.getTerminalIndices(list);
			}
			return list;
		}

	}

	private void getTerminalIndices(ArrayList<Integer> list) {
		if (this.isTerminal()) {
			list.add(terminalIndex + 1);
		} else {
			for (TreeNode child : this.children) {
				child.getTerminalIndices(list);
			}
		}
	}

	public ArrayList<TreeNode> getTerminalNodes() {

		// this could be a loner node
		if (parentTree != null)
			this.parentTree.updateTreeStructure();

		this.parentTree.updateTreeStructure();

		if (this.isTerminal()) {
			ArrayList<TreeNode> list = new ArrayList<TreeNode>(1);
			list.add(this);
			return list;
		} else {
			ArrayList<TreeNode> list = new ArrayList<TreeNode>();
			for (TreeNode child : this.children) {
				child.getTerminalNodes(list);
			}
			return list;
		}

	}

	private void getTerminalNodes(ArrayList<TreeNode> list) {
		if (this.isTerminal()) {
			list.add(this);
		} else {
			for (TreeNode child : this.children) {
				child.getTerminalNodes(list);
			}
		}
	}

	protected TreeNode getCopyWithNodeSubstitution(SmartTree parentTree, TreeNode parentNode,
			TreeNode deletedSubtree, TreeNode replacementSubtree) {

		TreeNode t = getShallowCopy(parentTree, parentNode);

		for (int i = 0; i < t.children.size(); i++) {
			TreeNode child = t.children.get(i);
			if (child == deletedSubtree) {
				t.children.set(i, replacementSubtree.getDeepCopy(parentTree, t));
			} else {
				t.children.set(i, child.getCopyWithNodeSubstitution(parentTree, t, deletedSubtree,
						replacementSubtree));
			}
		}
		return t;
	}

	protected TreeNode getCopyWithAddition(SmartTree parentTree, TreeNode copyParent,
			TreeNode addParentNode, TreeNode additionalSubtree) {

		TreeNode t = getShallowCopy(parentTree, copyParent);

		if (this == addParentNode) {
			t.children.add(additionalSubtree.getDeepCopy(parentTree, t));
		}

		for (int i = 0; i < t.children.size(); i++) {
			TreeNode child = t.children.get(i);
			t.children.set(i, child.getCopyWithAddition(parentTree, t, addParentNode,
					additionalSubtree));
		}
		return t;
	}

	/**
	 * @param subtree
	 *            MUST be a subtree of this TreeNode (compared by reference)
	 * @return
	 */
	protected TreeNode getCopyWithoutSubtree(SmartTree parentTree, TreeNode parentNode,
			TreeNode subtree) {
		TreeNode t = getShallowCopy(parentTree, parentNode);

		for (int i = 0; i < t.children.size(); i++) {
			if (t.children.get(i) == subtree) {
				t.children.remove(i);
				i--;
			} else {
				TreeNode child = t.children.get(i);
				t.children.set(i, child.getCopyWithoutSubtree(parentTree, t, subtree));
			}
		}
		return t;
	}

	private TreeNode getShallowCopy(SmartTree parent, TreeNode parentNode) {
		TreeNode t;
		try {
			t = (TreeNode) this.clone();
			t.parentTree = parentTree;
			t.parentNode = parentNode;
		} catch (CloneNotSupportedException e) {
			throw new Error(e);
		}
		return t;
	}

	private TreeNode getDeepCopy(SmartTree parentTree, TreeNode parentNode) {
		TreeNode t;
		try {
			t = (TreeNode) this.clone();
			t.parentTree = parentTree;
			t.parentNode = parentNode;
			for (int i = 0; i < t.children.size(); i++) {
				t.children.set(i, t.children.get(i).getDeepCopy(parentTree, t));
			}
		} catch (CloneNotSupportedException e) {
			throw new Error(e);
		}
		return t;
	}

	public String toString() {
		return toString(LabelDisplay.SQUARE_BRACKETS);
	}

	public String getNodeLabel() {
		if (parentTree == null) {
			return treeLabelingIndex + "";
		} else {
			return treeName + treeLabelingIndex;
		}
	}

	public TreeNode getParentNode() {
		return parentNode;
	}

	public SmartTree getParentTree() {
		return parentTree;
	}

	public ArrayList<TreeNode> getLexicalTerminals() {
		ArrayList<TreeNode> results = new ArrayList<TreeNode>();
		getLexicalTerminals(results);
		return results;
	}

	/**
	 * Recursively sort all children in this tree according to their first value
	 */
	public void sort() {
		Collections.sort(children, new Comparator<TreeNode>() {
			public int compare(TreeNode a, TreeNode b) {
				if (a.values.size() > 0 && b.values.size() > 0) {
					return a.values.get(0).compareTo(b.values.get(0));
					// } else if (a.values.size() > 0 || b.values.size() > 0) {
					// return a.values.size() - b.values.size();
					// } else if (a.children.size() > 0 && b.children.size() >
					// 0) {
					// return compare(a.children.get(0), b.children.get(0));
					// } else if (a.children.size() > 0 || b.children.size() >
					// 0) {
					// return a.children.size() - b.children.size();
				} else {
					return 0;
				}
			}
		});
		for (TreeNode child : children) {
			child.sort();
		}
	}

	/**
	 * This method should only be applied to constituent structures.
	 */
	private void getLexicalTerminals(ArrayList<TreeNode> results) {
		if (this.isTerminal()) {
			assert this.values.size() == 2 : "A lexical terminal node must contain exactly 2 values!";
			results.add(this);
		} else {
			for (TreeNode child : children) {
				child.getLexicalTerminals(results);
			}
		}
	}

	public Object getMetaData(String key) {
		if (metadata == null) {
			return null;
		} else {
			return metadata.get(key);
		}
	}

	public void putMetaData(String key, Object value) {
		if (metadata == null) {
			this.metadata = new TreeMap<String, Object>();
		}
		this.metadata.put(key, value);
	}

	public String getYield() {
		StringBuilder builder = new StringBuilder();
		getYield(builder);
		return builder.toString().trim();
	}

	public void getYield(StringBuilder builder) {
		if (this.isTerminal()) {
			for (int i = 1; i < values.size(); i++) {
				builder.append(values.get(i) + " ");
			}
		} else {
			for (final TreeNode child : children) {
				child.getYield(builder);
			}
		}
	}

	public String toString(LabelDisplay labelDisp) {
		StringBuilder builder = new StringBuilder();
		toString(builder, labelDisp);
		return builder.toString();
	}

	public void toString(StringBuilder builder, LabelDisplay labelDisp) {

		if (labelDisp == LabelDisplay.DASHES) {
			builder.append(" (");
		}

		// ugly ternary garbage to format the node labels correctly...
		if (labelDisp == LabelDisplay.SQUARE_BRACKETS) {
			boolean wantsTerminalIndex =
					((parentTree == null || parentTree.getLabelMode() == SmartTree.LabelMode.LABEL_ALL_NODES) && isTerminal());
			if (hasTreeLabelingIndex() || wantsTerminalIndex) {
				builder.append(" [");
				if (hasTreeLabelingIndex()) {
					builder.append(getNodeLabel());
				}
				if (wantsTerminalIndex) {
					builder.append("@T" + (terminalIndex + 1));
				}
				builder.append("]");
			}
		} else if (labelDisp == LabelDisplay.DASHES) {
			if (hasTreeLabelingIndex()) {
				builder.append(getNodeLabel() + "-");
			}
		}

		if (labelDisp != LabelDisplay.DASHES) {
			builder.append(" (");
		}

		appendValuesAsString(builder);
		for (TreeNode child : children) {
			child.toString(builder, labelDisp);
		}
		builder.append(") ");
	}

	public String getCfgLhs() {
		assert values.size() > 0 : "no LHS";
		String lhs = values.get(0);
		return lhs.trim();
	}

	private static final String[] EMPTY_ARRAY = new String[0];

	public String[] getCfgRhs() {
		return getCfgRhs(EMPTY_ARRAY);
	}

	public String[] getCfgRhs(String... stringsToLexicalize) {

		if (this.isTerminal()) {
			if (values.size() > 1) {
				return new String[] { values.get(1) };
			} else {
				return new String[0];
			}
		} else {
			String[] rhs = new String[children.size()];
			for (int i = 0; i < rhs.length; i++) {
				TreeNode child = children.get(i);
				assert child.values.size() > 0 : "child has no RHS";

				String value = child.values.get(0);
				if (ArrayUtils.unsortedArrayContains(stringsToLexicalize, value)) {
					assert child.values.size() > 1 : "child cannot be lexicalized: " + child;
					rhs[i] = "\"" + child.values.get(1) + "\"";
				} else {
					rhs[i] = value;
				}
			}
			return rhs;
		}
	}

	public CfgRule toCfgRule() {
		return new CfgRule(getCfgLhs(), getCfgRhs());
	}

	public String toLatexString(boolean includeNodeLabels) {

		String strIndex;
		if (includeNodeLabels && hasTreeLabelingIndex())
			strIndex = "\\emph{" + getNodeLabel() + "}\\\\";
		else
			strIndex = "";

		if (isTerminal()) {
			return " [." + strIndex + getValuesAsString() + " ] ";
		} else {

			StringBuilder childString = new StringBuilder();
			for (TreeNode child : children) {
				childString.append(child.toLatexString(includeNodeLabels));
			}

			return " [." + strIndex + getValuesAsString() + childString.toString() + " ] ";
		}
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof TreeNode) {
			TreeNode other = (TreeNode) obj;
			if (this.values.size() != other.values.size()) {
				return false;
			} else {
				for (int i = 0; i < values.size(); i++) {
					if (!this.values.get(i).equals(other.values.get(i))) {
						return false;
					}
				}
				return true;
			}
		} else {
			return false;
		}
	}

	@Override
	public int hashCode() {
		String str = this.toString(LabelDisplay.NONE);
		return str.hashCode();
	}
}
