/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.trees.smart;

import java.util.ArrayList;

public class TreeGrepper {

	/**
	 * Recursive helper for matches(). This method is essentially a tree grep. A
	 * match was found if the matchList has size > 0.
	 * 
	 * @return A modified version of variableValuesAllowed reflecting the
	 *         instantiated variable value matches encountered during the
	 *         matching process.
	 */
	public static Variable[] matchesRecursively(TreeNode patternNode, TreeNode subtreeToSearch,
			Variable[] variableValuesAllowed, SymbolTable staticVariableData) {

		// NOTE: variableValuesAllowed will always have 2nd dimension arrays of
		// length 1 if the variable has already been matched or >=1 if the value
		// has not yet been matched

		Variable[] origVariableValuesAllowed = variableValuesAllowed;

		// this tree level has fNodes, see if our pattern has any
		// restrictions for them, and, if so, do they match?

		// TODO: Use hashing here? Or does == still make more sense since
		// we're using intern()?
		ArrayList<String> valuesToMatch = patternNode.getValues();
		ArrayList<String> fNodeValues = subtreeToSearch.getValues();

		for (String valueToMatch : valuesToMatch) {
			boolean matched = false;

			// first try literal matches
			for (String fNodeValue : fNodeValues) {
				assert fNodeValue == fNodeValue.intern() : "fNodeValue not interned";
				assert valueToMatch == valueToMatch.intern() : "valueToMatch not interned";
				if (fNodeValue == valueToMatch) {
					matched = true;
					break;
				}
			}

			// fall back to variable matches (slower)
			if (!matched) {

				// consider that the valueToMatch might be a variable
				variableValuesAllowed =
						matchVariable(variableValuesAllowed, origVariableValuesAllowed,
								staticVariableData, subtreeToSearch, valueToMatch);

				// no match was made: we fail
				if (variableValuesAllowed == null) {
					return null;
				}
			}

			// if we get to this point, we are successful so far (note the
			// return null above)

		} // for valueToMatch

		// we matched everything we needed to at this tree level, so continue
		// recursive matching if needed
		if (patternNode.getChildren().size() > 0) {

			// we have more pattern left to match, but nothing to match it to
			if (subtreeToSearch.getChildren().size() < 1)
				return null;

			// now we have to try the cross-product of patternNode-fSubtree
			// pairings
			// generally, the cross-product should be relatively small
			for (TreeNode patternChild : patternNode.getChildren()) {
				boolean matched = false;
				for (TreeNode fChild : subtreeToSearch.getChildren()) {
					Variable[] newVariableValuesAllowed =
							matchesRecursively(patternChild, fChild, variableValuesAllowed,
									staticVariableData);
					if (newVariableValuesAllowed != null) {
						// TODO: does this limit our matching power?
						variableValuesAllowed = newVariableValuesAllowed;
						matched = true;
						break;
					}
				}
				if (!matched) {
					return null;
				}
			}

			// we matched all of the required children
			return variableValuesAllowed;
		} else {
			// we matched everything there was to match
			return variableValuesAllowed;
		}
	}

	/**
	 * Checks if the given parameters constitute a variable match. This may
	 * involve further recursive matching of subtrees and nested variable
	 * matching.
	 */
	protected static Variable[] matchVariable(Variable[] variableValuesAllowed,
			Variable[] prevVariableValuesAllowed, SymbolTable staticVariableData, TreeNode fNode,
			String variableName) {

		boolean matched = false;

		Integer iVariableIndex = staticVariableData.variableIndices.get(variableName);
		if (iVariableIndex != null) {
			int nVariableIndex = iVariableIndex;

			// we didn't get a literal match, so try with variable
			// alternations BUT make sure we add matches for all
			// legal alternations if they work

			Variable valuesForThisVariable = variableValuesAllowed[nVariableIndex];
			for (TreeNode variableValue : valuesForThisVariable.getValues()) {

				// create dummy loner node for fNode to put it on equal footing
				// with the variable
				TreeNode dummyFNode = TreeNode.createOrphanNode(-1, SmartTree.F_STRUCT_LABEL);
				dummyFNode.addChild(fNode);

				Variable[] actualVariableValues =
						matchesRecursively(variableValue, dummyFNode, variableValuesAllowed,
								staticVariableData);
				if (actualVariableValues != null) {

					// record the value for this variable
					// such that we limit that allowed values
					// for this variable; if this is our first
					// modification, make a copy of the array so
					// that we don't mess up future match paths
					// if this one fails later
					if (variableValuesAllowed == prevVariableValuesAllowed) {
						variableValuesAllowed = new Variable[prevVariableValuesAllowed.length];
						for (int i = 0; i < variableValuesAllowed.length; i++) {
							variableValuesAllowed[i] = prevVariableValuesAllowed[i];
						}
					}
					variableValuesAllowed[nVariableIndex] = new Variable(variableValue);

					matched = true;
					break;
				}

				// if inner loop matched, break out of this one, too
				if (matched) {
					break;
				}
			}
		}

		if (!matched) {
			return null;
		} else {
			return variableValuesAllowed;
		}
	}
}
