/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.trees.smart;

import info.jonclark.util.DebugUtils;
import info.jonclark.util.StringUtils;

import java.lang.ref.SoftReference;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Vector;

/**
 * In the current implementation, the tree structure should remain immutable
 * upon creation. There might still be a few architectual leaks, but the user
 * should still take care not to violate this constraint.
 */
public class SmartTree {

	public static final String SOURCE_C_STRUCT_LABEL = "n";
	public static final String TARGET_C_STRUCT_LABEL = "t";
	public static final String F_STRUCT_LABEL = "f";

	private boolean sorted = false;
	private boolean pleaseSort = false;

	private int hash = 0;
	private TreeNode root = null;
	private int nextTreeLabelingIndex = 1;
	private int nextAbsoluteTreeIndex = 0;
	private int nextTerminalIndex = 0;
	private Vector<TreeNode> labeledNodeList = new Vector<TreeNode>();
	private final String name;
	private SoftReference<ArrayList<TreeNode>> labeledNodes =
			new SoftReference<ArrayList<TreeNode>>(null);
	private SoftReference<ArrayList<TreeNode>> unlabeledNodes =
			new SoftReference<ArrayList<TreeNode>>(null);
	private Vector<TreeNode> allNodes = new Vector<TreeNode>();
	private boolean structureChanged = false;

	public enum LabelMode {
		LABEL_ALL_NODES, LABEL_ODD_NODES
	}

	public enum LabelDisplay {
		NONE, SQUARE_BRACKETS, DASHES
	}

	private final LabelMode labelMode;

	private SmartTree(String name, LabelMode labelMode) {
		this.name = name;
		this.labelMode = labelMode;
		// Remember: The root MUST be set!
	}

	public LabelMode getLabelMode() {
		return labelMode;
	}

	public String getName() {
		return name;
	}

	/**
	 * Creates a new SmartTree that is empty except for having a single root
	 * node.
	 * 
	 * @param name
	 * @param labelMode
	 * @return
	 */
	public static SmartTree createRootedTree(String name, LabelMode labelMode) {
		SmartTree tree = new SmartTree(name, labelMode);
		tree.root = new TreeNode(tree, null, 0, 0);
		return tree;
	}

	public static SmartTree createTreeFromRoot(String name, LabelMode labelMode, TreeNode root) {
		SmartTree tree = new SmartTree(name, labelMode);
		tree.root = root;
		tree.fireTreeStructureChanged();
		tree.updateTreeStructure();
		return tree;
	}

	public static SmartTree parse(String strTree, String name, LabelMode labelMode)
			throws ParseException {

		strTree = strTree.trim();
		if (strTree.startsWith("[")) {
			strTree = strTree.substring(strTree.indexOf(']') + 1).trim();
		}

		assert strTree.length() > 0 : "Tree must have length greater than zero: " + strTree;
		assert strTree.charAt(0) == '(' : "Tree must start with '(': " + strTree;
		assert strTree.charAt(strTree.length() - 1) == ')' : "Tree must end with ')':" + strTree;

		SmartTree tree = new SmartTree(name, labelMode);
		tree.nextTreeLabelingIndex = 1;
		tree.root =
				parseRecursively(strTree.substring(1, strTree.length() - 1), tree, null, 2,
						labelMode);

		if (tree.labeledNodeList.size() < 1)
			tree.labeledNodeList.setSize(1);
		tree.labeledNodeList.setElementAt(tree.root, 0);

		tree.fireTreeStructureChanged();
		tree.updateTreeStructure();

		if (DebugUtils.isAssertEnabled()) {
			for (int i = 0; i < tree.labeledNodeList.size(); i++) {
				assert tree.labeledNodeList.get(i) != null : "Null node found in nodeList at index "
						+ i + " for tree: " + tree;
			}
		}

		return tree;
	}

	/**
	 * Responsible for intern()ing all values
	 * 
	 * @param currentNode
	 * @param tree
	 * @param parentNode
	 * @param treeIndex
	 * @param depth
	 * @param labelMode
	 * @return
	 * @throws ParseException
	 */
	private static TreeNode parseRecursively(String currentNode, SmartTree tree,
			TreeNode parentNode, int depth, LabelMode labelMode) throws ParseException {

		// System.out.println("parsing: " + currentNode);

		currentNode = currentNode.trim();
		TreeNode treeNode = new TreeNode(tree, parentNode, -1, -1);

		int nChildStart;
		int nChildEnd = -1;
		while ((nChildStart = currentNode.indexOf('(', nChildEnd)) != -1) {

			if (nChildStart != nChildEnd + 1) {
				String values = currentNode.substring(nChildEnd + 1, nChildStart);
				parseValues(values, treeNode, tree);
			}

			nChildEnd = StringUtils.findMatching(currentNode, nChildStart, '(', ')');
			assert nChildEnd < currentNode.length() : "Matching paren out of range: " + nChildEnd
					+ " for string of length " + currentNode.length();

			String nestedNode = currentNode.substring(nChildStart + 1, nChildEnd);

			// only assign node index labels at odd depths
			if (labelMode == LabelMode.LABEL_ODD_NODES && depth % 2 == 0) {

				// do not give children node labels
				TreeNode child = parseRecursively(nestedNode, tree, treeNode, depth + 1, labelMode);
				treeNode.addChild(child);

			} else {
				TreeNode child = parseRecursively(nestedNode, tree, treeNode, depth + 1, labelMode);
				treeNode.addChild(child);
			}

		}

		if (nChildEnd + 1 < currentNode.length()) {
			String remainingValues = currentNode.substring(nChildEnd + 1, currentNode.length());
			parseValues(remainingValues, treeNode, tree);
		}

		return treeNode;
	}

	private static void parseValues(String values, TreeNode treeNode, SmartTree tree)
			throws ParseException {

		for (String value : StringUtils.tokenizeQuotedValues(values, " \t\n", "\"", "\"", false,
				true, Integer.MAX_VALUE)) {

			// ignore node labels
			if (value.startsWith("["))
				continue;

			treeNode.addValue(value.intern());

			if (value.contains("("))
				throw new ParseException("Values contains leading paren: " + values, -1);
			if (value.contains(")"))
				throw new ParseException("Values contains trailing paren: " + values, -1);

			// System.out.println("Added value: " + value);
		}
	}

	/**
	 * Clear items in the cache. This cache is used purely as a space-time
	 * tradeoff and clearing it will only affect performance.
	 */
	public void clearCache() {
		labeledNodes.clear();
		unlabeledNodes.clear();
	}

	/**
	 * Called by a child node when a structural change in the tree occurs (i.e.
	 * a child is added).
	 * 
	 * @param node
	 */
	protected void fireTreeStructureChanged() {
		this.structureChanged = true;
		this.sorted = false;
	}

	/**
	 * Updates terminal indices, but not tree indices.
	 */
	public void updateTreeStructure() {

		if (structureChanged) {
			this.structureChanged = false;

			this.nextTerminalIndex = 0;
			labelTerminals(this, root);

			labeledNodeList.clear();
			labeledNodes.clear();
			unlabeledNodes.clear();
			allNodes.clear();

			this.nextTreeLabelingIndex = 1;
			this.nextAbsoluteTreeIndex = 0;
			labelNodes(this, root, 0);

			if (DebugUtils.isAssertEnabled()) {
				for (int i = 0; i < labeledNodeList.size(); i++) {
					assert labeledNodeList.get(i) != null : "Null node found in nodeList at index "
							+ i + " for tree: " + this;
				}
			}

			if (pleaseSort && !sorted) {
				root.sort();
			}
		}
	}

	private static void labelTerminals(SmartTree tree, TreeNode node) {

		if (node.isTerminal()) {

			node.terminalIndex = tree.nextTerminalIndex;
			tree.nextTerminalIndex++;

		} else {
			node.terminalIndex = -1;

			for (TreeNode child : node.getChildren()) {
				labelTerminals(tree, child);
			}
		}
	}

	private void labelNodes(SmartTree tree, TreeNode node, int depth) {

		node.parentTree = tree;

		// tree labeling index is ONE BASED
		if (labelMode == LabelMode.LABEL_ALL_NODES
				|| (labelMode == LabelMode.LABEL_ODD_NODES && depth % 2 == 0)) {
			node.treeLabelingIndex = tree.nextTreeLabelingIndex;
			tree.nextTreeLabelingIndex++;

			if (tree.labeledNodeList.size() <= node.getTreeLabelingIndex()) {
				tree.labeledNodeList.setSize(node.getTreeLabelingIndex());
			}
			tree.labeledNodeList.setElementAt(node, node.getTreeLabelingIndex() - 1);
		}

		node.absoluteTreeIndex = tree.nextAbsoluteTreeIndex;
		tree.nextAbsoluteTreeIndex++;

		// absolute tree index is ZERO BASED
		if (tree.allNodes.size() <= node.getAbsoluteTreeIndex()) {
			tree.allNodes.setSize(node.getAbsoluteTreeIndex() + 1);
		}
		tree.allNodes.setElementAt(node, node.getAbsoluteTreeIndex());

		for (TreeNode child : node.getChildren()) {
			labelNodes(tree, child, depth + 1);
		}
	}

	/**
	 * Gets a tree node by its 1-based tree labeling index.
	 * 
	 * @param treeLabelingIndex
	 * @return
	 */
	public TreeNode getByTreeLabelingIndex(int treeLabelingIndex) {
		updateTreeStructure();

		assert treeLabelingIndex > 0 : "treeIndex should be 1-based";
		TreeNode node = labeledNodeList.elementAt(treeLabelingIndex - 1);
		assert node != null : "Null node found";
		assert node.getTreeLabelingIndex() == treeLabelingIndex : "Index/element# mismatch (requested tree index = "
				+ treeLabelingIndex
				+ ") (actual tree index = "
				+ node.getTreeLabelingIndex()
				+ "): " + node;
		return node;
	}

	public TreeNode getByAbsoluteTreeIndex(int absoluteTreeIndex) {
		updateTreeStructure();

		assert absoluteTreeIndex >= 0 : "treeIndex should be 0-based";
		TreeNode node = getAllNodes().get(absoluteTreeIndex);
		assert node != null : "Null node found";
		assert node.getAbsoluteTreeIndex() == absoluteTreeIndex : "Index/element# mismatch (requested tree index = "
				+ absoluteTreeIndex
				+ ") (actual tree index = "
				+ node.getAbsoluteTreeIndex()
				+ "): " + node;
		return node;
	}

	public TreeNode getRootNode() {
		return root;
	}

	/**
	 * For a constituent structure, gets all nodes. For a feature structure,
	 * gets the participant nodes, clause nodes, and modifier nodes. Order is
	 * bottom-up (post-order traversal).
	 * 
	 * @return
	 */
	public ArrayList<TreeNode> getLabeledNodes() {
		updateTreeStructure();

		// make use of memory-sensitive caching to improve performance
		if (labeledNodes.get() == null) {
			labeledNodes = new SoftReference<ArrayList<TreeNode>>(root.getLabeledNodes());
		}

		return labeledNodes.get();
	}

	public ArrayList<TreeNode> getUnlabeledNodes() {
		updateTreeStructure();

		// make use of memory-sensitive caching to improve performance
		if (unlabeledNodes.get() == null) {
			unlabeledNodes = new SoftReference<ArrayList<TreeNode>>(root.getUnlabeledNodes());
		}

		return unlabeledNodes.get();
	}

	public Vector<TreeNode> getAllNodes() {
		updateTreeStructure();
		return allNodes;
	}

	public ArrayList<TreeNode> getTerminalNodes() {
		updateTreeStructure();
		return root.getTerminalNodes();
	}

	public String toString() {
		return toString(LabelDisplay.SQUARE_BRACKETS);
	}

	public String toString(LabelDisplay labelDisp) {
		return root.toString(labelDisp).trim();
	}

	public String toLatexString(boolean showNodeLabels) {
		return "\\Tree " + root.toLatexString(showNodeLabels);
	}

	/**
	 * @param subtree
	 *            MUST be a subtree of this SmartTree (compared by reference)
	 * @return
	 */
	public SmartTree getCopyWithoutSubtree(TreeNode subtree) {
		SmartTree tree = new SmartTree(this.name, this.labelMode);
		tree.root = this.root.getCopyWithoutSubtree(tree, null, subtree);
		tree.fireTreeStructureChanged();
		tree.updateTreeStructure();
		return tree;
	}

	public SmartTree getCopyWithValueSubstitution(TreeNode affectedSubtree, String possibleClause) {
		SmartTree tree = new SmartTree(this.name, this.labelMode);
		throw new Error("unimplemented");
		// tree.root = this.root.getCopyWithValueSubstitution(tree, null,
		// affectedSubtree, possibleClause);
		// tree.fireTreeStructureChanged();
		// tree.updateTreeStructure();
		// return tree;
	}

	public SmartTree getCopyWithAddition(TreeNode parentNode, TreeNode additionalSubtree) {
		SmartTree tree = new SmartTree(this.name, this.labelMode);
		tree.root =
				this.root.getCopyWithNodeSubstitution(tree, null, parentNode, additionalSubtree);
		tree.fireTreeStructureChanged();
		tree.updateTreeStructure();
		return tree;
	}

	/**
	 * Recursively sort all children in this tree according to their first value
	 */
	public void sort() {
		pleaseSort = true;
		if (pleaseSort && !sorted) {
			root.sort();
		}
	}

	public String getYield() {
		return root.getYield();
	}

	public int hashCode() {
		if (hash == 0) {
			hash = toString(LabelDisplay.NONE).hashCode();
		}
		return hash;
	}

	public static void main(String[] args) throws Exception {
		// String strTest = "((agent ((feature value) (another-feature 2))))";
		String strTest =
				"((actor ((np-function fn-actor) (np-general-type proper-noun-type)(np-person person-third) (np-biological-gender bio-gender-female)(np-animacy anim-human) (np-identifiability identifiable)(np-pronoun-antecedent antecedent-n/a)(np-number num-sg)(np-specificity specific)  (np-distance distance-neutral)(np-pronoun-exclusivity inclusivity-n/a)))    (undergoer ((np-function fn-undergoer)(np-person person-third)(np-identifiability unidentifiable) (np-number num-pl)(np-specificity non-specific)(np-animacy anim-inanimate)(np-biological-gender bio-gender-n/a)(np-general-type common-noun-type)(np-pronoun-exclusivity inclusivity-n/a)(np-pronoun-antecedent antecedent-n/a)(np-distance distance-neutral)))  (c-polarity polarity-positive) (c-v-absolute-tense past) (c-v-lexical-aspect activity-accomplishment)(c-general-type declarative-clause)(c-my-causer-intentionality intentionality-n/a)(c-comparison-type comparison-n/a)(c-relative-tense relative-n/a)(c-our-boundary boundary-n/a)(c-comparator-function comparator-n/a)(c-causee-control control-n/a)(c-our-situations situations-n/a)(c-comparand-type comparand-n/a)(c-causation-directness directness-n/a)(c-source source-neutral)(c-causee-volitionality volition-n/a)(c-assertiveness assertiveness-neutral)(c-solidarity solidarity-neutral)(c-v-grammatical-aspect gram-aspect-neutral)(c-adjunct-clause-type adjunct-clause-type-n/a)(c-v-phase-aspect phase-aspect-neutral)(c-secondary-type secondary-neutral)(c-event-modality event-modality-none)(c-function fn-main-clause)(c-minor-type minor-n/a)(c-copula-type copula-n/a)(c-power-relationship power-peer)(c-our-shared-subject shared-subject-n/a)(c-question-gap gap-n/a))";
		SmartTree tree = parse(strTest, "n", LabelMode.LABEL_ODD_NODES);

		System.out.println(tree.toString());

		for (int i = 0; i < tree.labeledNodeList.size(); i++)
			System.out.println(i + ": " + tree.labeledNodeList.get(i));
	}
}
