package edu.cmu.cs.lti.avenue.projection.ordering;

import info.jonclark.util.ArrayUtils;
import info.jonclark.util.DebugUtils;

import java.util.ArrayList;
import java.util.Arrays;

import edu.cmu.cs.lti.avenue.corpus.Alignment;
import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.corpus.RawAlignment;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

public abstract class OrderingModel {

	/**
	 * Get the absolute target position of node a as projected to a target
	 * language as given by the alignment. Note that the ordering of multiple
	 * nodes as given by <code>getOrder()</code> may depend on more features
	 * than just this absolute position.
	 * 
	 * @param t
	 * @return -1 If target language position cannot be determined.
	 * @throws CorpusException
	 */
	public abstract double getPosition(TreeNode targetNode, Alignment alignment)
			throws CorpusException;

	/**
	 * Gets the relative order of a list of TreeNodes as projected into a target
	 * language with the given word alignments.<br>
	 * NOTE: This method should fail (return NULL) if there is an ambiguity due
	 * to alignments coming from two different source-side constituents as well.
	 * 
	 * @param sourceNodes
	 * @param alignment
	 * @return An array of the same length of nodes with each integer element in
	 *         the array referring to the index of the reordered element in the
	 *         original source side node list *OR* NULL if no ordering could be
	 *         determined.
	 * @throws CorpusException
	 */
	// XXX: We never actually wanted to order the sourceNodes since we already
	// know how they're ordered; we're really only interested in ordering the
	// target nodes. So why did we ever try to order the soruceNodes?
	public Ordering getOrder(ArrayList<TreeNode> targetNodes, Alignment alignment)
			throws CorpusException {

		Ordering ordering = new Ordering();

		// 1) get the absolute positions of each element in the array
		double[] doublePositions = new double[targetNodes.size()];
		for (int i = 0; i < targetNodes.size(); i++) {
			doublePositions[i] = getPosition(targetNodes.get(i), alignment);
		}

		// 2) map these positions to integer indices
		// this should be fairly quick since we're dealing with arrays of < 5
		// elements
		double[] mapping = new double[doublePositions.length];
		System.arraycopy(doublePositions, 0, mapping, 0, doublePositions.length);
		Arrays.sort(mapping);

		// 3) use the mapping to great a final ordering array based on integer
		// indices
		int[] pos = new int[doublePositions.length];
		for (int i = 0; i < pos.length; i++) {
			pos[i] = ArrayUtils.indexOf(mapping, doublePositions[i]);
		}

		// 4) check if any duplicated indices came from the same alignment
		// groups (meaning we can disambiguate them)
		int prev = pos.length > 0 ? pos[0] : -1;
		for (int i = 1; i < pos.length; i++) {
			if (pos[i] != prev) {
				prev = pos[i];
			} else {
				prev = pos[i];

				if (hasSameAlignments(alignment, targetNodes.get(i - 1), targetNodes.get(i))) {
					// there is no interference here, just keep the original
					// target ordering
					pos[i] = pos[i - 1] + 1;
				} else {
					ordering.ambiguitiesByGrouping++;
					ordering.ambiguous = true;
				}
			} // end if pos[i] == prev
		}

		// 5) turn the indices back into nodes
		TreeNode[] result = new TreeNode[doublePositions.length];
		for (int i = 0; i < pos.length; i++) {
			result[pos[i]] = targetNodes.get(i);
		}

		ordering.orderedNodes = ArrayUtils.toArrayList(result);
		ordering.absolutePositions = doublePositions;
		ordering.relativePositions = pos;
		
		if(DebugUtils.isAssertEnabled()) {
			for(int i=0; i<result.length; i++) {
				assert result[i] != null : "null result element";
				assert ordering.orderedNodes.get(i) != null : "null orderedNode";
			}
		}
		
		return ordering;
	}

	private boolean hasSameAlignments(Alignment alignment, TreeNode targetNodeA,
			TreeNode targetNodeB) {
		
		ArrayList<RawAlignment> alignmentA = alignment.getRawAlignmentsForTarget(targetNodeA);
		ArrayList<RawAlignment> alignmentB = alignment.getRawAlignmentsForTarget(targetNodeB);

		boolean sameAlignment = true;
		if (alignmentA.size() != alignmentB.size()) {
			sameAlignment = false;
		} else {
			for (int j = 0; j < alignmentA.size(); j++) {

				// we can do a reference comparison on these
				if (alignmentA.get(j) != alignmentB.get(j)) {
					sameAlignment = false;
					break;
				}
			}
		}
		return sameAlignment;
	}
}
