package edu.cmu.cs.lti.avenue.projection;

import java.util.ArrayList;

import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.trees.cfg.SyncCfgRule;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

public class ProjectionAnalyzer {

	/**
	 * Should only be applied to sentence pairs that have already been run
	 * through <code>project()</code>
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static ArrayList<SyncCfgRule> getLearnedRules(SentencePair pair, boolean includeFeatures) {
		ArrayList<TreeNode> sourceNodes = pair.getSourceConstituentStructure().getLabeledNodes();
		ArrayList<SyncCfgRule> rules = new ArrayList<SyncCfgRule>(sourceNodes.size());

		for (final TreeNode sourceNode : sourceNodes) {
			ProjectionFeatures features =
					(ProjectionFeatures) sourceNode.getMetaData(ConstituentStructureProjector.FEATURES);
			ArrayList<TreeNode> targetNodes =
					(ArrayList<TreeNode>) sourceNode.getMetaData(ConstituentStructureProjector.LINKS);

			if (targetNodes == null) {
				continue;
			} else if (targetNodes.size() != 1) {
				// log.warning("Ignoring rule with multiple LHS.");
			} else {
				TreeNode targetNode = targetNodes.get(0);
				SyncCfgRule rule =
						new SyncCfgRule(sourceNode.getCfgLhs(), targetNode.getCfgLhs(),
								sourceNode.getCfgRhs(ConstituentStructureProjector.LEX),
								targetNode.getCfgRhs(ConstituentStructureProjector.LEX));

				if (includeFeatures) {
					rule.putMetaData(ConstituentStructureProjector.FEATURES, features.toString());
				}
				rules.add(rule);
				// TODO: Correspondences ((X1::Y4))
			}
		}

		return rules;
	}

	/**
	 * Should only be applied to sentence pairs that have already been run
	 * through <code>project()</code>. Returns the source nodes that could
	 * not be unambiguously linked to target nodes.
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static ArrayList<String> getFailedRules(SentencePair pair) {
		ArrayList<TreeNode> sourceNodes = pair.getSourceConstituentStructure().getLabeledNodes();
		ArrayList<String> rules = new ArrayList<String>(sourceNodes.size());

		for (final TreeNode sourceNode : sourceNodes) {
			ProjectionFeatures features =
					(ProjectionFeatures) sourceNode.getMetaData(ConstituentStructureProjector.FEATURES);
			ArrayList<TreeNode> targetNodes =
					(ArrayList<TreeNode>) sourceNode.getMetaData(ConstituentStructureProjector.LINKS);

			if (targetNodes == null) {
				rules.add(sourceNode.toCfgRule() + " (" + features.toString() + ")");
			}
		}

		return rules;
	}

	/**
	 * Should only be applied to sentence pairs that have already been run
	 * through <code>project()</code>
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static ArrayList<String> getLearnedLinks(SentencePair pair) {
		ArrayList<TreeNode> sourceNodes = pair.getSourceConstituentStructure().getLabeledNodes();
		ArrayList<String> listOfLinks = new ArrayList<String>(sourceNodes.size());

		for (final TreeNode sourceNode : sourceNodes) {
			ArrayList<TreeNode> targetLinks =
					(ArrayList<TreeNode>) sourceNode.getMetaData(ConstituentStructureProjector.LINKS);
			if (targetLinks == null) {
				continue;
			} else {
				StringBuilder builder = new StringBuilder();
				builder.append("[" + sourceNode.getNodeLabel() + "] -> [");
				for (final TreeNode targetNode : targetLinks) {
					builder.append(targetNode.getNodeLabel() + " ");
				}
				builder.append("]");
				listOfLinks.add(builder.toString());
			}
		}

		return listOfLinks;
	}

	// TODO: Frequency of lexicalized words.
	public static ProjectionFeatures getSentenceLevelStatistics(SentencePair pair) {
		ProjectionFeatures overall = new ProjectionFeatures();

		ArrayList<TreeNode> sourceNodesBottomUp =
				pair.getSourceConstituentStructure().getLabeledNodes();
		for (final TreeNode sourceNode : sourceNodesBottomUp) {

			Object obj = sourceNode.getMetaData(ConstituentStructureProjector.FEATURES);
			if (obj != null) {
				ProjectionFeatures nodeFeatures = (ProjectionFeatures) obj;
				boolean linked =
						(sourceNode.getMetaData(ConstituentStructureProjector.LINKS) != null);

				// count source constituent level that had to be skipped due to
				// interruptions or dual alignments
				if (linked) {
					overall.ed += nodeFeatures.ed;
					overall.dus += nodeFeatures.dus;
					overall.dut += nodeFeatures.dut;
					overall.lo += nodeFeatures.lo; // only for (top - 1) node
				} else {
					overall.di += nodeFeatures.di;
					// overall.ag += nodeFeatures.ag;
					if (nodeFeatures.dd > 0) {
						overall.dd++;
					}
				}
			}
		}

		overall.r =
				ProjectionFeatureExtractor.countReorderings(pair, null, 0,
						pair.getNormalizedSourceTokens().length - 1);

		return overall;
	}
}
