/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.tools;

import info.jonclark.properties.SmartProperties;
import info.jonclark.util.ArrayUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import edu.cmu.cs.lti.avenue.corpus.Corpus;
import edu.cmu.cs.lti.avenue.corpus.PhiPlusMapping;
import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.corpus.Serializer;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelDisplay;

/**
 * A tool to aide in creating "Phi Plus" mappings. LFG Phi mappings that are
 * augmented with information about the lexical head.
 */
public class PhiPlusMapper {

	public static void main(String[] args) throws Exception {

		if (args.length != 1) {
			System.err.println("Usage: <properties_file>");
			System.exit(1);
		}

		SmartProperties props = new SmartProperties(args[0]);
		String encoding = props.getPropertyString("global.encoding");

		File inputFile = props.getPropertyFile("paths.cfUntranslatedCorpus");
		File outputFile = props.getPropertyFile("paths.phiUntranslatedCorpus");
		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		PrintWriter out = new PrintWriter(outputFile);

		int autoPhi = 0;
		int autoHead = 0;

		System.out.println("Reading CF Structure Untranslated Corpus: "
				+ inputFile.getAbsolutePath());
		Corpus inputPairs = Serializer.loadSentencePairs(inputFile, encoding);
		System.out.println("Loaded " + inputPairs.getSentences().size() + " input sentences.");

		// for each CF-struct in file
		for (SentencePair pair : inputPairs.getSentences()) {

			SmartTree fStruct = pair.getFeatureStructure();
			SmartTree cStruct = pair.getSourceConstituentStructure();

			assert fStruct != null : "null fStruct: " + pair.getMyLine();
			assert cStruct != null : "null cStruct: " + pair.getMyLine();

			// show user the labeled c-struct and f-struct
			System.out.println(fStruct.toString(LabelDisplay.SQUARE_BRACKETS));
			System.out.println(cStruct.toString(LabelDisplay.SQUARE_BRACKETS));
			System.out.println(pair.getConstituentSourceLine());
			System.out.println(pair.getFeatureSourceLine());
			System.out.println(pair.getMyLine());

			// for each node of the f-struct, ask the user to name the
			// corresponding
			// c-struct nodes

			ArrayList<TreeNode> fNodes = fStruct.getLabeledNodes();
			Vector<TreeNode> cNodes = cStruct.getAllNodes();

			int[][] phiInverse = new int[fNodes.size()][];
			int[] headMapping = new int[cNodes.size()];

			// initialize arrays
			for (int i = 0; i < phiInverse.length; i++) {
				phiInverse[i] = new int[0];
			}
			for (int i = 0; i < headMapping.length; i++) {
				headMapping[i] = -1;
			}

			// TODO: Try to automatically determine the phi mapping:
			// determine whether first NP is actor or undergoer based on what
			// type of actor there is

			// ask for phi mapping
			int i = 0;
			for (final TreeNode fNode : fNodes) {
				int[] indices = new int[1];
				indices[0] = findMatchingNode(fNode, cNodes);

				if (indices[0] == -1) {
					// get the space-delimited numbers from the user
					System.out.print(PhiPlusMapping.PHI_INVERSE + "(" + fNode.getNodeLabel()
							+ ")=? ");
					// String line = in.readLine();
					// String[] tokens = StringUtils.tokenize(line);
					// indices = StringUtils.toIntArray(tokens);
					autoPhi++;
					indices = new int[1];
					indices[0] = 1;
				}

				// convert to zero-base from one-base
				for (int j = 0; j < indices.length; j++)
					indices[j]--;

				phiInverse[i] = indices;
				i++;
			}

			// ask for head mapping
			for (final TreeNode cNode : cNodes) {

				// attempt to automatically determine the head
				int nHead = findHead(cNode, headMapping);

				// we must ask the user for the head
				if (nHead == -1) {
					System.out.print(PhiPlusMapping.HEAD + "(" + cNode.getNodeLabel() + ")=? ");
					// String line = in.readLine();
					// nHead = Integer.parseInt(line);
					nHead = cNode.getTreeLabelingIndex();
					autoHead++;
				}

				// convert one-base to zero-base
				headMapping[cNode.getTreeLabelingIndex() - 1] = nHead - 1;
			}

			// build our phi-plus mapping
			PhiPlusMapping mapping = new PhiPlusMapping(phiInverse, headMapping);
			pair.setPhiPlusMapping(mapping);
			for (int j = 1; j < pair.getNormalizedSourceTokens().length + 1; j++) {
				mapping.getUltimateHead(j);
			}
			System.out.println(mapping.toString() + "\n\n");

			System.out.println("HEAD: "
					+ pair.getDisplaySourceSentence()
					+ " ("
					+ mapping.getUltimateHead(pair,
							pair.getSourceConstituentStructure().getRootNode()).toString() + ")");

			// save incrementally
			out.println(pair.serialize());
			out.flush();
		}

		in.close();
		System.out.println("WARNING: " + autoHead
				+ " heads were assigned the default value of self and " + autoPhi
				+ " phi mappings were assigned the default value of main clause.");
		System.out.println("Automatically detected and broke " + cycles + " cycles out of " + heads
				+ " heads.");
		System.out.println("WROTE CF + PHI UNTRANSLATED CORPUS: " + outputFile.getAbsolutePath());
		System.out.println("Done.");

	}

	/**
	 * A one-time use hack for mapping the current incarnation of the
	 * elicitation corpus f-structures to the current incarnation of the
	 * c-structures. If you intend to use this method elsewhere, look carefully
	 * at what it does; the intent was not for reuse.
	 * 
	 * @param fNode
	 * @param cNodes
	 * @return
	 */
	public static int findMatchingNode(TreeNode fNode, List<TreeNode> cNodes) {
		TreeNode parent = fNode.getParentNode();

		// TODO: HAS_SAID and HAS_WANTED constructions (...but not all?)
		final String[] SUBJECTS =
				new String[] { "focus-undergoer", "actor", "perceiver", "cognizer" };
		final String[] UNDERGOER_LIKE_OBJECTS =
				new String[] { "causee", "undergoer", "stimulus", "pred", "recipient" };
		final String[] COMPLEMENTS = new String[] { "comp", "complement" };
		final String[] RELATIVE_CLAUSE = new String[] { "rel-clause" };

		if (parent != null) {

			// check if we're nested
			TreeNode grandparent = parent.getParentNode();
			if (grandparent != null && grandparent.getValues().size() > 0) {

				String grandparentValue = grandparent.getValues().get(0);
				if (ArrayUtils.unsortedArrayContains(COMPLEMENTS, grandparentValue)) {
					// we're in a complement, so zoom in to the complement
					// clause of the c-structure
					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// remember to update criteria for children in
						// non-recursive rule below!
						if (cValue.startsWith("SBAR")) {
							return findMatchingNode(fNode, cNode.getChildren());
						}
					}

					// we could also be looking for a SQ following a quote
					for (int i = 0; i < cNodes.size(); i++) {
						TreeNode cNode = cNodes.get(i);
						String cValue = cNode.getValues().get(0);

						if (cValue.equals("``")) {
							while (i < cNodes.size()) {
								cNode = cNodes.get(i);
								cValue = cNode.getValues().get(0);
								if (cValue.equals("SQ") || cValue.equals("S")) {
									return findMatchingNode(fNode, cNode.getChildren());
								}
								i++;
							}
						}
					}

				} else if (ArrayUtils.unsortedArrayContains(UNDERGOER_LIKE_OBJECTS,
						grandparentValue)) {

					// we're nested in something like an undergoer
					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("VP") || cValue.equals("SQ")) {
							for (TreeNode cNode2 : cNode.getLabeledNodes()) {
								String cValue2 = cNode2.getValues().get(0);
								// ADJP is specifically for pred
								if (cValue2.equals("NP") || cValue2.equals("ADJP")) {
									return findMatchingNode(fNode, cNode2.getChildren());
								}
							}
						}
					}
				} else if (ArrayUtils.unsortedArrayContains(RELATIVE_CLAUSE, grandparentValue)) {

					// we're nested in a rel-clause
					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("SBAR")) {
							return findMatchingNode(fNode, cNode.getChildren());
						}
					}
				} else {
					System.out.println("Uknown type of nested clause: " + grandparentValue);
				} // checking what type of clause we're in

			} else {

				// not nested in any important way
				assert parent.getValues().size() > 0 : "Zero-length value list for: "
						+ parent.toString();
				String fParentValue = parent.getValues().get(0);

				if (fParentValue.equals("instrumental")) {
					// match to the phrase starting with "with"

					for (TreeNode cNode : cNodes) {
						if (cNode.getValues().size() > 1 && cNode.getValues().get(1).equals("with")) {
							return cNode.getParentNode().getTreeLabelingIndex();
						}
					}

				} else if (fParentValue.equals("beneficiary")) {
					// match to the phrase starting with "for"

					for (TreeNode cNode : cNodes) {
						if (cNode.getValues().size() > 1 && cNode.getValues().get(1).equals("for")) {
							return cNode.getParentNode().getTreeLabelingIndex();
						}
					}

				} else if (fParentValue.equals("non-focus-actor")) {
					// match to the phrase starting with "by"

					for (TreeNode cNode : cNodes) {
						if (cNode.getValues().size() > 1 && cNode.getValues().get(1).equals("by")) {
							return cNode.getParentNode().getTreeLabelingIndex();
						}
					}

				} else if (ArrayUtils.unsortedArrayContains(SUBJECTS, fParentValue)) {
					// TODO: this will probably work for all "focus" features
					// find the first NP and hope for the best

					return findSubject(grandparent, cNodes);

				} else if (fParentValue.equals("causer")) {
					// TODO: this will probably work for all "focus" features
					// find the first NP and hope for the best

					return findSubject(grandparent, cNodes);

				} else if (ArrayUtils.unsortedArrayContains(UNDERGOER_LIKE_OBJECTS, fParentValue)) {

					// if this is an undergoer and the f-struct has a recipient,
					// then we take the second NP
					// that is:
					// sentence has DO && undergoer => first NP under VP
					// sentence has IO && undergoer => second NP under VP
					boolean takeNextNPunderVP =
							!(fParentValue.equals("undergoer") && hasParentLabel(grandparent,
									"recipient"));

					boolean findSubject = true;
					if (fParentValue.equals("undergoer")) {
						for (String subject : SUBJECTS) {
							if (hasParentLabel(grandparent, subject)) {
								findSubject = false;
								break;
							}
						}
					}

					if (findSubject) {
						return findSubject(grandparent, cNodes);
					} else {
						return findDirectObject(cNodes, takeNextNPunderVP);
					}

				} else if (fParentValue.equals("causative-undergoer")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("VP") || cValue.equals("SQ")) {
							for (TreeNode cNode2 : cNode.getLabeledNodes()) {
								String cValue2 = cNode2.getValues().get(0);
								if (cValue.equals("VP") || cValue.equals("SQ")) {
									for (TreeNode cNode3 : cNode.getLabeledNodes()) {
										String cValue3 = cNode2.getValues().get(0);
										if (cValue3.equals("NP")) {
											return cNode3.getTreeLabelingIndex();
										}
									}
								}
							}
						}
					}

				} else if (ArrayUtils.unsortedArrayContains(COMPLEMENTS, fParentValue)) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// remember to update criteria for children in recursive
						// rule above!
						if (cValue.startsWith("SBAR")) {
							return cNode.getTreeLabelingIndex();
						}
					}

					// we could also be looking for a SQ following a quote
					for (int i = 0; i < cNodes.size(); i++) {
						TreeNode cNode = cNodes.get(i);
						String cValue = cNode.getValues().get(0);

						if (cValue.equals("``")) {
							while (i < cNodes.size()) {
								cNode = cNodes.get(i);
								cValue = cNode.getValues().get(0);
								if (cValue.equals("SQ") || cValue.equals("S")) {
									return cNode.getTreeLabelingIndex();
								}
								i++;
							}
						}
					}

				} else if (fParentValue.equals("modifier")) {

					for (TreeNode modChild : fNode.getChildren()) {
						if (!modChild.getValues().get(0).equals("mod-role"))
							continue;

						String modRole = modChild.getValues().get(1);
						if (modRole.equals("mod-ordinal-first")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";
								if (cLex.equals("first")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-ordinal-second")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";
								if (cLex.equals("second")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole == "mod-cardinal-one" || modRole == "mod-cardinal-two"
								|| modRole == "mod-cardinal-high-number") {
							for (TreeNode cNode : cNodes) {
								String cTag = cNode.getValues().get(0);
								if (cTag == "CD") {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-addition")) {
							for (int i = 0; i < cNodes.size(); i++) {
								TreeNode cNode = cNodes.get(i);
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								// XXX: TODO: Search for "in addition", but
								// match addition first

								if (cLex.equals("addition")) {
									while (!cLex.equals("in") && i > 0) {
										i--;
										cNode = cNodes.get(i);
										cLex =
												cNode.getValues().size() > 1
														? cNode.getValues().get(1) : "";
									}

									if (cLex.equals("in")) {
										return cNode.getTreeLabelingIndex();
									} else {
										return -1;
									}
								}
							}
						} else if (modRole.equals("mod-all-2-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equalsIgnoreCase("both")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-universal-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cTag = cNode.getValues().get(0);
								if (cTag.equals("PDT")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-existential-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equalsIgnoreCase("some")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-distributive-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equalsIgnoreCase("each")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-paucal-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equalsIgnoreCase("few")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-multal-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equalsIgnoreCase("many")) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("mod-all-higher-quantifier")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equalsIgnoreCase("times")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-source")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equals("from")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-freq-time-of-day")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equals("everyday")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-distance")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equals("miles")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-circumstance")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equals("with")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole == "mod-descriptor"
								|| modRole == "mod-ordinal-high-number") {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								final String[] DESCRIPTORS =
										new String[] { "angry", "twentieth", "quickly" };

								if (ArrayUtils.unsortedArrayContains(DESCRIPTORS, cLex)) {
									return cNode.getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-at-month-of-year")
								|| modRole.equals("role-freq-month-of-year")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								final String[] MONTHS =
										new String[] { "January", "February", "March", "April",
												"May", "June", "July", "August", "September",
												"October", "November", "December" };

								if (ArrayUtils.unsortedArrayContains(MONTHS, cLex)) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole.equals("role-loc-proximate-past")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equals("close")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						} else if (modRole.startsWith("role-freq")) {
							for (TreeNode cNode : cNodes) {
								String cLex =
										cNode.getValues().size() > 1 ? cNode.getValues().get(1)
												: "";

								if (cLex.equals("every") || cLex.equals("everyday")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						}

						// // fall back to matching non-article determiners
						// for (TreeNode cNode : cNodes) {
						// String cValue = cNode.getValues().get(0);
						// if (cValue.equals("DT") && cNode.getValues().size() >
						// 1
						// && !cNode.getValues().get(1).equals("a")
						// && !cNode.getValues().get(1).equals("an")
						// && !cNode.getValues().get(1).equals("the")) {
						// return cNode.getIndex();
						// }
						// }
					} // end for modifier children

					// fallback to assuming this might be a verb modifier
					if (grandparent.getTreeLabelingIndex() == 1) {
						// we are modifying the verb, not another sentence
						// element

						// first try to match an ADVP
						for (TreeNode cNode : cNodes) {
							String cTag = cNode.getValues().get(0);
							if (cTag.equals("ADVP")) {
								return cNode.getTreeLabelingIndex();
							}
						}

						// fall back to a PP
						for (TreeNode cNode : cNodes) {
							String cTag = cNode.getValues().get(0);
							if (cTag.equals("PP")) {
								return cNode.getTreeLabelingIndex();
							}
						}
					}

				} else if (ArrayUtils.unsortedArrayContains(RELATIVE_CLAUSE, fParentValue)) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// match non-article determiners
						if (cValue.equals("SBAR")) {
							return cNode.getTreeLabelingIndex();
						}
					}

				} else if (fParentValue.equals("poss")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// match non-article determiners
						if (cValue.equals("POS") || cValue.equals("PRP$") || cValue.equals("WP$")) {
							return cNode.getTreeLabelingIndex();
						}
					}

				} else if (fParentValue.equals("goal")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// match non-article determiners
						if (cValue.equals("TO")) {
							return cNode.getParentNode().getTreeLabelingIndex();
						}
					}

				} else if (fParentValue.equals("pred-nom")) {

					for (TreeNode modChild : fNode.getChildren()) {
						if (!modChild.getValues().get(0).equals("mod-role"))
							continue;

						String modRole = modChild.getValues().get(1);

						if (modRole.equals("descriptive-modifier")) {
							for (TreeNode cNode : cNodes) {
								String cTag = cNode.getValues().get(0);
								if (cTag.equals("ADJP")) {
									return cNode.getParentNode().getTreeLabelingIndex();
								}
							}
						}
					}
				} // end massive fValue matching if-else

			} // end if parent == comp

		} else {
			// this is a root node, so match it to S1
			return 1;
		} // end if parent != null

		return -1;
	}

	private static boolean hasFStructureValue(TreeNode subtree, String value) {
		for (TreeNode fNode : subtree.getTerminalNodes()) {

			if (fNode.getValues().size() > 1) {
				String nodeValue = fNode.getValues().get(1);
				assert nodeValue == nodeValue.intern() : "nodeValue without intern()";
				if (nodeValue == value) {
					return true;
				}
			}
		}
		return false;
	}

	private static int findSubject(TreeNode grandparent, List<TreeNode> cNodes) {
		for (TreeNode cNode : cNodes) {
			String cValue = cNode.getValues().get(0);
			if (cValue.equals("NP") || cValue.equals("WHNP")) {
				return cNode.getTreeLabelingIndex();
			}
		}

		// if this is an imperative sentence, there may legitimately be no actor
		// in the c-structure
		if (grandparent != null && hasFStructureValue(grandparent, "minor-imperative")) {
			return PhiPlusMapping.NO_CONSTITUENT;
		} else {
			return -1;
		}
	}

	private static int findDirectObject(List<TreeNode> cNodes, boolean takeNextNPunderVP) {
		for (TreeNode cNode : cNodes) {
			String cValue = cNode.getValues().get(0);
			if (cValue.equals("VP") || cValue.equals("SQ")) {
				for (TreeNode cNode2 : cNode.getLabeledNodes()) {
					String cValue2 = cNode2.getValues().get(0);
					// ADJP is specifically for pred
					if (cValue2.equals("NP") || cValue2.equals("ADJP")) {
						if (takeNextNPunderVP) {
							return cNode2.getTreeLabelingIndex();
						} else {
							// we found the first, so take the next
							// one we see
							takeNextNPunderVP = true;
						}
					}
				}
			}
		}
		return -1;
	}

	private static boolean hasParentLabel(TreeNode subtree, String label) {
		for (TreeNode labeledNode : subtree.getLabeledNodes()) {
			TreeNode parent = labeledNode.getParentNode();
			if (parent != null) {
				ArrayList<String> values = parent.getValues();
				if (values.size() > 0) {
					if (values.get(0).equals(label)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	private static int findChild(TreeNode parent, String... values) {
		for (TreeNode child : parent.getChildren()) {
			for (String value : values) {
				if (child.getValues().get(0).startsWith(value)) {
					return child.getTreeLabelingIndex();
				}
			}
		}
		return -1;
	}

	public static int findHead(TreeNode cNode, int[] headMapping) {

		int result = -1;
		String value = cNode.getValues().get(0);

		if (value.equals("S1")) {
			result = findChild(cNode, "SBARQ", "S");
		} else if (value.equals("SQ") || value.equals("S")) {
			result = findChild(cNode, "VP", "S");

			// fallback to AUX
			if (result == -1)
				result = findChild(cNode, "AUX");

			// fallback again to SBARQ
			if (result == -1)
				result = findChild(cNode, "SBARQ");

		} else if (value.equals("SBARQ")) {
			result = findChild(cNode, "SQ", "S");
		} else if (value.equals("SBAR")) {
			result = findChild(cNode, "S", "WHNP");
		} else if (value.equals("NP")) {
			result = findChild(cNode, "NN", "PRP", "CD", "EX");

			// fallback to searching for another NP
			if (result == -1) {
				result = findChild(cNode, "NP");
			}

			// fallback to searching for a DT
			// covers cases such as (NP (DT These)) (VP (V ate))
			if (result == -1) {
				result = findChild(cNode, "DT");
			}

		} else if (value.equals("WHNP")) {

			result = findChild(cNode, "NN", "PRP", "CD", "WP", "S");

			// fallback to searching for another NP
			if (result == -1) {
				result = findChild(cNode, "NP");
			}

		} else if (value.equals("VP")) {

			for (TreeNode child : cNode.getChildren()) {
				for (String childValue : new String[] { "VB", "VP", "S" }) {
					if (child.getValues().get(0).startsWith(childValue)) {
						
						// except in the "used to" construction
						if (child.getValues().get(0).startsWith("VB")
								&& child.getValues().size() > 1
								&& child.getValues().get(1).equals("used")) {
							
							// the "to" in "used to" will always be under a S
							// node
							result = findChild(cNode, "S");
						} else {
							result = child.getTreeLabelingIndex();
						}
					}
				}
			}

			// fallback to self
			if (result == -1) {
				result = findChild(cNode, "AUX");
			}

		} else if (value.equals("PP")) {

			result = findChild(cNode, "NP");

			// fall back to looking for another PP (e.g. "out of")
			if (result == -1) {
				result = findChild(cNode, "PP");
			}

		} else if (value.equals("ADJP")) {
			result = findChild(cNode, "JJ");

		} else if (value.equals("ADVP")) {
			result = findChild(cNode, "RB");

		} else if (value.equals("WHADVP")) {
			result = findChild(cNode, "WRB");

		} else if (value.startsWith("JJ") || value.startsWith("NN") || value.startsWith("VB")
				|| value.startsWith("RB") || value.startsWith("WHP") || value.startsWith("WP")
				|| value.startsWith("PRP") || value.startsWith("EX") || value.startsWith("CD")
				|| value.startsWith("WRB")) {

			// this node IS a head
			result = cNode.getTreeLabelingIndex();

		} else if (value.equals("DT") || value.equals("PDT") || value.equals("POS")
				|| value.equals("UH") || value.equals("WDT") || value.equals("AUX")
				|| value.equals("AUXG") || value.equals("MD") || value.equals("IN")
				|| value.equals("TO") || value.equals("PRT") || value.equals("PRP$")
				|| value.equals("RP") || value.equals(".") || value.equals(",")
				|| value.equals("''") || value.equals("``")) {

			// TODO: Instead of having return statements scattered anywhere,
			// move return to the bottom and before returning a value, make sure
			// we won't be creating a loop! -- if so, choose the bottom value?

			// this node is a terminal, but not a head, so look up a level
			// ...unless our parent thinks we're the head (e.g. the case of an
			// AUX actually being a copula
			result = cNode.getParentNode().getTreeLabelingIndex();

		}

		if (result != -1) {
			if (headMapping[result - 1] + 1 == cNode.getTreeLabelingIndex()) {
				// if parent thinks we're a head, go with it so that we don't
				// create a cycle
				result = cNode.getTreeLabelingIndex();
				cycles++;
			}
		}

		heads++;
		return result;
	}
	static int cycles = 0;
	static int heads = 0;
}
