package edu.cmu.cs.lti.avenue.navigation.search.generation2.heuristics.expressions;

import java.util.List;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureContext;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureValueSpec;
import edu.cmu.cs.lti.avenue.morphology.SegmenterException;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraph;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraphManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureValueCluster;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.simulation.SimulatedNode;
import edu.cmu.cs.lti.avenue.navigation.search.generation2.heuristics.HeuristicException;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

public class HeuristicExpression {

	private SmartTree lhs;
	private SmartTree rhs;
	private double weight;

	private static final double TRUE = Double.POSITIVE_INFINITY;
	private static final double FALSE = Double.NEGATIVE_INFINITY;

	public HeuristicExpression(SmartTree lhs, SmartTree rhs, double weight) {
		this.lhs = lhs;
		this.rhs = rhs;
		this.weight = weight;

		lhs.sort();
		rhs.sort();
	}

	public SmartTree getLhs() {
		return lhs;
	}

	public SmartTree getRhs() {
		return rhs;
	}

	public boolean isLhsTriggered(FeatureExpressionGraphManager currentState)
			throws SegmenterException, CorpusException {

		return evaluateFullExpressionLhs(currentState, lhs);
	}

	public double scoreUsingRhs(SmartTree featureStructure, List<SimulatedNode> simulatedNodes)
			throws HeuristicException {

		featureStructure.sort();
		return compareTrees(featureStructure.getRootNode(), rhs.getRootNode(), simulatedNodes);
	}

	private double compareTrees(TreeNode input, TreeNode pattern,
			List<SimulatedNode> simulatedNodes) throws HeuristicException {

		double score = 0.0;
		
		// recursively match correct node on the RHS
		double temp = compareValues(input, pattern, simulatedNodes);
		if (temp == FALSE) {
			return FALSE;
		} else if(temp == TRUE) {
			// noop
		} else {
			score += temp;
		}

		// we matched the values, now let's check the children
		List<TreeNode> inputChildren = input.getChildren();
		List<TreeNode> patternChildren = pattern.getChildren();

		// TODO: Optimize this by taking advantage of the fact that the feature
		// structures are sorted (see TreeGrepper for an example of how to do
		// this)
		for (TreeNode patternChild : patternChildren) {
			boolean found = false;
			for (TreeNode inputChild : inputChildren) {
				
				temp = compareTrees(inputChild, patternChild, simulatedNodes);
				if (temp == FALSE) {
					// noop
				} else if(temp == TRUE) {
					found = true;
					break;
				} else {
					score += temp;
					found = true;
					break;
				}
			}
			if (found == false) {
				return FALSE;
			}
		}

		return score;
	}

	/**
	 * The core method for determining scores for MILES Implicational Universal
	 * Heuristic scores.
	 * 
	 * @param input
	 * @param pattern
	 * @param simulatedNodes
	 * @return
	 * @throws HeuristicException
	 */
	private double compareValues(TreeNode input, TreeNode pattern,
			List<SimulatedNode> simulatedNodes) throws HeuristicException {

		List<String> inputValues = input.getValues();
		List<String> patternValues = pattern.getValues();

		if (patternValues.size() == 2) {

			// match context in feature structure
			if (inputValues.equals(patternValues) == true) {
				return TRUE;
			} else {
				return FALSE;
			}

		} else if (patternValues.size() == 4) {

			// match context in feature structure AND make sure it is part of a
			// minimal pair

			String patternFeatureName = patternValues.get(0);
			String patternFeatureValue = patternValues.get(1);
			String patternOperator = patternValues.get(2);
			String patternState = patternValues.get(3);

			// match context in feature structure
			if (inputValues.get(0).equals(patternFeatureName) == false) {
				return FALSE;
			}
			if (patternFeatureValue.equals("*") == false
					&& inputValues.get(1).equals(patternFeatureValue) == false) {
				return FALSE;
			}

			double polarity;
			if (patternOperator.equals("=")) {
				polarity = 1;
			} else if (patternOperator.equals("!=")) {
				// reverse the polarity of the weight for unexpressed
				// features
				polarity = -1;
			} else {
				throw new HeuristicException("Invalid pattern node (unknown operator): " + pattern);
			}

			if (patternState.equals("EXPRESSED") == false) {
				throw new HeuristicException("Invalid pattern node (state should be EXPRESSED): "
						+ pattern);
			}

			double score = 0.0;
			// now make sure this feature value will participate in a minimal
			// pair
			for (SimulatedNode simulatedNode : simulatedNodes) {
				String name = simulatedNode.getFeatureName();
				String value = simulatedNode.getFeatureValue();
				int evidenceCount = simulatedNode.getEvidenceCount();

				if (name.equals(patternFeatureName)) {
					if (patternFeatureValue.equals("*") || value.equals(patternFeatureValue)) {
						score += polarity * evidenceCount * weight;
					}
				}
			}

			return score;
		} else {
			throw new HeuristicException("Invalid pattern node: " + pattern);
		}
	}

	private boolean evaluateFullExpressionLhs(FeatureExpressionGraphManager currentState,
			SmartTree pattern) throws SegmenterException, CorpusException {

		assert pattern.getRootNode().getChildren().size() == 1 : "Pattern node must have 1 child node (an expression): "
				+ pattern.toString();

		String featureContext;
		TreeNode expressionNode;

		if (currentState.shouldConsiderContext()) {
			featureContext = pattern.getRootNode().getValues().get(0);
		} else {
			featureContext = FeatureContext.ANY.name;
		}

		// since pattern has only one child, we know that has to be the
		// correct child
		expressionNode = pattern.getRootNode().getChildren().get(0);
		return evaluateExpressionNodeLhs(currentState, featureContext, expressionNode);
	}

	private boolean evaluateExpressionNodeLhs(FeatureExpressionGraphManager currentState,
			String featureContext, TreeNode expressionNode) throws SegmenterException,
			CorpusException {

		List<String> expressionValues = expressionNode.getValues();

		assert expressionValues.size() == 4 : "Expected 4 expressionValues, but got "
				+ expressionValues.size() + " in " + expressionNode.toString();

		String expressionFeatureName = expressionValues.get(0);
		String expressionFeatureValue = expressionValues.get(1);
		String expressionOperator = expressionValues.get(2);
		String expressionState = expressionValues.get(3);

		// the feature names matched, now check values
		// values can be wildcard values

		if (expressionFeatureValue.equals("ALL")) {

			FeatureExpressionGraph feg =
					currentState.getFeatureExpressionGraphFor(featureContext, expressionFeatureName);

			boolean result = true;

			// check if ALL values of this feature are/aren't joint/unique
			if (expressionOperator.equals("==") && expressionState.equals("EXPRESSED")) {

				for (FeatureValueCluster cluster : feg.getAllValueClusters()) {

					assert cluster.getFeatureInteractions().size() != 0 : "Zero feature interaction size";

					if (cluster.isUnobserved() || cluster.getFeatureInteractions().size() == 1) {
						result = false;
						break;
					}
				}
			} else if (expressionOperator.equals("!=") && expressionState.equals("EXPRESSED")) {

				for (FeatureValueCluster cluster : feg.getAllValueClusters()) {

					assert cluster.getFeatureInteractions().size() != 0 : "Zero feature interaction size";

					if (cluster.isUnobserved() || cluster.getFeatureInteractions().size() > 1) {
						result = false;
						break;
					}
				}
			} else {
				throw new RuntimeException("Unknown operator or state: " + expressionOperator + " "
						+ expressionState);
			}

			return result;

		} else if (expressionFeatureValue.equals("ANY")) {

			FeatureExpressionGraph feg =
					currentState.getFeatureExpressionGraphFor(featureContext, expressionFeatureName);

			boolean result = false;

			// check if this inputFeatureValue is/isn't joint/unique
			if (expressionOperator.equals("!=") && expressionState.equals("EXPRESSED")) {

				for (FeatureValueCluster cluster : feg.getAllValueClusters()) {

					assert cluster.getFeatureInteractions().size() != 0 : "Zero feature interaction size";

					if (cluster.getFeatureInteractions().size() > 1) {
						result = true;
						break;
					}
				}
			} else if (expressionOperator.equals("==") && expressionState.equals("EXPRESSED")) {
				for (FeatureValueCluster cluster : feg.getAllValueClusters()) {

					assert cluster.getFeatureInteractions().size() != 0 : "Zero feature interaction size";

					if (cluster.getFeatureInteractions().size() == 1) {
						result = true;
						break;
					}
				}

			} else {
				throw new RuntimeException("Unknown operator or state: " + expressionOperator + " "
						+ expressionState);
			}

			return result;

		} else {

			FeatureExpressionGraph feg =
					currentState.getFeatureExpressionGraphFor(featureContext, expressionFeatureName);

			boolean result = true;

			// check if ALL values of this feature are/aren't joint/unique
			if (expressionOperator.equals("==") && expressionState.equals("EXPRESSED")) {

				for (FeatureValueCluster cluster : feg.getAllValueClusters()) {

					assert cluster.getFeatureInteractions().size() != 0 : "Zero feature interaction size";

					boolean matchedName = matchesFeatureName(expressionFeatureValue, cluster);
					if (matchedName) {
						if (cluster.getFeatureInteractions().size() == 1) {
							result = false;
							break;
						}
					}
				}

			} else if (expressionOperator.equals("!=") && expressionState.equals("EXPRESSED")) {

				for (FeatureValueCluster cluster : feg.getAllValueClusters()) {

					assert cluster.getFeatureInteractions().size() != 0 : "Zero feature interaction size";

					boolean matchedName = matchesFeatureName(expressionFeatureValue, cluster);
					if (matchedName) {
						if (cluster.getFeatureInteractions().size() > 1) {
							result = false;
							break;
						}
					}
				}
			} else {
				throw new RuntimeException("Unknown operator or state: " + expressionOperator + " "
						+ expressionState);
			}

			return result;

		}
	}

	private boolean matchesFeatureName(String expressionFeatureValue, FeatureValueCluster cluster) {
		boolean matchedName = false;
		for (FeatureValueSpec spec : cluster.getFeatureValues()) {
			if (spec.getName().equals(expressionFeatureValue)) {
				matchedName = true;
				break;
			}
		}
		return matchedName;
	}
}
