package edu.cmu.cs.lti.avenue.navigation.search.generation2.heuristics;

import info.jonclark.lang.Pair;
import info.jonclark.log.LogUtils;
import info.jonclark.properties.PropertiesException;
import info.jonclark.properties.SmartProperties;
import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Properties;
import java.util.logging.Logger;

import edu.cmu.cs.lti.avenue.morphology.SegmenterException;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraphManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.simulation.SimulatedNode;
import edu.cmu.cs.lti.avenue.navigation.search.generation2.Hypothesis;

public class ExpressionNavigationHeuristic implements NavigationHeuristic {

	private static final Logger log = LogUtils.getLogger();
	
	private final HashMap<Pair<String, String>, Integer> heuristicData =
			new HashMap<Pair<String, String>, Integer>();

	public ExpressionNavigationHeuristic(Properties props, String propertiesNamespace)
			throws FileNotFoundException, IOException, PropertiesException, HeuristicException,
			ParseException {

		SmartProperties smartProps = new SmartProperties(props);

		// make sure we have a "." on the end of the namespace
		propertiesNamespace = StringUtils.forceSuffix(propertiesNamespace, ".");

		File heuristicDataFile = smartProps.getPropertyFile(propertiesNamespace + "dataFile");
		
		log.info("Reading expression heuristic data file: " + heuristicDataFile);

		BufferedReader in = new BufferedReader(new FileReader(heuristicDataFile));
		boolean inExpressions = false;
		String line;
		while ((line = in.readLine()) != null) {
			line = line.trim();

			if (line.equals("")) {
				continue;
			}

			// see if we're in the right section of the file
			else if (line.startsWith("<")) {
				if (line.equals("<px>")) {
					inExpressions = true;
				} else {
					inExpressions = false;
				}
				continue;
			}

			else if (inExpressions) {
				if (line.startsWith("(*")) {
					String fv = StringUtils.substringBetween(line, "(* (", "))");
					String feature = StringUtils.substringBefore(fv, " ");
					String value = StringUtils.substringAfter(fv, " ");
					String strFreq = StringUtils.substringAfter(line, " = ");
					int freq = parseFrequency(strFreq);

					heuristicData.put(new Pair<String, String>(feature, value), freq);
				} else {
					throw new HeuristicException("Unknown line format: " + line);
				}
			}

			// TODO: Parse each line (* (feature value)) = frequency
			// into some sort of useful data structure

		}
		in.close();
		
		log.info("Read " + heuristicData.size() + " entries.");
	}

	private static int parseFrequency(String str) throws ParseException {

		if (str.equals("always")) {
			return 3;
		} else if (str.equals("frequently")) {
			return 2;
		} else if (str.equals("occasionally")) {
			return 1;
		} else {
			throw new ParseException("Unknown frequency: " + str, 0);
		}
	}

	public double combineHypothesisScores(double... hyps) {
		double sum = 0.0;
		for (double hyp : hyps)
			sum += hyp;
		return sum;
	}

	public double scoreHypothesis(Hypothesis hyp) throws HeuristicException {

		double score = 0.0;

		// 1) Extract the feature value for each min pair
		for (SimulatedNode simulatedNode : hyp.simulationResult.getCreatedNodes()) {

			String name = simulatedNode.getFeatureName();
			
			// remove context of "*Xfeature-name"
			name = StringUtils.substringAfter(name, "X");
			
			String value = simulatedNode.getFeatureValue();
			int evidenceCount = simulatedNode.getEvidenceCount();

			Pair<String, String> key = new Pair<String, String>(name, value);
			Integer freq = heuristicData.get(key);
			if (freq == null) {
				score += 0.0;
			} else {
				score += evidenceCount * freq;
			}
		}

		return score;
	}

	public void updateCurrentState(FeatureExpressionGraphManager fegMan) throws SegmenterException {

	}
}
