package edu.cmu.cs.lti.avenue.navigation.search.generation2;

import info.jonclark.lang.Pair;
import info.jonclark.log.LogUtils;
import info.jonclark.properties.PropertiesException;
import info.jonclark.properties.PropertyUtils;
import info.jonclark.properties.SmartProperties;
import info.jonclark.stat.TaskListener;
import info.jonclark.stat.TextProgressBar;
import info.jonclark.util.FileUtils;
import info.jonclark.util.FormatUtils;
import info.jonclark.util.StringUtils;
import info.jonclark.util.TransformException;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Properties;
import java.util.Queue;
import java.util.Random;
import java.util.logging.Logger;

import javax.xml.parsers.ParserConfigurationException;

import org.xml.sax.SAXException;

import edu.cmu.cs.lti.avenue.corpus.Corpus;
import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.corpus.Serializer;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager;
import edu.cmu.cs.lti.avenue.morphology.Segmenter;
import edu.cmu.cs.lti.avenue.morphology.SegmenterException;
import edu.cmu.cs.lti.avenue.navigation.elicitation.ElicitationException;
import edu.cmu.cs.lti.avenue.navigation.elicitation.ElicitationInterface;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.ExpressedThemeManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraph;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraphManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.InductiveResultsEvaluator;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.LexicalClusterManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.MinimalPairManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.MinimalPairMapping;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.ObservableSentencePair;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.PlateauFunction;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.RulelessFeatureDetector;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.LexicalCluster.LexClusterScores;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.FeatureMarking;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.reports.CsvFeatureDetectionReport;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.reports.FeatureDetectionReport;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.simulation.SimulationResult;
import edu.cmu.cs.lti.avenue.navigation.search.generation1.tables.ConfigurationException;
import edu.cmu.cs.lti.avenue.navigation.search.generation2.heuristics.HeuristicException;
import edu.cmu.cs.lti.avenue.navigation.search.generation2.heuristics.NavigationHeuristic;

// TODO: Unit tests
public class CorpusNavigator {

	private static final Logger log = LogUtils.getLogger();

	private final static int nInteractions = 1;

	private final Queue<ObservableSentencePair> openingBook =
			new LinkedList<ObservableSentencePair>();
	private final ArrayList<ObservableSentencePair> candidates =
			new ArrayList<ObservableSentencePair>();
	private final Queue<ObservableSentencePair> translatedOnly =
			new LinkedList<ObservableSentencePair>();
	private final ArrayList<ObservableSentencePair> alignedTranslated =
			new ArrayList<ObservableSentencePair>();

	private final Corpus elicitationCorpus;

	// track which sentences we already have alignment for
	// in case we want to hypothesize these alignments and
	// just ask the user to confirm them later instead
	private final HashMap<String, ObservableSentencePair> alignmentHash =
			new HashMap<String, ObservableSentencePair>();

	private final ElicitationInterface elicitor;
	private final ArrayList<NavigationListener> navigationListeners =
			new ArrayList<NavigationListener>();
	private double motivationToContinue = 1.0;

	private final NavigationHeuristic heuristic;
	private final MinimalPairManager minPairMan;
	private final ExpressedThemeManager themeMan;
	private final Random rand = new Random();
	private final boolean randomMode;

	// fegMan contains "current state" of navigation
	// and its simulation mode can be used to help determine
	// what the "next state" should be
	private final FeatureExpressionGraphManager fegMan;

	private int nTargetSentenceCount;
	private int beamSize;
	private int nLookahead;

	private final HashSet<String> includedFeatures;

	private final HashSet<String> excludedFeatures;

	/**
	 * The order of the sentences in the openingBook corpus will determine the
	 * order in which we elicit them here.
	 * 
	 * @param elicit
	 * @param nTargetSentenceCount
	 * @param beamSize
	 * @param nBestSize
	 * @param nSequenceWindow
	 *            The number of possible elements (sentences) in a sentence
	 *            sequence that will be used when scoring each possible path
	 *            through the search space of sentence sequences
	 * @throws PropertiesException
	 */
	public CorpusNavigator(Properties props, String namespace) throws IOException, SAXException,
			ParserConfigurationException, ConfigurationException, ParseException,
			HeuristicException, ElicitationException, SegmenterException, CorpusException,
			PropertiesException {

		namespace = StringUtils.forceSuffix(namespace, ".");
		SmartProperties smartProps = new SmartProperties(props);

		// use reflection to instantiate heuristic
		this.heuristic = loadHeuristic(props, namespace);
		this.elicitor = loadElicitor(props, namespace);

		this.nTargetSentenceCount = smartProps.getPropertyInt(namespace + "targetSentenceCount");
		this.beamSize = smartProps.getPropertyInt(namespace + "beamSize");
		this.nLookahead = smartProps.getPropertyInt(namespace + "lookaheadWindowSize");

		String encoding = smartProps.getPropertyString("global.encoding");
		File elicitationCorpusFile =
				smartProps.getPropertyFile(namespace + "elicitationCorpusFile");
		this.elicitationCorpus =
				Serializer.loadSentencePairs(elicitationCorpusFile, true, encoding);

		File openingBookFile = smartProps.getPropertyFile(namespace + "openingBookCorpusFile");
		Corpus openingBook = Serializer.loadSentencePairs(openingBookFile, true, encoding);

		FeatureStructureManager fsMan =
				new FeatureStructureManager(smartProps.getPropertyFile("paths.featureSpec"));

		String[] featureWhitelist = smartProps.getPropertyStringArray("inference.featureWhitelist");
		String[] featureBlacklist = smartProps.getPropertyStringArray("inference.featureBlacklist");
		excludedFeatures = new HashSet<String>(Arrays.asList(featureBlacklist));
		includedFeatures = new HashSet<String>(Arrays.asList(featureWhitelist));

		HashSet<String> stopwords = RulelessFeatureDetector.initStopwords(smartProps);

		double inferenceExpressionThreshold =
				smartProps.getPropertyFloat("inference.expressionThreshold");
		boolean considerFeatureContext =
				smartProps.getPropertyBoolean("inference.considerFeatureContext");

		Segmenter segmenter = RulelessFeatureDetector.loadSegmenter(props);

		boolean useExactMinimalPairs =
				smartProps.getPropertyBoolean("inference.useExactMinimalPairs");

		LexClusterScores lexicalThresholds =
				RulelessFeatureDetector.readLexicalThresholds(smartProps);

		int noPenaltyDistance = smartProps.getPropertyInt("inference.noPenaltyDistance");
		int somePenaltyDistance = smartProps.getPropertyInt("inference.somePenaltyDistance");

		LexicalClusterManager lexMan = new LexicalClusterManager(lexicalThresholds, stopwords);
		PlateauFunction distancePenalty =
				new PlateauFunction(noPenaltyDistance, somePenaltyDistance);

		this.fegMan =
				new FeatureExpressionGraphManager(inferenceExpressionThreshold, fsMan, segmenter,
						distancePenalty, lexMan, includedFeatures, excludedFeatures, nInteractions,
						considerFeatureContext);

		this.minPairMan =
				new MinimalPairManager(includedFeatures, excludedFeatures, fsMan,
						useExactMinimalPairs);

		precomputeMinimalPairs(openingBook.getSentences(), this.openingBook);
		precomputeMinimalPairs(elicitationCorpus.getSentences(), this.candidates);

		// theme man
		double themeExpressionThreshold = smartProps.getPropertyFloat("themes.expressionThreshold");
		boolean minPairsWithMoreChangesAreLessReliable =
				smartProps.getPropertyBoolean("inference.minPairsWithMoreChangesAreLessReliable");
		boolean scoreThemesWithInvertedVocabulary =
				smartProps.getPropertyBoolean("inference.scoreThemesWithInvertedVocabulary");

		boolean normalizeWeightedFrequencies =
				smartProps.getPropertyBoolean("inference.normalizeWeightedFrequencies");

		HashMap<String, Integer> corpusCounts =
				RulelessFeatureDetector.analyzeFrequencies(elicitationCorpus);
		HashMap<String, ArrayList<SentencePair>> invertedCorpusVocabulary =
				RulelessFeatureDetector.invertCorpusVocabulary(elicitationCorpus);

		InductiveResultsEvaluator gold = new InductiveResultsEvaluator(new SmartProperties(props));
		this.themeMan =
				new ExpressedThemeManager(corpusCounts, invertedCorpusVocabulary,
						themeExpressionThreshold, minPairsWithMoreChangesAreLessReliable,
						scoreThemesWithInvertedVocabulary, normalizeWeightedFrequencies, gold);

		boolean allowMarkingsOnMe =
				smartProps.getPropertyBoolean("inference.markings.allowMarkingsOnMe");
		boolean allowMarkingsOnMyDependends =
				smartProps.getPropertyBoolean("inference.markings.allowMarkingsOnMyDependends");
		boolean allowMarkingsOnMyGovernor =
				smartProps.getPropertyBoolean("inference.markings.allowMarkingsOnMyGovernor");
		boolean allowMarkingsOnOthers =
				smartProps.getPropertyBoolean("inference.markings.allowMarkingsOnOthers");
		FeatureMarking.init(allowMarkingsOnMe, allowMarkingsOnMyDependends,
				allowMarkingsOnMyGovernor, allowMarkingsOnOthers);

		this.randomMode = smartProps.getPropertyBoolean("navigation.randomMode");
	}

	private static NavigationHeuristic loadHeuristic(Properties props, String namespace)
			throws HeuristicException {

		SmartProperties smartProps = new SmartProperties(props);
		String heuristicClass = smartProps.getPropertyString(namespace + "heuristic");
		try {
			return (NavigationHeuristic) Class.forName(heuristicClass).getConstructor(
					Properties.class, String.class).newInstance(props, namespace + "heuristic");
		} catch (IllegalArgumentException e) {
			throw new HeuristicException("Error loading heuristic: " + heuristicClass, e);
		} catch (SecurityException e) {
			throw new HeuristicException("Error loading heuristic: " + heuristicClass, e);
		} catch (InstantiationException e) {
			throw new HeuristicException("Error loading heuristic: " + heuristicClass, e);
		} catch (IllegalAccessException e) {
			throw new HeuristicException("Error loading heuristic: " + heuristicClass, e);
		} catch (InvocationTargetException e) {
			throw new HeuristicException("Error loading heuristic: " + heuristicClass, e);
		} catch (NoSuchMethodException e) {
			throw new HeuristicException(
					"Heuristic class "
							+ heuristicClass
							+ " must define constructor parameterized by (Properties props, String namespace)");
		} catch (ClassNotFoundException e) {
			throw new HeuristicException("Heuristic class not found: " + heuristicClass);
		}
	}

	private static ElicitationInterface loadElicitor(Properties props, String namespace)
			throws ElicitationException {

		SmartProperties smartProps = new SmartProperties(props);
		String elicitorClass = smartProps.getPropertyString(namespace + "elicitor");
		try {
			return (ElicitationInterface) Class.forName(elicitorClass).getConstructor(
					Properties.class, String.class).newInstance(props, namespace + "elicitor");
		} catch (IllegalArgumentException e) {
			throw new ElicitationException("Error loading elicitor: " + elicitorClass, e);
		} catch (SecurityException e) {
			throw new ElicitationException("Error loading elicitor: " + elicitorClass, e);
		} catch (InstantiationException e) {
			throw new ElicitationException("Error loading elicitor: " + elicitorClass, e);
		} catch (IllegalAccessException e) {
			throw new ElicitationException("Error loading elicitor: " + elicitorClass, e);
		} catch (InvocationTargetException e) {
			throw new ElicitationException("Error loading elicitor: " + elicitorClass, e);
		} catch (NoSuchMethodException e) {
			throw new ElicitationException(
					"Elicitor class "
							+ elicitorClass
							+ " must define constructor parameterized by (Properties props, String namespace)");
		} catch (ClassNotFoundException e) {
			throw new ElicitationException("Elicitator class not found: " + elicitorClass);
		}
	}

	private void precomputeMinimalPairs(Collection<SentencePair> sents,
			Collection<ObservableSentencePair> result) throws CorpusException {

		for (SentencePair sent : sents) {
			ArrayList<MinimalPairMapping> mappings =
					minPairMan.mapToMinimalPairs(sent.getFeatureStructure(), nInteractions, true);
			ObservableSentencePair os = new ObservableSentencePair(sent, mappings);
			result.add(os);
		}

	}

	protected void fireNavigationStatusChanged(String status) {
		for (NavigationListener l : navigationListeners) {
			l.statusChanged(status);
		}
	}

	public void addNavigationListener(NavigationListener l) {
		navigationListeners.add(l);
	}

	private static String makePrimaryKey(ObservableSentencePair os) {
		SentencePair pair = os.getSentencePair();
		return StringUtils.untokenize(pair.getNormalizedSourceTokens()) + " ||| "
				+ StringUtils.untokenize(pair.getNormalizedTargetTokens());
	}

	/**
	 * Called when a new sentence has been both translated and aligned, so that
	 * we can update our current state of knowledge about this language.
	 * 
	 * @throws CorpusException
	 * @throws SegmenterException
	 */
	public void updateCurrentState(ObservableSentencePair osAligned) throws CorpusException,
			SegmenterException {
		fegMan.observe(osAligned.getSentencePair(), osAligned.getMinPairMappings());
		heuristic.updateCurrentState(fegMan);
	}

	/**
	 * Returns a weight indicating how "motiviated" the system is to continue
	 * asking our bilingual person about more translations or alignments.
	 * 
	 * @return
	 */
	public double getMotivationToContinue() {

		if (candidates.size() == 0 || alignedTranslated.size() >= nTargetSentenceCount) {

			motivationToContinue = 0.0;
		}
		return motivationToContinue;
	}

	public boolean wantsToContinue() {
		return (getMotivationToContinue() > 0.0);
	}

	public boolean isOpeningBookExhausted() {
		return openingBook.isEmpty();
	}

	/**
	 * Returns true if we acquired a new aligned+translated setence
	 * 
	 * @return
	 * @throws ParseException
	 * @throws CorpusException
	 * @throws SegmenterException
	 * @throws HeuristicException
	 */
	public boolean askNext() throws ParseException, CorpusException, SegmenterException,
			HeuristicException {

		boolean gotNewAlignedTranslated = false;

		// make sure we have all sentenced aligned
		// if not, get them aligned first
		if (translatedOnly.size() > 0) {
			ObservableSentencePair unaligned = translatedOnly.poll();

			// see if we already have alignments for the sentence
			// by checking for sentences that have the same source
			// and target strings
			//
			// TODO: Optimize this hideously slow method of creating a primary
			// key via string concatenation (slow in java)
			ObservableSentencePair previouslyAligned = alignmentHash.get(makePrimaryKey(unaligned));

			ObservableSentencePair osAligned;
			if (previouslyAligned != null) {

				// hypothesize our old alignment
				fireNavigationStatusChanged("Requesting alignment confirmation for "
						+ unaligned.getSentencePair().getDisplaySourceSentence());
				unaligned.getSentencePair().setAlignments(
						previouslyAligned.getSentencePair().getDisplayAlignment());

				SentencePair aligned = elicitor.confirmAlignment(unaligned.getSentencePair());
				osAligned = new ObservableSentencePair(aligned, unaligned.getMinPairMappings());
				alignedTranslated.add(osAligned);
				alignedTranslated.add(new ObservableSentencePair(aligned,
						unaligned.getMinPairMappings()));
				gotNewAlignedTranslated = true;

			} else {

				// ask the user for a new alignment
				fireNavigationStatusChanged("Requesting new alignment for "
						+ unaligned.getSentencePair().getDisplaySourceSentence());

				SentencePair aligned = elicitor.elicitAlignment(unaligned.getSentencePair());
				osAligned = new ObservableSentencePair(aligned, unaligned.getMinPairMappings());
				alignedTranslated.add(osAligned);
				alignmentHash.put(makePrimaryKey(osAligned), osAligned);
				gotNewAlignedTranslated = true;
			}

			// update our current state with our new aligned & translated
			// sentence
			fireNavigationStatusChanged("Updating current state...");
			updateCurrentState(osAligned);

		} else {

			// if we already have alignments for all translated sentences,
			// decide which feature structure we want to elicit

			ObservableSentencePair nextSentence;
			if (randomMode) {
				nextSentence = candidates.get(rand.nextInt(candidates.size()));

				fireNavigationStatusChanged("Requesting translation for random sentence: "
						+ nextSentence.getSentencePair().getDisplaySourceSentence());
			} else {

				if (openingBook.size() > 0) {

					// if we haven't completed the opening book yet,
					// do that first
					nextSentence = openingBook.poll();

					fireNavigationStatusChanged("Requesting translation for next sentence in Opening Book ("
							+ openingBook.size()
							+ " remaining): "
							+ nextSentence.getSentencePair().getDisplaySourceSentence());
				} else {

					// if we've completed the opening book,
					// do a search for the next best sentence
					Beam<HypothesisSequence> nBest = getNBestList();
					nextSentence = nBest.getBest().getFirstHypothesis().sent;

					System.out.println(nBest.currentSize() + " hypothesis sequences found.");
					for (Pair<Double, HypothesisSequence> entry : nBest.getAll()) {
						double score = entry.first;
						HypothesisSequence seq = entry.second;
						System.out.println("[" + score + "] " + seq.toString());
					}

					// mark this sentence as elicited
					candidates.remove(nextSentence);

					fireNavigationStatusChanged("Requesting translation for top best sentence from search: "
							+ nextSentence.getSentencePair().getDisplaySourceSentence());
				}
			} // end if randomMode

			SentencePair translation = elicitor.elicitTranslation(nextSentence.getSentencePair());
			ObservableSentencePair osTranslated =
					new ObservableSentencePair(translation, nextSentence.getMinPairMappings());
			translatedOnly.add(osTranslated);
		}

		return gotNewAlignedTranslated;
	}

	public Corpus getNavigatedCorpus() {

		Corpus corpus = new Corpus();
		corpus.encoding = this.elicitationCorpus.encoding;
		corpus.srcLang = this.elicitationCorpus.srcLang;
		corpus.tgtLang = this.elicitationCorpus.tgtLang;

		for (ObservableSentencePair s : alignedTranslated) {
			corpus.addSentence(s.getSentencePair());
		}

		return corpus;
	}

	protected Beam<HypothesisSequence> getNBestList() throws CorpusException, SegmenterException,
			HeuristicException {

		// 1) generate a list of untranslated sentences
		// 2) if requested (if this is the first sentence in a lookahead
		// sequence), add candidate sentences to the list
		// 3) annotate each candidate sentence with:
		// a) the minimal pairs that can be formed from a
		// translated sentence and that candidate sentence
		// b) the different feature values that corresponds to that minimal pair
		// 4) rank the list by ranking the varied feature values of each minimal
		// pair
		// 5) prune the list down to the beam size
		// 6) if we haven't completed our full lookahead yet, move to the next
		// lookahead state and repeat to step 1

		// use 2 beams: 1 for sentences that will immediately produce minimal
		// pairs
		// and 1 for sentences that will produce minimal pairs after a few more
		// iterations

		// perform the multi-beam search

		// TODO: Keep each intermediate stage of the beam search?
		Beam<HypothesisSequence> contiguousBeam = new Beam<HypothesisSequence>(beamSize);
		Beam<HypothesisSequence> disjointBeam = new Beam<HypothesisSequence>(beamSize);

		System.out.println("Starting search...");
		for (int i = 1; i <= nLookahead; i++) {

			// At this point, only already-elicited sentences are in the FEG
			// data, so we know that simulation results won't be compared with
			// any unelicited sentences.

			if (i == 1) {
				fireNavigationStatusChanged("Creating hypotheses for i=1...");
				createHypotheses(contiguousBeam, disjointBeam);
			} else {
				fireNavigationStatusChanged("Extending hypotheses for i=" + i + "...");
				contiguousBeam = extendHypotheses(contiguousBeam);
				disjointBeam = extendHypotheses(disjointBeam);
			}
		}

		// now combine the results of the 2 beams
		for (Pair<Double, HypothesisSequence> pair : disjointBeam.getAll()) {
			contiguousBeam.add(pair.first, pair.second);
		}

		return contiguousBeam;
	}

	/**
	 * Called by multiBeamSearch when the depth is 1. It creates an empty beam
	 * with hypothesis sequences of length 1.
	 * 
	 * @param prevBeam
	 * @param contiguousMinPairMan
	 * @param depth
	 * @param disjointMinPairs
	 * @return
	 * @throws CorpusException
	 * @throws HeuristicException
	 * @throws SegmenterException
	 */
	private void createHypotheses(Beam<HypothesisSequence> contiguousBeam,
			Beam<HypothesisSequence> disjointBeam) throws CorpusException, HeuristicException,
			SegmenterException {

		// create minimal pairs and rank each candidate sentence (hypothesis)
		// no already-elicited sentences can be in the candidate list
		for (ObservableSentencePair candidate : candidates) {

			// simulate observing the only sentence that will be in this
			// hypothesis sequence so far
			fegMan.beginSimulation();
			fegMan.simulateObserving(candidate.getSentencePair(), candidate.getMinPairMappings());
			SimulationResult simulationResult = fegMan.getSimulationResult();
			fegMan.endSimulation();

			Hypothesis hyp = new Hypothesis();
			hyp.sent = candidate;
			hyp.simulationResult = simulationResult;

			// if none of the translated sentences created any new arc or node
			// evidence during simulation, then we know that this hypothesis is
			// disjoint with the rest of the search space and so we should keep
			// it in a separate beam if we ever have any hope of exploring
			// non-contiguous parts of the search space
			hyp.isDisjoint =
					(simulationResult.getCreatedArcs().size() == 0 && simulationResult.getCreatedNodes().size() == 0);
			hyp.score = heuristic.scoreHypothesis(hyp);

			HypothesisSequence seq = new HypothesisSequence(nLookahead);
			seq.addHypothesis(hyp);
			seq.score = hyp.score;

			if (hyp.isDisjoint) {
				disjointBeam.add(seq.score, seq);
			} else {
				contiguousBeam.add(seq.score, seq);
			}
		}
	}

	/**
	 * Given a beam with a hypothesis sequence extended n steps, creates a beam
	 * containing hypotheses extended n+1 steps.
	 * 
	 * @param prevBeam
	 * @param contiguousMinPairMan
	 * @param depth
	 * @param contiguousMinPairsOnly
	 * @return
	 * @throws CorpusException
	 * @throws SegmenterException
	 * @throws HeuristicException
	 */
	private Beam<HypothesisSequence> extendHypotheses(Beam<HypothesisSequence> prevBeam)
			throws CorpusException, SegmenterException, HeuristicException {

		Beam<HypothesisSequence> newBeam = new Beam<HypothesisSequence>(prevBeam.maxSize());

		// try all possible extensions of each item in the beam
		for (HypothesisSequence seq : prevBeam) {

			// create minimal pairs and rank each hypothesis
			for (ObservableSentencePair candidate : candidates) {

				// Don't allow candidates from previous steps to be reused
				if (seq.containsSentence(candidate)) {
					continue;
				}

				Hypothesis hyp = new Hypothesis();
				hyp.sent = candidate;
				fegMan.beginSimulation();
				for (Hypothesis prevHyp : seq.getHypotheses()) {
					fegMan.simulateObserving(prevHyp.sent.getSentencePair(),
							prevHyp.sent.getMinPairMappings());
				}
				fegMan.simulateObserving(candidate.getSentencePair(),
						candidate.getMinPairMappings());
				hyp.simulationResult = fegMan.getSimulationResult();
				fegMan.endSimulation();
				hyp.isDisjoint = false;
				hyp.score = heuristic.scoreHypothesis(hyp);

				double newSeqScore = heuristic.combineHypothesisScores(seq.score, hyp.score);

				if (newSeqScore >= newBeam.getWorstScore()) {
					// clone seq (so that we don't overwrite the original)
					// and
					// then add hyp
					HypothesisSequence newSeq = new HypothesisSequence(seq);
					newSeq.addHypothesis(hyp);
					newSeq.score = newSeqScore;
					newBeam.add(newSeq.score, newSeq);
				}
			}
		}

		// boundary condition: if there were no more sentences to extend this
		// beam
		// with, then just return the old beam
		if (newBeam.currentSize() == 0) {
			return prevBeam;
		} else {
			return newBeam;
		}

	}

	public void writeFeatureDetectionReport(File resultsFile) throws IOException,
			SegmenterException, CorpusException, ParseException, TransformException {

		FeatureDetectionReport report =
				new CsvFeatureDetectionReport(resultsFile, false, themeMan, this.excludedFeatures,
						this.includedFeatures);

		for (final FeatureExpressionGraph feg : fegMan.getFeatureExpressionGraphs()) {
			final int step = 8;
			feg.calculate(step);
			report.addFEG(feg);
			feg.releaseStrangleHoldOnMemory();
		}

		report.close();
	}

	public static void main(String[] args) throws Exception {

		if (args.length != 1) {
			System.err.println("Usage: program <properties_file>");
			System.exit(1);
		}

		Properties props = PropertyUtils.getProperties(args[0]);
		SmartProperties smartProps = new SmartProperties(props);
		File navigatedCorpusFile = smartProps.getPropertyFile("navigation.navigatedCorpusFile");
		File resultsFile = smartProps.getPropertyFile("inference.eval.actualResults");
		int nScreenWidth = smartProps.getPropertyInt("progressBar.screenWidth");
		File logDir = smartProps.getPropertyFile("global.logDir");

		Date startTime = new Date();

		CorpusNavigator navigator = new CorpusNavigator(props, "navigation");
		navigator.addNavigationListener(new NavigationListener() {
			public void statusChanged(String status) {
				System.out.println("STATUS: " + status);
			}
		});
		InductiveResultsEvaluator gold = new InductiveResultsEvaluator(smartProps);

		int nWindowSize = 100;
		TaskListener bar = new TextProgressBar(System.err, "sent", nWindowSize, nScreenWidth);

		// don't count the opening book sentences since they go so fast
		if (navigator.randomMode) {
			bar.beginTask(navigator.nTargetSentenceCount);
		} else {
			bar.beginTask(navigator.nTargetSentenceCount - navigator.openingBook.size());
		}

		while (navigator.wantsToContinue()) {
			boolean gotNewAlignedTranslated = navigator.askNext();

			// only evaluate after we're done with the opening book and we
			// really got a new sentence
			// -- or, if we're selecting random sentences, only evaluate once,
			// at the end
			if (gotNewAlignedTranslated
					&& (navigator.randomMode || navigator.isOpeningBookExhausted())) {
				bar.recordEventCompletion();

				navigator.writeFeatureDetectionReport(resultsFile);

				String morphemeReport = gold.evaluateMorphemes(resultsFile);
				String expressionReport = gold.evaluateExpression(resultsFile);

				File dir =
						new File(logDir + "/navigation-"
								+ FormatUtils.formatDateTimeShort(startTime));
				dir.mkdir();
				File logFile =
						new File(dir, "sentence"
								+ StringUtils.forceNumberLength(navigator.alignedTranslated.size()
										+ "", 4) + ".results");
				PrintWriter permenantRecord = new PrintWriter(logFile);
				FileUtils.insertFile(new File(args[0]), permenantRecord);
				permenantRecord.append("#####\n");
				permenantRecord.append(morphemeReport + "\n");
				permenantRecord.append(expressionReport + "\n");
				permenantRecord.close();
			}
		}
		bar.endTask();

		Corpus navigatedCorpus = navigator.getNavigatedCorpus();
		log.info("Selected " + navigatedCorpus.getSentences().size() + " sentences.");

		// save navigated corpus to file
		Serializer.saveSentencePairs(navigatedCorpus, navigatedCorpusFile);
		System.exit(0);
	}
}
