package edu.cmu.cs.lti.avenue.navigation.search.generation1.tables;

import info.jonclark.log.LogUtils;
import info.jonclark.util.HashUtils;
import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Logger;

import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager.Type;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelMode;

/**
 * Stores seed feature structures for full sentences and for partial clauses.
 * 
 * @author jon
 */
public class SeedManager {

	private final HashMap<String, ArrayList<Seed>> clauseSeeds =
			new HashMap<String, ArrayList<Seed>>();
	private final HashMap<String, ArrayList<Seed>> participantSeeds =
			new HashMap<String, ArrayList<Seed>>();

	private int nClauseSeeds = 0;
	private int nParticipantSeeds = 0;
	private int maxClauseSeeds = 0;
	private int maxParticipantSeeds = 0;

	private static final Logger log = LogUtils.getLogger();

	public SeedManager(File file) throws IOException, ConfigurationException {
		// read in seeds from file
		BufferedReader in = new BufferedReader(new InputStreamReader(new FileInputStream(file)));

		boolean inSeed = false;
		Seed seed = null;
		String key = null;
		Type type = null;

		int nLine = 0;
		String line;
		while ((line = in.readLine()) != null) {
			nLine++;
			line = line.trim();

			if (line.equals("<seed>")) {
				inSeed = true;
				seed = new Seed();

			} else if (line.equals("</seed>")) {
				inSeed = false;

				assert key != null : "null key";
				assert seed != null : "null seed";
				assert type != null : "null type";

				if (type == Type.PARTICIPANTS) {
					HashUtils.append(participantSeeds, key, seed);
					nParticipantSeeds++;
				} else if (type == Type.CLAUSES) {
					HashUtils.append(clauseSeeds, key, seed);
					nClauseSeeds++;
				}
				seed = null;
				key = null;

			} else if (inSeed && line.startsWith("<value>")) {
				try {
					String strValue = StringUtils.substringAfter(line, "<value>");
					SmartTree tree =
							SmartTree.parse(strValue, SmartTree.SOURCE_C_STRUCT_LABEL,
									LabelMode.LABEL_ODD_NODES);
					seed.value = tree.getRootNode();
					assert seed.value.getValues().size() == 1 : "Expected 1 value";
					key = seed.value.getValues().get(0);

					log.warning("TODO: get type from FS manager.");
					type = Type.PARTICIPANTS;

				} catch (ParseException e) {
					throw new ConfigurationException("Parse error at " + file.getName() + ":"
							+ nLine + " :: " + line);
				}
			}
		}

		in.close();

		// now determine the maximum number of seeds for participants/clauses
		for (ArrayList<Seed> list : clauseSeeds.values()) {
			maxClauseSeeds = Math.max(maxClauseSeeds, list.size());
		}
		for (ArrayList<Seed> list : participantSeeds.values()) {
			maxParticipantSeeds = Math.max(maxParticipantSeeds, list.size());
		}

		log.info("SEEDS: " + "Loaded " + getNumClauseSeeds() + " clause seeds ("
				+ getMaximumClauseSeeds() + " max)");
		log.info("SEEDS: " + "Loaded " + getNumParticipantSeeds() + " participant seeds ("
				+ getMaximumParticipantSeeds() + " max)");
	}

	public ArrayList<Seed> getSeedsFor(String node, Type t) {
		if (t == Type.CLAUSES) {
			return getClauseSeeds(node);
		} else if (t == Type.PARTICIPANTS) {
			return getParticipantSeeds(node);
		} else {
			throw new RuntimeException("Unknown type: " + t);
		}
	}

	public ArrayList<Seed> getClauseSeeds(String clauseName) {
		return clauseSeeds.get(clauseName);
	}

	public ArrayList<Seed> getParticipantSeeds(String participantName) {
		return participantSeeds.get(participantName);
	}

	public int getMaximumClauseSeeds() {
		return maxClauseSeeds;
	}

	public int getMaximumParticipantSeeds() {
		return maxParticipantSeeds;
	}

	public int getNumClauseSeeds() {
		return nClauseSeeds;
	}

	public int getNumParticipantSeeds() {
		return nParticipantSeeds;
	}

	public static void main(String... args) throws Exception {
		SeedManager man = new SeedManager(new File(args[0]));
		System.out.println(man.getParticipantSeeds("actor").get(0).value);
	}
}
