package edu.cmu.cs.lti.avenue.navigation.search.generation1;

import info.jonclark.log.LogUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.logging.Logger;

import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager.Type;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.CurrentState;
import edu.cmu.cs.lti.avenue.navigation.search.generation1.tables.Seed;
import edu.cmu.cs.lti.avenue.navigation.search.generation1.tables.SeedManager;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

public class SearchSpace {

	private ScoringFunction scoringFunction;
	private FeatureStructureManager featureStructureManager;
	private SeedManager seedManager;
	private static final Logger log = LogUtils.getLogger();

	public SearchSpace(ScoringFunction scoringFunction) {
		this.scoringFunction = scoringFunction;
	}

	/**
	 * Determines the maximum (usually an overestimate) number of hypotheses
	 * that might be generated given the current state.
	 * 
	 * @param currentState
	 * @return
	 */
	public long getUpperBound(CurrentState currentState) {

		int nElicited = currentState.getElicitedSentences().size();
		int maxCompatibleClauses = featureStructureManager.getMaximumCompatibleClauses();
		int numClauses = featureStructureManager.getNumClauses();
		int maxClauseSeeds = seedManager.getMaximumClauseSeeds();
		int maxCompatibleParticipants = featureStructureManager.getMaximumCompatibleParticipants();
		int numParticipants = featureStructureManager.getNumParticipants();
		int maxParticipantSeeds = seedManager.getMaximumParticipantSeeds();
		int numClauseFeatures = featureStructureManager.getNumClauseFeatures();
		int maxClauseFeatureValues = featureStructureManager.getMaximumClauseFeatureValues();
		int numNounFeatures = featureStructureManager.getNumNounFeatures();
		int maxNounFeatureValues = featureStructureManager.getMaximumNounFeatureValues();

		long bound =
				nElicited
						* factorial(maxCompatibleClauses)
						* numClauses
						* (maxClauseSeeds + numClauseFeatures
								* maxClauseFeatureValues
								* (factorial(maxCompatibleParticipants) * numParticipants * (maxParticipantSeeds + numNounFeatures
										* maxNounFeatureValues)));
		return bound;
	}

	public long factorial(int n) {
		long result = 1;
		for (int i = 0; i < n; i++) {
			result *= result;
		}
		return result;
	}

	/**
	 * For most cases, this will be the main method that users will wish to
	 * call. It calls the other public methods of this class to get a 1-best
	 * hypothesis.
	 * 
	 * @return
	 */
	public Hypothesis performSearch(CurrentState currentState) {

		log.info("Building hypotheses... (Maximum of " + getUpperBound(currentState) + " expected)");
		ArrayList<Hypothesis> hypotheses = buildSearchSpace(currentState);
		log.info(hypotheses.size() + " hypotheses built. Scoring... ");

		// if we are going to make this a beam search, the search space creation
		// and scoring must be interwoven
		// we keep these separate for now so that we can show a progress
		// indicator at runtime

		ArrayList<Hypothesis> scoredHypotheses = scoreHypotheses(hypotheses, currentState);
		return findBestHypothesis(scoredHypotheses);
	}

	/**
	 * Build the search space, but don't score any hypotheses yet.
	 * 
	 * @param currentState
	 * @return
	 */
	public ArrayList<Hypothesis> buildSearchSpace(CurrentState currentState) {
		ArrayList<Hypothesis> hypotheses = new ArrayList<Hypothesis>();

		for (final SentencePair currentSentencePair : currentState.getElicitedSentences()) {
			SmartTree featureStructure = currentSentencePair.getFeatureStructure();

			// consider adding the same struture with a different realization
			ArrayList<SentencePair> elicitedRealizations =
					currentState.getElicitedRealizations(featureStructure);
			hypotheses.add(new Hypothesis(featureStructure, elicitedRealizations));

			// build hypotheses by deleting each element type
			// (clauses/participants) and by mutating their children
			buildHypotheses(currentState, hypotheses, currentSentencePair, featureStructure,
					Type.CLAUSES);
			buildHypotheses(currentState, hypotheses, currentSentencePair, featureStructure,
					Type.PARTICIPANTS);
			
			// TODO: what about changing each feature? conjunction? modifier?

			// consider adding a child clause AT THE MAIN CLAUSE LEVEL ONLY
			for (final String clause : featureStructureManager.getCompatibleElements(
					featureStructure.getRootNode(), featureStructure, Type.CLAUSES)) {

				for (final Seed seedClause : seedManager.getSeedsFor(clause, Type.CLAUSES)) {

					SmartTree featureStructureWithAddition =
							featureStructure.getCopyWithAddition(featureStructure.getRootNode(),
									seedClause.value);

					addHypothesesWithRealizations(currentState, hypotheses, currentSentencePair,
							featureStructureWithAddition);
				}
			}

			// consider adding child participants AT ALL CLAUSE LEVELS
			ArrayList<TreeNode> currentClauses =
					featureStructureManager.getChildren(featureStructure.getRootNode(),
							Type.CLAUSES);

			for (final TreeNode currentClause : currentClauses) {
				for (final String clause : featureStructureManager.getCompatibleElements(
						currentClause, featureStructure, Type.PARTICIPANTS)) {

					for (final Seed seedParticipant : seedManager.getSeedsFor(clause,
							Type.PARTICIPANTS)) {

						SmartTree featureStructureWithAddition =
								featureStructure.getCopyWithAddition(currentClause,
										seedParticipant.value);

						addHypothesesWithRealizations(currentState, hypotheses,
								currentSentencePair, featureStructureWithAddition);
					}
				}
			}

			// TODO: Include conjunctions and modifiers

		} // for each elicited pair

		return hypotheses;
	}

	private void buildHypotheses(CurrentState currentState, ArrayList<Hypothesis> hypotheses,
			final SentencePair currentSentencePair, SmartTree featureStructure, Type elementType) {

		ArrayList<TreeNode> currentClauses =
				featureStructureManager.getChildren(featureStructure.getRootNode(), elementType);

		for (final TreeNode currentClause : currentClauses) {

			// consider removing each child clause/participant
			SmartTree featureStructureMinusClause =
					featureStructure.getCopyWithoutSubtree(currentClause);
			addHypothesesWithRealizations(currentState, hypotheses, currentSentencePair,
					featureStructureMinusClause);

			// consider changing each of the clauses/participants
			for (final String possibleClause : featureStructureManager.getCompatibleElements(
					currentClause, featureStructure, elementType)) {

				// replacing a value with itself would not create any
				// minimal pair
				if (currentClause.getValues().get(0) != possibleClause) {
					SmartTree featureStructureSubstitute =
							featureStructure.getCopyWithValueSubstitution(currentClause,
									possibleClause);
					addHypothesesWithRealizations(currentState, hypotheses, currentSentencePair,
							featureStructureSubstitute);
				}
			} // end for featureNodes
		} // end for clauses/participants
	}

	private void addHypothesesWithRealizations(CurrentState currentState,
			ArrayList<Hypothesis> hypotheses, final SentencePair currentSentencePair,
			SmartTree featureStructureSubstitute) {

		ArrayList<SentencePair> realizations =
				currentState.getUnelicitedRealizations(featureStructureSubstitute);

		if (realizations.size() > 0) {
			for (SentencePair realization : realizations) {
				hypotheses.add(new Hypothesis(featureStructureSubstitute, realization));
			}
		} else {
			// if no realizations exist/remain, then
			// consider writing a new one
			hypotheses.add(new Hypothesis(featureStructureSubstitute, currentSentencePair));
		}
	}

	/**
	 * Later, we may add pruning here, so the set of returned hypotheses may not
	 * have the same size as the input hypotheses.
	 * 
	 * @param hypotheses
	 * @param currentState
	 * @return
	 */
	public ArrayList<Hypothesis> scoreHypotheses(ArrayList<Hypothesis> hypotheses,
			CurrentState currentState) {

		for (final Hypothesis h : hypotheses) {
			scoringFunction.score(h, currentState);
		}

		return hypotheses;
	}

	public Hypothesis findBestHypothesis(ArrayList<Hypothesis> hypotheses) {
		return Collections.max(hypotheses);
	}

	public void rankHypotheses(ArrayList<Hypothesis> hypotheses) {
		Collections.sort(hypotheses);
	}
}
