package edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.reports;

import info.jonclark.util.FileUtils;
import info.jonclark.util.FormatUtils;
import info.jonclark.util.LatexUtils;
import info.jonclark.util.StringUtils;
import info.jonclark.util.Transducer;
import info.jonclark.util.TransformException;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureContext;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureSpec;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureException;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureValueSpec;
import edu.cmu.cs.lti.avenue.morphology.SegmenterException;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.ExpressedTheme;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.ExpressedThemeManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraph;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureExpressionGraphManager;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureInteraction;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.FeatureValueCluster;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.LexicalCluster;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.ArcEvidenceCluster;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.FeatureMarking;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.FeatureSimilarity;
import edu.cmu.cs.lti.avenue.navigation.tools.UtfUtils;

public class LatexFeatureDetectionReport implements FeatureDetectionReport {

	private final String sourceLanguageName;
	private final String targetLanguageName;
	private final FeatureStructureManager fsMan;
	private final FeatureExpressionGraphManager fegMan;
	private final ExpressedThemeManager themeMan;

	private final HashSet<String> excludedFeatures;
	private final HashSet<String> includedFeatures;

	private final int nMaxDiffEvidence;
	private final int nMaxSimEvidence;
	private final int nMaxDiffCEvidence;
	private final int nMaxSimCEvidence;

	private final PrintWriter overviewOut;
	private final PrintWriter highlightsOut;
	private final PrintWriter allEvidenceOut;

	private final String strHeaderTemplate;
	private final String strOverviewTemplate;
	private final String strHighlightsTemplate;
	private final String strAllEvidenceTemplate;
	private final String strLexClusterTemplate;
	private final String strSpecTemplate;
	private final String strFooterTemplate;

	private final String outputPrefix;
	private final String outputDir;

	private final boolean reportEmptyCategories;

	private static final int LARGE_SIZE = 100000;
	private static final int VERY_LARGE_SIZE = 10000000;

	private enum ReportContext {
		OVERVIEW, HIGHLIGHTS, ALL_EVIDENCE, ALL_BY_LEX, SPEC
	}

	public LatexFeatureDetectionReport(File latexTemplate, String sourceLanuageName,
			String targetLanguageName, FeatureStructureManager fsMan,
			FeatureExpressionGraphManager fegMan, ExpressedThemeManager themeMan, String outputDir,
			String outputPrefix, int nMaxDiffEvidence, int nMaxSimEvidence, int nMaxDiffCEvidence,
			int nMaxSimCEvidence, boolean reportEmptyCategories, HashSet<String> excludedFeature,
			HashSet<String> includedFeatures) throws IOException, ParseException,
			TransformException {

		this.sourceLanguageName = sourceLanuageName;
		this.targetLanguageName = targetLanguageName;
		this.fsMan = fsMan;
		this.fegMan = fegMan;
		this.themeMan = themeMan;
		this.outputPrefix = outputPrefix;
		this.nMaxDiffEvidence = nMaxDiffEvidence;
		this.nMaxSimEvidence = nMaxSimEvidence;
		this.nMaxDiffCEvidence = nMaxDiffCEvidence;
		this.nMaxSimCEvidence = nMaxSimCEvidence;
		this.reportEmptyCategories = reportEmptyCategories;
		this.excludedFeatures = excludedFeature;
		this.includedFeatures = includedFeatures;

		String template = FileUtils.getFileAsString(latexTemplate);
		strHeaderTemplate = StringUtils.substringBetween(template, "%<header>", "%</header>");
		strOverviewTemplate = StringUtils.substringBetween(template, "%<overview>", "%</overview>");
		strHighlightsTemplate =
				StringUtils.substringBetween(template, "%<highlights>", "%</highlights>");
		strAllEvidenceTemplate =
				StringUtils.substringBetween(template, "%<all-evidence>", "%</all-evidence>");
		strLexClusterTemplate =
				StringUtils.substringBetween(template, "%<all-by-lex>", "%</all-by-lex>");
		strSpecTemplate = StringUtils.substringBetween(template, "%<spec>", "%</spec>");
		strFooterTemplate = StringUtils.substringBetween(template, "%<footer>", "%</footer>");

		outputDir = StringUtils.forceSuffix(outputDir, "/");
		this.outputDir = outputDir;

		{
			PrintWriter lexClusterOut =
					new PrintWriter(outputDir + outputPrefix + "_lex_clusters.tex");
			String lexClusterHeader =
					StringUtils.substringBefore(strLexClusterTemplate, "%<item>", false);
			lexClusterOut.println(lexClusterHeader);

			Iterable<LexicalCluster> clusters = fegMan.getLexicalClusters();
			String lexClusters =
					StringUtils.substringBetween(strLexClusterTemplate, "%<item>", "%</item>");
			lexClusters = replaceLexicalClusters(lexClusters, clusters, ReportContext.ALL_BY_LEX);
			lexClusterOut.println(lexClusters);

			String lexClusterFooter =
					StringUtils.substringAfter(strLexClusterTemplate, "%</item>", false);
			lexClusterOut.println(lexClusterFooter);
			lexClusterOut.flush();
			lexClusterOut.close();
		}

		overviewOut = new PrintWriter(outputDir + outputPrefix + "_overview.tex");
		highlightsOut = new PrintWriter(outputDir + outputPrefix + "_highlights.tex");
		allEvidenceOut = new PrintWriter(outputDir + outputPrefix + "_all_evidence.tex");

		// write headers to each file
		String overviewHeader = StringUtils.substringBefore(strOverviewTemplate, "%<item>", false);
		String highlightsHeader =
				StringUtils.substringBefore(strHighlightsTemplate, "%<item>", false);
		String allEvidenceHeader =
				StringUtils.substringBefore(strAllEvidenceTemplate, "%<item>", false);

		overviewOut.println(overviewHeader);
		highlightsOut.println(highlightsHeader);
		allEvidenceOut.println(allEvidenceHeader);
	}

	private void writeHeader(String outputPrefix) throws FileNotFoundException {
		PrintWriter out = new PrintWriter(outputDir + outputPrefix + ".tex");
		String header = replaceGlobals(strHeaderTemplate);
		String footer = replaceGlobals(strFooterTemplate);
		out.println(header);
		out.println("\\include{" + outputPrefix + "_overview}");
		out.println("\\include{" + outputPrefix + "_highlights}");
		out.println("\\include{" + outputPrefix + "_all_evidence}");
		out.println("\\include{" + outputPrefix + "_lex_clusters}");
		out.println("\\include{" + outputPrefix + "_spec}");
		out.println(footer);
		out.close();
	}

	private void writeSpec(String outputPrefix) throws ParseException, TransformException,
			FileNotFoundException {
		String result = replaceFeatureNames(strSpecTemplate, ReportContext.SPEC);
		PrintWriter specOut = new PrintWriter(outputDir + outputPrefix + "_spec.tex");
		specOut.println(result);
		specOut.close();
	}

	private String replaceFeatureInteractions(String template, final FeatureExpressionGraph feg,
			final ReportContext context) throws ParseException, TransformException {

		// scary-looking, yet straight-forward callback implementation of a
		// context-sensitive transducer
		return StringUtils.replaceBetweenMatching(template, "%<interaction>\n",
				"%</interaction>\n", false, false, new Transducer() {

					public String transform(String in) throws TransformException {

						StringBuilder builder = new StringBuilder(LARGE_SIZE);

						FeatureInteraction featureInteraction = feg.getFeatureInteractions();
						String replacement = in;
						replacement =
								StringUtils.replaceFast(replacement, "@interaction-name@",
										clean(featureInteraction.getName()));

						if (context == ReportContext.HIGHLIGHTS) {
							replacement = replaceEvidenceClusters(replacement, feg, context);
						} else {
							replacement = replaceFeatureClusters(replacement, feg, context);
						}

						// don't show empty interactions if user didn't request
						// it
						if (reportEmptyCategories || replacement != null) {
							builder.append(replacement + "\n");
							return builder.toString();
						} else {
							return "";
						}
					}
				});
	}

	/**
	 * This should called only in the feature spec context now.
	 * 
	 * @param template
	 * @param reportContext
	 * @return
	 * @throws ParseException
	 * @throws TransformException
	 */
	private String replaceFeatureNames(String template, final ReportContext reportContext)
			throws ParseException, TransformException {

		// scary-looking, yet straight-forward callback implementation of a
		// context-sensitive transducer
		return StringUtils.replaceBetweenMatching(template, "%<feature-name>\n",
				"%</feature-name>\n", false, false, new Transducer() {

					public String transform(String in) throws TransformException {

						StringBuilder builder = new StringBuilder(LARGE_SIZE);
						for (final FeatureSpec featureSpec : fsMan.getFeatureSpecs()) {

							if (FeatureExpressionGraphManager.shouldUseFeature(excludedFeatures,
									includedFeatures, featureSpec)) {

								String replacement = in;
								replacement =
										StringUtils.replaceFast(replacement, "@feature-name@",
												clean(featureSpec.name));
								replacement =
										StringUtils.replaceFast(replacement, "@feature-notes@",
												clean(featureSpec.notes));

								if (reportContext == ReportContext.SPEC) {
									String[] featureName = new String[] { featureSpec.name };

									// iterate over featureContexts
									for (final FeatureContext featureContext : fegMan.getUsedFeatureContexts()) {
										replacement =
												replaceFeatureValues(replacement,
														featureContext.name, featureName,
														Arrays.asList(featureSpec.values),
														reportContext);
									}
								} else {
									throw new Error("Unexpected context: " + reportContext);
								}
								builder.append(replacement + "\n");
							}
						}
						return builder.toString();

					}
				});
	}

	/**
	 * Should be called once per instance of matching feature-name tags
	 * 
	 * @param template
	 * @return
	 * @throws ParseException
	 * @throws TransformException
	 */
	private String replaceFeatureClusters(String template, final FeatureExpressionGraph feg,
			final ReportContext context) throws TransformException {

		return StringUtils.replaceBetweenMatching(template, "%<cluster>\n", "%</cluster>\n", false,
				false, new Transducer() {

					public String transform(String in) throws TransformException {

						try {
							StringBuilder builder = new StringBuilder(VERY_LARGE_SIZE);
							Collection<FeatureValueCluster> featureValueClusters =
									feg.getObservedValueClusters();

							for (final FeatureValueCluster featureValueCluster : featureValueClusters) {

								String clusterSubstitution =
										getClusterSubstitution(context, feg, featureValueCluster);

								String replacement = in;
								replacement =
										StringUtils.replaceFast(replacement, "@cluster-data@",
												clusterSubstitution);
								replacement =
										StringUtils.replaceFast(replacement, "@cluster-name@",
												featureValueCluster.getShortName());
								replacement =
										StringUtils.replaceFast(replacement, "@cluster-id@",
												featureValueCluster.getId());
								replacement =
										replaceFeatureValues(replacement,
												feg.getFeatureInteractions().featureContext.name,
												feg.getFeatureInteractions().featureNames,
												featureValueCluster.getFeatureValues(), context);
								builder.append(replacement + "\n");
							}

							if (featureValueClusters.size() > nMaxDiffEvidence) {
								int nRemaining = featureValueClusters.size() - nMaxDiffEvidence;
								builder.append(clean("+ " + nRemaining + " more") + " \\\\ \n");
							}

							if (featureValueClusters.size() == 0) {
								if (!reportEmptyCategories) {
									return null;
								} else {
									try {
										ArrayList<FeatureValueSpec> featureValues =
												new ArrayList<FeatureValueSpec>();
										for (final String featureName : feg.getFeatureInteractions().featureNames) {
											for (final FeatureValueSpec spec : fsMan.getSpecForFeature(featureName).values) {
												featureValues.add(spec);
											}
										}

										String replacement = in;
										replacement =
												StringUtils.replaceFast(replacement,
														"@cluster-data@",
														"No minimal pairs could be formed for this feature.");
										replacement =
												StringUtils.replaceFast(replacement,
														"@cluster-name@", "none");
										replacement =
												StringUtils.replaceFast(replacement,
														"@cluster-id@", "none");
										replacement =
												replaceFeatureValues(
														replacement,
														feg.getFeatureInteractions().featureContext.name,
														feg.getFeatureInteractions().featureNames,
														featureValues, context);
										builder.append(replacement + "\n");

									} catch (FeatureStructureException e) {
										throw new TransformException(e);
									}
								}
							}

							return builder.toString();
						} catch (SegmenterException e) {
							throw new TransformException(e);
						} catch (CorpusException e) {
							throw new TransformException(e);
						}
					}
				});
	}

	private String replaceFeatureValues(String template, final String featureContext,
			final String[] featureNames, final Iterable<FeatureValueSpec> featureValues,
			final ReportContext reportContext) throws TransformException {

		return StringUtils.replaceBetweenMatching(template, "%<feature-value>\n",
				"%</feature-value>\n", false, false, new Transducer() {

					public String transform(String in) throws TransformException {

						StringBuilder builder = new StringBuilder(LARGE_SIZE);
						for (final FeatureValueSpec valueSpec : featureValues) {
							String replacement = in;
							replacement =
									StringUtils.replaceFast(replacement, "@feature-value@",
											clean(valueSpec.getName()));
							replacement =
									StringUtils.replaceFast(replacement, "@feature-value-notes@",
											clean(valueSpec.getNotes()));
							if (reportContext == ReportContext.SPEC) {
								FeatureExpressionGraph feg =
										fegMan.getFeatureExpressionGraphFor(featureContext,
												featureNames);
								replacement =
										replaceFeatureClusters(replacement, feg, reportContext);
							}
							builder.append(replacement + "\n");
						}
						return builder.toString();

					}
				});
	}

	private String replaceLexicalClusters(String template, final Iterable<LexicalCluster> clusters,
			final ReportContext context) throws TransformException {

		return StringUtils.replaceBetweenMatching(template, "%<cluster>\n", "%</cluster>\n", false,
				false, new Transducer() {

					public String transform(String in) throws TransformException {
						try {

							StringBuilder builder = new StringBuilder(LARGE_SIZE);
							for (final LexicalCluster cluster : clusters) {
								String replacement = in;
								replacement =
										StringUtils.replaceFast(replacement, "@cluster-name@",
												clean(cluster.toString()));
								replacement =
										StringUtils.replaceFast(replacement, "@cluster-data@",
												cluster.toLatexString());
								builder.append(replacement + "\n");
							}
							return builder.toString();
						} catch (CorpusException e) {
							throw new TransformException(e);
						}
					}
				});
	}

	private String replaceEvidenceClusters(String template, final FeatureExpressionGraph feg,
			final ReportContext context) throws TransformException {

		String result =
				StringUtils.replaceBetweenMatching(template, "%<cluster>\n", "%</cluster>\n",
						false, false, new Transducer() {

							public String transform(String in) throws TransformException {

								StringBuilder builder = new StringBuilder(LARGE_SIZE);

								// first get the themes expressed by this
								// evidence
								Collection<FeatureValueCluster> valueClusters;
								try {
									valueClusters = feg.getObservedValueClusters();
								} catch (SegmenterException e) {
									throw new TransformException(e);
								} catch (CorpusException e) {
									throw new TransformException(e);
								}
								
								for (final FeatureValueCluster valueCluster : valueClusters) {

									// show the lexical items that we found to
									// be relevant
									builder.append(valueCluster.getShortName() + ": ");
									try {
										themeMan.addFeatureValueCluster(valueCluster);
										for (final ExpressedTheme theme : themeMan.getExpressedThemes(valueCluster)) {
											builder.append(clean(theme.getMorpheme().morpheme)
													+ "("
													+ FormatUtils.formatDouble2(theme.getProbability() * 100)
													+ "\\%), ");
										}
									} catch (SegmenterException e) {
										throw new TransformException(e);
									}
									builder.append("\\\n");

									// TODO: Rank the markings and only show
									// the best (and worst?) ones

									// show which minimal pairs support this
									int nDiffEvidence = 0;
									ArrayList<ArcEvidenceCluster<FeatureMarking>> markings =
											valueCluster.getDifferenceEvidence();
									for (final ArcEvidenceCluster<FeatureMarking> marking : markings) {

										if (nDiffEvidence > nMaxDiffEvidence)
											break;
										nDiffEvidence++;

										String replacement = in;
										// replacement =
										// StringUtils.replaceFast(replacement,
										// "@cluster-name@",
										// valueCluster.getShortName());
										
										try {
											replacement =
													StringUtils.replaceFast(replacement,
															"@cluster-data@", marking.toLatexString());
										} catch (CorpusException e) {
											throw new TransformException(e);
										}
										
										builder.append(replacement + "\n");
									}
									if (markings.size() > nMaxDiffEvidence) {
										int nRemaining = markings.size() - nMaxDiffEvidence;
										builder.append(clean("+ " + nRemaining + " more")
												+ " \\\\ \n \\hrule \n");
									}

								}

								// now output evidence clusters
								// for (final EvidenceCluster cluster :
								// feg.getEvidenceClusters()) {
								// String replacement = in;
								// replacement =
								// StringUtils.replaceFast(replacement,
								// "@cluster-name@",
								// cluster.getName());
								// replacement =
								// StringUtils.replaceFast(
								// replacement,
								// "@cluster-data@",
								// cluster.getRepresentativeMarking().toLatexString());
								// builder.append(replacement + "\n");
								// }
								//
								// if (feg.getEvidenceClusters().size() == 0) {
								// builder.append("No data.");
								// }

								if (builder.length() == 0) {
									return null;
								} else {
									return builder.toString();
								}
							}
						});
		return result;
	}

	private String getClusterSubstitution(ReportContext context, FeatureExpressionGraph feg,
			FeatureValueCluster valueCluster) throws SegmenterException, CorpusException {

		String clusterSubstitution = "null";

		if (context == ReportContext.OVERVIEW) {

			clusterSubstitution =
					feg.getExpressionMatrixAsStringTable().toLatexString() + " \\\\ \n"
							+ feg.getExpressionSummary().toLatexString();

		} else if (context == ReportContext.ALL_EVIDENCE) {

			StringBuilder allEvidence = new StringBuilder(LARGE_SIZE);

			int nDiffEvidence = 0;
			int nSimEvidence = 0;
			int nDiffCEvidence = 0;
			int nSimCEvidence = 0;

			for (final ArcEvidenceCluster<FeatureMarking> marking : valueCluster.getDifferenceEvidence()) {
				if (nDiffEvidence > nMaxDiffEvidence)
					break;
				nDiffEvidence++;
				allEvidence.append(marking.toLatexString() + "\n");
			}

			for (final ArcEvidenceCluster<FeatureSimilarity> sim : valueCluster.getSimilarityEvidence()) {
				if (nSimEvidence > nMaxSimEvidence)
					break;
				nSimEvidence++;
				allEvidence.append(sim.toLatexString() + "\n");
			}

			// TODO: Label which evidence is in which category (color code?)

			for (final ArcEvidenceCluster<FeatureSimilarity> sim : valueCluster.getDifferenceCounterEvidence()) {
				if (nDiffCEvidence > nMaxDiffCEvidence)
					break;
				nDiffCEvidence++;
				allEvidence.append(sim.toLatexString() + "\n");
			}

			for (final ArcEvidenceCluster<FeatureMarking> marking : valueCluster.getSimilarityCounterEvidence()) {
				if (nSimCEvidence > nMaxSimCEvidence)
					break;
				nSimCEvidence++;
				allEvidence.append(marking.toLatexString() + "\n");
			}
			clusterSubstitution = UtfUtils.replaceUnicodeCharsWith(clusterSubstitution, 'x');
			clusterSubstitution = allEvidence.toString();
		}
		return clusterSubstitution;
	}

	private static String clean(String str) {
		return LatexUtils.replaceLatexKillers(str);
	}

	private String replaceGlobals(String template) {
		String result = template;
		result = StringUtils.replaceFast(result, "@sourceLanguage@", sourceLanguageName);
		result = StringUtils.replaceFast(result, "@targetLanguage@", targetLanguageName);
		return result;
	}

	public void addFEG(FeatureExpressionGraph feg) throws ParseException, TransformException {
		{
			String overview =
					StringUtils.substringBetween(strOverviewTemplate, "%<item>", "%</item>");
			overview = replaceFeatureInteractions(overview, feg, ReportContext.OVERVIEW);
			overviewOut.println(overview);
		}
		{
			String highlights =
					StringUtils.substringBetween(strHighlightsTemplate, "%<item>", "%</item>");
			highlights = replaceFeatureInteractions(highlights, feg, ReportContext.HIGHLIGHTS);
			highlightsOut.println(highlights);
		}
		{
			String allEvidence =
					StringUtils.substringBetween(strAllEvidenceTemplate, "%<item>", "%</item>");
			allEvidence = replaceFeatureInteractions(allEvidence, feg, ReportContext.ALL_EVIDENCE);
			allEvidenceOut.println(allEvidence);
		}
	}

	public void close() throws FileNotFoundException, ParseException, TransformException {

		// write footers to each file
		String overviewFooter = StringUtils.substringAfter(strOverviewTemplate, "%</item>", false);
		String highlightsFooter =
				StringUtils.substringAfter(strHighlightsTemplate, "%</item>", false);
		String allEvidenceFooter =
				StringUtils.substringAfter(strAllEvidenceTemplate, "%</item>", false);

		overviewOut.println(overviewFooter);
		highlightsOut.println(highlightsFooter);
		allEvidenceOut.println(allEvidenceFooter);

		overviewOut.flush();
		highlightsOut.flush();
		allEvidenceOut.flush();

		overviewOut.close();
		highlightsOut.close();
		allEvidenceOut.close();

		writeHeader(outputPrefix);
		writeSpec(outputPrefix);
	}
}
