package edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive;

import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureContext;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTreeDelta;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

/**
 * Given a new sentence, decides which minimal pair it is an element of, lexical
 * overlap not considered (That is, only the feature structure is considered).
 */
public class MinimalPairManager {

	private int id = 0;
	private final Object2IntOpenHashMap<SmartTreeDelta> globalMinPairs =
			new Object2IntOpenHashMap<SmartTreeDelta>();
	// private final HashMap<Integer, ArrayList<MinimalPairMapping>>
	// minPairClusters =
	// new HashMap<Integer, ArrayList<MinimalPairMapping>>();
	private final HashSet<String> excludedFeatures;
	private final HashSet<String> includedFeatures;
	private final FeatureStructureManager fsMan;
	private final boolean useExactMinimalPairs;

	public MinimalPairManager(HashSet<String> includedFeatures, HashSet<String> excludedFeatures,
			FeatureStructureManager fsMan, boolean useExactMinimalPairs) {

		this.excludedFeatures = excludedFeatures;
		this.includedFeatures = includedFeatures;
		this.fsMan = fsMan;
		this.useExactMinimalPairs = useExactMinimalPairs;
	}

	/**
	 * Maps a sentence pair to all possible Wildcard Feature Structure (WFS)
	 * assigning each globally unique WFS a unique integer identifier. A list of
	 * these WFS's and their associated information is returned encapsulated in
	 * the MinimalPairMapping class. Note that a MinimalPairMapping represents a
	 * possible minimal pair, though no compatible sentence may exist to form a
	 * minimal pair.
	 * 
	 * @param pair
	 * @param nInteractions
	 * @param useSentenceInLaterDeltas
	 * @return
	 * @throws CorpusException
	 */
	public ArrayList<MinimalPairMapping> mapToMinimalPairs(SmartTree featureStructure,
			int nInteractions, boolean useSentenceInLaterDeltas) throws CorpusException {

		ArrayList<MinimalPairMapping> minPairs = new ArrayList<MinimalPairMapping>();

		String[] wildcardFeatures = new String[nInteractions];
		String[] wildcardValues = new String[nInteractions];
		int[] wildcardIndices = new int[nInteractions];
		HashSet<String> featureLocks = new HashSet<String>();
		createWildcardNodes(featureStructure, minPairs, featureLocks, wildcardFeatures,
				wildcardValues, wildcardIndices, 0, 0, useSentenceInLaterDeltas);

		featureStructure.clearCache();

		return minPairs;
	}

	/**
	 * A recursive helper for mapToMinimalPairs()
	 * 
	 * @param pair
	 * @param minPairs
	 * @param featureStructure
	 * @param allNodes
	 * @param wildcardFeatures
	 * @param wildcardValues
	 * @param nDepth
	 * @param useSentenceInLaterDeltas
	 *            Adds this sentencePair to the accumulation of sentence pairs
	 *            for use in a later delta with other sentencePairs.
	 * @throws CorpusException
	 */
	private void createWildcardNodes(SmartTree featureStructure,
			ArrayList<MinimalPairMapping> minPairs, HashSet<String> featureLocks,
			String[] wildcardFeatures, String[] wildcardValues, int[] wildcardIndices,
			int firstNode, int nDepth, boolean useSentenceInLaterDeltas) throws CorpusException {

		featureStructure.sort();
		List<TreeNode> allNodes = featureStructure.getAllNodes();

		for (int i = firstNode; i < allNodes.size(); i++) {

			TreeNode node = allNodes.get(i);
			ArrayList<String> values = node.getValues();
			if (values.size() == 2) {

				wildcardFeatures[nDepth] = values.get(0);
				if (includedFeatures.isEmpty()
						|| includedFeatures.contains(wildcardFeatures[nDepth])) {

					if (!excludedFeatures.contains(wildcardFeatures[nDepth])
							&& !featureLocks.contains(wildcardFeatures[nDepth])) {

						wildcardValues[nDepth] = values.get(1);
						int treeIndex = node.getAbsoluteTreeIndex();
						wildcardIndices[nDepth] = treeIndex;

						featureLocks.add(wildcardFeatures[nDepth]);

						if (nDepth < wildcardFeatures.length - 1) {

							// keep filling the arrays
							createWildcardNodes(featureStructure, minPairs, featureLocks,
									wildcardFeatures, wildcardValues, wildcardIndices, i + 1,
									nDepth + 1, useSentenceInLaterDeltas);
						} else {

							// we've hit a node, output the result
							SmartTreeDelta wildcardFeatureStructure =
									new SmartTreeDelta(featureStructure, wildcardIndices,
											useExactMinimalPairs);

							// make copy since we will be changing the
							// referenced
							// arrays
							String[] wildcardFeaturesCopy = new String[wildcardFeatures.length];
							String[] wildcardValuesCopy = new String[wildcardValues.length];
							System.arraycopy(wildcardFeatures, 0, wildcardFeaturesCopy, 0,
									wildcardFeatures.length);
							System.arraycopy(wildcardValues, 0, wildcardValuesCopy, 0,
									wildcardValues.length);

							Integer minPair = globalMinPairs.get(wildcardFeatureStructure);
							if (minPair == null) {
								// System.out.println("Creating m" + (id + 1) +
								// ": "
								// +
								// wildcardFeatureStructure.toString(LabelDisplay.NONE));
								minPair = id;
								if (useSentenceInLaterDeltas) {
									globalMinPairs.put(wildcardFeatureStructure, id);
								}
								id++;
							} else {
								// System.out.println("Adding to m" + (minPair +
								// 1) + ": "
								// +
								// wildcardFeatureStructure.toString(LabelDisplay.NONE));
							}
							FeatureContext featureContext = getFeatureContext(node);

							assert minPairs != null : "null minPairs list";
							assert minPair != null : "null minPair";
							assert wildcardFeaturesCopy != null : "null wildcardFeaturesCopy";
							assert wildcardValuesCopy != null : "null wildcardValuesCopy";
							assert featureContext != null : "null featureContext";

							minPairs.add(new MinimalPairMapping(minPair, wildcardFeaturesCopy,
									wildcardValuesCopy, wildcardFeatureStructure, featureContext));
						}

						// leave things like we found them
						wildcardIndices[nDepth] = -1;
						featureLocks.remove(wildcardFeatures[nDepth]);
					} // if excluded feature
				} // if included feature
			} else if (values.size() == 1) {
				// this is a feature context node, but we'll get at it from a
				// different angle
			}
		}
	}

	private FeatureContext getFeatureContext(TreeNode node) throws CorpusException {
		if (node.getParentNode() == null) {
			return FeatureContext.ROOT;
		} else {

			// determine whether this is a clause-level feature
			// (context depth=1)
			// or a participant-level feature
			// (context depth=2)

			String strFeature = node.getValues().get(0);
			int depth;
			if (fsMan.isClauseFeature(strFeature)) {
				depth = 1;
			} else {
				// if (fsMan.isParticipantFeature(strFeature)) {
				depth = 2;
			}
			// else {
			// throw new RuntimeException("Not a clause-level or
			// participant-level feature: "
			// + strFeature);
			// }

			TreeNode parent1 = node.getParentNode();
			if (parent1.getParentNode() == null) {
				return FeatureContext.ROOT;
			} else {

				String strContext1 = parent1.getParentNode().getValues().get(0);
				if (depth == 1) {
					return fsMan.getFeatureContextFor(strContext1);
				} else {
					TreeNode parent2 = parent1.getParentNode().getParentNode();
					if (parent2.getValues().size() == 0) {
						return fsMan.getFeatureContextFor(FeatureContext.ROOT.name + "/"
								+ strContext1);
					} else {
						String strContext2 = parent2.getValues().get(0);
						return fsMan.getFeatureContextFor(strContext2 + "/" + strContext1);
					}
				}

			}
		}
	}
}
