package edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;

import edu.cmu.cs.lti.avenue.featurespecification.FeatureValueSpec;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.ArcEvidenceCluster;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.FeatureMarking;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.FeatureSimilarity;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.evidence.NodeEvidence;

/**
 * Represents a cluster of feature values that are believed to be expressed
 * jointly. For example, in English, both dual number and plural number are
 * expressed in the same way (jointly). Thus, we would expect these feature
 * values to find their way into the same FeatureValueCluster.
 * 
 * @author jon
 */
public class FeatureValueCluster {

	private static final int SMALL_SIZE = 1;
	public final static int MAX_VALUES_IN_SHORT_NAME = 3;

	private boolean unobserved;
	private HashSet<FeatureValueInteraction> featureInteractions =
			new HashSet<FeatureValueInteraction>(SMALL_SIZE);

	private ArrayList<ArcEvidenceCluster<FeatureSimilarity>> similarityEvidence =
			new ArrayList<ArcEvidenceCluster<FeatureSimilarity>>(SMALL_SIZE);
	private ArrayList<ArcEvidenceCluster<FeatureMarking>> differenceEvidence =
			new ArrayList<ArcEvidenceCluster<FeatureMarking>>(SMALL_SIZE);

	private ArrayList<ArcEvidenceCluster<FeatureMarking>> similarityCounterEvidence =
			new ArrayList<ArcEvidenceCluster<FeatureMarking>>(SMALL_SIZE);
	private ArrayList<ArcEvidenceCluster<FeatureSimilarity>> differenceCounterEvidence =
			new ArrayList<ArcEvidenceCluster<FeatureSimilarity>>(SMALL_SIZE);

	private HashMap<String, NodeEvidence> words;

	private String id = null;

	public HashMap<String, NodeEvidence> getWords() {
		if (words == null) {
			words = new HashMap<String, NodeEvidence>();
		}
		return words;
	}

	public void addInteraction(FeatureValueInteraction s) {
		featureInteractions.add(s);
	}

	public boolean contains(FeatureValueInteraction s) {
		return featureInteractions.contains(s);
	}

	public void addDifferenceEvidence(Collection<ArcEvidenceCluster<FeatureMarking>> markings,
			FeatureValueInteraction includedFeatureValue) {
		differenceEvidence.addAll(markings);
		featureInteractions.add(includedFeatureValue);
	}

	public void addSimilarityEvidence(
			Collection<ArcEvidenceCluster<FeatureSimilarity>> similarities,
			FeatureValueInteraction valueA, FeatureValueInteraction valueB) {

		similarityEvidence.addAll(similarities);
		featureInteractions.add(valueA);
		featureInteractions.add(valueB);
	}

	public void addDifferenceCounterEvidence(
			Collection<ArcEvidenceCluster<FeatureSimilarity>> markings) {
		differenceCounterEvidence.addAll(markings);
	}

	public void addSimilarityCounterEvidence(
			Collection<ArcEvidenceCluster<FeatureMarking>> similarities) {
		similarityCounterEvidence.addAll(similarities);
	}

	public void addCluster(FeatureValueCluster cluster) {
		featureInteractions.addAll(cluster.featureInteractions);
		similarityEvidence.addAll(cluster.similarityEvidence);
		differenceEvidence.addAll(cluster.differenceEvidence);
		similarityCounterEvidence.addAll(cluster.similarityCounterEvidence);
		differenceCounterEvidence.addAll(cluster.differenceCounterEvidence);
	}

	public FeatureValueCluster subtract(FeatureValueInteraction featureValue) {

		// TODO: Make the subtraction process faster by hashing the evidence
		// arrays
		FeatureValueCluster result = new FeatureValueCluster();
		result.featureInteractions.addAll(featureInteractions);
		result.featureInteractions.remove(featureValue);

		for (final ArcEvidenceCluster<FeatureSimilarity> s : similarityEvidence) {
			if (!s.getFeatureValueA().equals(featureValue)
					&& !s.getFeatureValueB().equals(featureValue)) {
				result.similarityEvidence.add(s);
			}
		}
		for (final ArcEvidenceCluster<FeatureMarking> s : differenceEvidence) {
			if (!s.getFeatureValueA().equals(featureValue)
					&& !s.getFeatureValueB().equals(featureValue)) {
				result.differenceEvidence.add(s);
			}
		}
		for (final ArcEvidenceCluster<FeatureMarking> s : similarityCounterEvidence) {
			if (!s.getFeatureValueA().equals(featureValue)
					&& !s.getFeatureValueB().equals(featureValue)) {
				result.similarityCounterEvidence.add(s);
			}
		}
		for (final ArcEvidenceCluster<FeatureSimilarity> s : differenceCounterEvidence) {
			if (!s.getFeatureValueA().equals(featureValue)
					&& !s.getFeatureValueB().equals(featureValue)) {
				result.differenceCounterEvidence.add(s);
			}
		}

		return result;
	}

	public HashSet<FeatureValueInteraction> getFeatureInteractions() {
		return featureInteractions;
	}

	public ArrayList<FeatureValueSpec> getFeatureValues() {
		ArrayList<FeatureValueSpec> values = new ArrayList<FeatureValueSpec>();
		for (final FeatureValueInteraction interaction : featureInteractions) {
			for (final FeatureValueSpec spec : interaction.featureValueSpecs) {
				values.add(spec);
			}
		}
		return values;
	}

	public ArrayList<ArcEvidenceCluster<FeatureSimilarity>> getSimilarityEvidence() {
		return similarityEvidence;
	}

	public ArrayList<ArcEvidenceCluster<FeatureMarking>> getDifferenceEvidence() {
		return differenceEvidence;
	}

	public ArrayList<ArcEvidenceCluster<FeatureMarking>> getSimilarityCounterEvidence() {
		return similarityCounterEvidence;
	}

	public ArrayList<ArcEvidenceCluster<FeatureSimilarity>> getDifferenceCounterEvidence() {
		return differenceCounterEvidence;
	}

	public void pruneNonuniqueEvidence() {
		prune(differenceEvidence);
		prune(differenceCounterEvidence);
		prune(similarityEvidence);
		prune(similarityCounterEvidence);
	}

	private static <T> void prune(ArrayList<T> list) {
		HashSet<T> set = new HashSet<T>(list.size());
		for (int i = 0; i < list.size(); i++) {
			T element = list.get(i);
			if (!set.contains(element)) {
				set.add(element);
			} else {
				list.remove(i);
				i--;
			}
		}
	}

	public String getParentFeatureNames() {

		HashSet<String> names = new HashSet<String>();
		for (FeatureValueInteraction interaction : this.featureInteractions) {
			for (String featureName : interaction.getParent().featureNames) {
				names.add(featureName);
			}
		}

		StringBuilder builder = new StringBuilder();
		for (String name : names) {
			builder.append(name + "_");
		}
		if (builder.lastIndexOf("_") == builder.length() - 1) {
			builder.deleteCharAt(builder.length() - 1);
		}
		
		return builder.toString();
	}

	public String getShortName() {
		StringBuilder builder = new StringBuilder();

		int i = 0;
		final String JOINER = "_";
		for (final FeatureValueInteraction featureValueInteraction : featureInteractions) {
			if (i == MAX_VALUES_IN_SHORT_NAME) {
				break;
			}
			builder.append(featureValueInteraction.getName() + JOINER);
			i++;
		}
		if (builder.length() > JOINER.length())
			builder.delete(builder.length() - JOINER.length(), builder.length());
		return builder.toString();
	}

	public String getId() {
		if (id == null) {
			StringBuilder builder = new StringBuilder();
			boolean firstTime = true;
			for (final FeatureValueInteraction spec : featureInteractions) {
				if (firstTime) {
					firstTime = false;
					builder.append(spec.getParent().getName() + "-");
				}
				builder.append(spec.getName() + "-");
			}
			if (builder.length() > 0)
				builder.deleteCharAt(builder.length() - 1);
			id = builder.toString();
		}
		return id;
	}

	protected void setUnobserved(boolean b) {
		this.unobserved = b;
	}

	public boolean isUnobserved() {
		return unobserved;
	}

	public int hashCode() {
		return getId().hashCode();
	}

	public boolean equals(Object obj) {
		if (obj instanceof FeatureValueCluster) {
			FeatureValueCluster other = (FeatureValueCluster) obj;
			return this.id.equals(other.id);
		} else {
			return false;
		}
	}

	public String toString() {
		return getId();
	}
}
