package edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive;

import info.jonclark.log.LogUtils;
import info.jonclark.util.CollectionUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.parsers.ParserConfigurationException;

import org.xml.sax.SAXException;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureContext;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureSpec;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureException;
import edu.cmu.cs.lti.avenue.featurespecification.FeatureStructureManager;
import edu.cmu.cs.lti.avenue.morphology.Segmenter;
import edu.cmu.cs.lti.avenue.morphology.SegmenterException;
import edu.cmu.cs.lti.avenue.navigation.featuredetection.inductive.simulation.SimulationResult;
import edu.cmu.cs.lti.avenue.navigation.search.generation1.tables.ConfigurationException;

/**
 * Given a new sentence, file it in the correct feature expression graph. This
 * class may be seen as a near top-level class for ruleless feature detection.
 */
public class FeatureExpressionGraphManager {

	private static final Logger log = LogUtils.getLogger();

	private static final int EXPECTED_FEGS = 4000;

	private final LexicalClusterManager lexMan;

	private final int nInteractions;
	private final boolean separateFegPerContext;
	private final HashSet<String> excludedFeatures;
	private final HashSet<String> includedFeatures;
	private final HashMap<String, FeatureExpressionGraph>[] fegs;
	private final ArrayList<FeatureInteraction> interactions = new ArrayList<FeatureInteraction>();
	private final PlateauFunction distancePenalty;

	private final FeatureStructureManager fsMan;

	@SuppressWarnings("unchecked")
	public FeatureExpressionGraphManager(double expressionThreshold, FeatureStructureManager fsMan,
			Segmenter segmenter, PlateauFunction distancePenalty, LexicalClusterManager lexMan,
			HashSet<String> includedFeatures, HashSet<String> excludedFeatures, int nInteractions,
			boolean separateFegPerContext) throws IOException, SAXException,
			ParserConfigurationException, ConfigurationException {

		this.fsMan = fsMan;
		this.excludedFeatures = excludedFeatures;
		this.includedFeatures = includedFeatures;
		this.distancePenalty = distancePenalty;

		this.lexMan = lexMan;

		this.nInteractions = nInteractions;
		this.separateFegPerContext = separateFegPerContext;
		fegs = (HashMap<String, FeatureExpressionGraph>[]) new HashMap[nInteractions];

		for (int i = 0; i < nInteractions; i++) {
			fegs[i] = new HashMap<String, FeatureExpressionGraph>(EXPECTED_FEGS);
			Collection<FeatureContext> featureContexts = fsMan.getFeatureContexts();

			if (separateFegPerContext) {
				for (final FeatureContext featureContext : featureContexts) {
					initializeInteractionsForCount(expressionThreshold, fsMan, segmenter,
							featureContext, i + 1);
				}
			} else {
				initializeInteractionsForCount(expressionThreshold, fsMan, segmenter,
						FeatureContext.ANY, i + 1);
			}
		}
	}

	public static boolean shouldUseFeature(HashSet<String> excludedFeatures,
			HashSet<String> includedFeatures, FeatureSpec featureSpec) {

		if (excludedFeatures.contains(featureSpec.name))
			return false;

		if (includedFeatures.isEmpty() || includedFeatures.contains(featureSpec.name)) {
			return true;
		} else {
			return false;
		}
	}

	private void initializeInteractionsForCount(double expressionThreshold,
			FeatureStructureManager fsMan, Segmenter segmenter, FeatureContext featureContext,
			int nInteractions) {

		FeatureSpec[] interactingSpecs = new FeatureSpec[nInteractions];
		initializeInteractions(expressionThreshold, fsMan, segmenter, featureContext,
				interactingSpecs, nInteractions, 0);
	}

	private void initializeInteractions(double expressionThreshold, FeatureStructureManager fsMan,
			Segmenter segmenter, FeatureContext featureContext, FeatureSpec[] interactingSpecs,
			int nInteractions, int nCurrentDepth) {

		// initialize FEGs so that we know if any features are unobserved
		if (nCurrentDepth < nInteractions) {
			for (final FeatureSpec featureSpec : fsMan.getFeatureSpecs()) {

				if (shouldUseFeature(excludedFeatures, includedFeatures, featureSpec)) {

					interactingSpecs[nCurrentDepth] = featureSpec;

					initializeInteractions(expressionThreshold, fsMan, segmenter, featureContext,
							interactingSpecs, nInteractions, nCurrentDepth + 1);
				}
			}
		} else {
			FeatureSpec[] interactingSpecsCopy = new FeatureSpec[interactingSpecs.length];
			System.arraycopy(interactingSpecs, 0, interactingSpecsCopy, 0, interactingSpecs.length);
			FeatureInteraction interaction =
					new FeatureInteraction(interactingSpecsCopy, featureContext);
			interactions.add(interaction);
			FeatureExpressionGraph feg =
					new FeatureExpressionGraph(interaction, expressionThreshold, segmenter,
							distancePenalty);
			fegs[nInteractions - 1].put(interaction.getName(), feg);
		}

	}

	/**
	 * Returns the global configuration setting for whether or not we use
	 * feature context (e.g. actor, undergoer, relative-clause), to create
	 * separate FEG's for the same feature having different contexts.
	 * 
	 * @return
	 */
	public boolean shouldConsiderContext() {
		return separateFegPerContext;
	}

	public void observe(SentencePair pair, List<MinimalPairMapping> minPairs)
			throws CorpusException {

		doObservation(pair, minPairs, false);
	}

	public void simulateObserving(SentencePair pair, List<MinimalPairMapping> minPairs)
			throws CorpusException {

		doObservation(pair, minPairs, true);
	}

	private void doObservation(SentencePair pair, List<MinimalPairMapping> minPairs,
			boolean simulate) throws CorpusException {

		int nLexCluster = lexMan.mapToCluster(pair);

		try {
			for (final MinimalPairMapping minPair : minPairs) {

				// skip features as requested by the user
				if (CollectionUtils.containsAny(excludedFeatures, minPair.featureNames))
					continue;
				if (includedFeatures.isEmpty()
						|| CollectionUtils.containsAll(includedFeatures, minPair.featureNames)) {

					// do we care about the context this was found in? (e.g.
					// participant/clause type)
					String featureContext;
					if (separateFegPerContext) {
						featureContext = minPair.featureContext.name;
					} else {
						featureContext = FeatureContext.ANY.name;
					}
					String strFeatureNamesWithContext =
							FeatureInteraction.makeName(featureContext, minPair.featureNames);

					int interactionsIndex = minPair.featureNames.length - 1;
					HashMap<String, FeatureExpressionGraph> fegsForI = fegs[interactionsIndex];
					FeatureExpressionGraph feg = fegsForI.get(strFeatureNamesWithContext);
					if (feg == null) {
						throw new RuntimeException(
								"null FEG: expected all FEGs to already be present, but didn't find: "
										+ strFeatureNamesWithContext + "; see "
										+ pair.getFeatureSourceLine());
					}

					// System.out.println("OBSERVED: "
					// + StringUtils.untokenize(minPair.featureNames, "X") +
					// "\tm"
					// + (minPair.id + 1) + "\tc" + (minPair.featureContext.id +
					// 1) + "\tL"
					// + (nLexCluster + 1) + "\t"
					// + StringUtils.untokenize(minPair.featureValues, "X") +
					// "\tS"
					// + pair.getId());

					if (simulate) {
						feg.simulateObserving(pair, minPair.featureValues, nLexCluster, minPair.id,
								minPair.featureContext.id, minPair);
					} else {
						feg.addObservation(pair, minPair.featureValues, nLexCluster, minPair.id,
								minPair.featureContext.id, minPair);
					}
				}
			}
		} catch (FeatureStructureException e) {
			log.warning("Error in feature structure for sentence: "
					+ pair.getDisplaySourceSentence() + "\n" + pair.getFeatureSourceLine() + "\n"
					+ e.getMessage() + "\n");
		}
	}

	public void beginSimulation() {
		for (int i = 0; i < fegs.length; i++) {
			for (FeatureExpressionGraph feg : fegs[i].values()) {
				feg.beginSimulation();
			}
		}
	}

	public void endSimulation() {
		for (int i = 0; i < fegs.length; i++) {
			for (FeatureExpressionGraph feg : fegs[i].values()) {
				feg.endSimulation();
			}
		}
	}

	public SimulationResult getSimulationResult() throws SegmenterException, CorpusException {

		SimulationResult result = new SimulationResult();

		for (int i = 0; i < fegs.length; i++) {
			for (FeatureExpressionGraph feg : fegs[i].values()) {
				SimulationResult fegResult = feg.getSimulationResult();
				result.add(fegResult);
			}
		}

		return result;
	}

	public ArrayList<FeatureInteraction> getFeatureInteractions() {
		return interactions;
	}

	/**
	 * Get the feature expression graphs, which have a 1-to-1 correspondence
	 * with the features. That is featureSpec.length is equal to the size of the
	 * collection returned by this method.
	 * 
	 * @return
	 */
	public ArrayList<FeatureExpressionGraph> getFeatureExpressionGraphs() {
		ArrayList<FeatureExpressionGraph> values = new ArrayList<FeatureExpressionGraph>();
		for (int i = 0; i < fegs.length; i++) {
			values.addAll(fegs[i].values());
			if (i == 0) {
				// assert values.size() == (fsMan.getNumFeatures() -
				// excludedFeatures.size())
				// || (values.size() == includedFeatures.size());
			}
		}
		return values;
	}

	public Collection<FeatureContext> getUsedFeatureContexts() {
		if (separateFegPerContext) {
			return fsMan.getFeatureContexts();
		} else {
			ArrayList<FeatureContext> list = new ArrayList<FeatureContext>(1);
			list.add(FeatureContext.ANY);
			return list;
		}
	}

	public FeatureExpressionGraph getFeatureExpressionGraphFor(String featureContext,
			String... featureNames) {

		String strFeatureNamesWithContext =
				FeatureInteraction.makeName(featureContext, featureNames);
		FeatureExpressionGraph value =
				fegs[featureNames.length - 1].get(strFeatureNamesWithContext);
		assert value != null : "no result found for: " + strFeatureNamesWithContext;
		return value;
	}

	public Iterable<LexicalCluster> getLexicalClusters() {
		return lexMan.getClusters();
	}
}
