/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import info.jonclark.properties.SmartProperties;
import info.jonclark.stat.SecondTimer;
import info.jonclark.util.FileUtils;

import java.io.File;
import java.io.PrintWriter;
import java.util.ArrayList;

import edu.cmu.cs.lti.avenue.corpus.Corpus;
import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.corpus.Serializer;
import edu.cmu.cs.lti.avenue.navigation.wals.WALS;

/**
 * Given a merged c-f-phi-plus-structure file and a rules file, output which
 * sentences the rules apply to.
 * 
 * This is the main class for stand-alone decutive feature detection.
 */
public class RuleTester {
	public static void main(String[] args) throws Exception {
		if (args.length != 1) {
			System.err.println("Usage: program <properties_config_file>");
			System.exit(1);
		}
		
		File propsFile = new File(args[0]);
		SmartProperties props = new SmartProperties(propsFile);
		
		String encoding = props.getPropertyString("data.encoding");

		File cfStructFile = props.getPropertyFile("data.cfStructFile");
		File rulesFile = props.getPropertyFile("deduction.rulesFile");
		File walsDir = props.getPropertyFile("global.walsDir");
		File outputFile = props.getPropertyFile("deduction.outputFile");
		File logDir = props.getPropertyFile("deduction.logDir");

		BooleanFunctionEvaluator.VERBOSITY = props.getPropertyInt("deduction.verbosity");
		final boolean verbose = BooleanFunctionEvaluator.VERBOSITY > 0; 

		SecondTimer sentencesTime = new SecondTimer(true, true);
		Corpus corpus = Serializer.loadSentencePairs(cfStructFile, true, encoding);
		System.out.println("Loaded " + corpus.getSentences().size() + " sentences in "
				+ sentencesTime.getSecondsFormatted() + " seconds");

		WALS wals = new WALS(walsDir);
		FeatureManager featureMan =
				new FeatureManager(wals.getFeatureNames(), wals.getAllowableFeatureNames());

		SecondTimer rulesTime = new SecondTimer(true, true);
		ArrayList<Rule> rules = Serializer.loadRules(rulesFile, featureMan);
		System.out.println("Loaded " + rules.size() + " rules in "
				+ rulesTime.getSecondsFormatted() + " seconds");
		SecondTimer compTime = new SecondTimer(true, true);
		int i = 0;
		for (Rule rule : rules) {
			i++;
			PrintWriter ruleLog = new PrintWriter(logDir.getAbsolutePath() + "/rule" + i + ".log");

			System.out.println("RULE: " + rule.toString());
			ruleLog.println("RULE: " + rule.toString());
			for (SentencePair pair : corpus.getSentences()) {

				ArrayList<SentencePatternMatch> matches = rule.match(pair, false);
				for (SentencePatternMatch match : matches) {

					if (verbose) {
						System.out.println("MATCHED: " + match + "\n"
								+ match.getSentencePair().getFeatureStructure() + "\n"
								+ match.getSentencePair().getSourceConstituentStructure());
						System.out.println("FOR VARIABLE VALUES: "
								+ match.getVariableValuesAsString());

						ruleLog.println("MATCHED: " + match + "\n"
								+ match.getSentencePair().getFeatureStructure() + "\n"
								+ match.getSentencePair().getSourceConstituentStructure());
						ruleLog.println("FOR VARIABLE VALUES: " + match.getVariableValuesAsString());
					} else {
//						System.out.println("MATCHED: " + match);
//						ruleLog.println("MATCHED: " + match);
					}
				}
			} // for each sentence pair

			if (rule.isReadyToEvaluate()) {
				final boolean includeSentenceInstances = verbose;
				System.out.println(rule.getRuleMatchSummary(includeSentenceInstances));
				ruleLog.println(rule.getRuleMatchSummary(includeSentenceInstances));

				ArrayList<ComparisonResult> results = rule.evaluate();
				System.out.println("Rule fired " + results.size() + " implications.");
				ruleLog.println(rule.getRuleMatchSummary(includeSentenceInstances));

				for (ComparisonResult result : results) {

					System.out.println("---------------------------------------");
					System.out.println("IMPLICATION: " + result.toString());
					ruleLog.println("---------------------------------------");
					ruleLog.println("IMPLICATION: " + result.toString());
					if (verbose) {
						System.out.println("EVIDENCE: " + result.getFullTrace());
						ruleLog.println("EVIDENCE: " + result.getFullTrace());
					} else {
						System.out.println("EVIDENCE: " + result.getShortTrace());
						ruleLog.println("EVIDENCE: " + result.getShortTrace());
					}

				} // for each result
				
				// save implications to file for later evaluation
				
				// TODO: cluster best results
				Implication[] maxImplications = rule.resolveFinalImplications(results);
				for (Implication implication : maxImplications) {
					System.out.println("MAX IMPLICATION: " + implication);
					ruleLog.println("MAX IMPLICATION: " + implication);
					featureMan.setFeatureValue(implication.getFeatureName(),
							implication.getFeatureValue());
				} // for each implication
			} else {
				System.out.println("Rule did not have enough data to be evaluated.");
				System.out.println(rule.getRuleMatchSummary(true));
				ruleLog.println("Rule did not have enough data to be evaluated.");
				ruleLog.println(rule.getRuleMatchSummary(true));
			} // isReadyToEvaluate()

			System.out.println("=================================================");
			ruleLog.close();
		} // for rules

		System.out.println("Evaluated rules in " + compTime.getSecondsFormatted() + " seconds");

		System.out.println("Saving results to file: " + outputFile.getAbsolutePath());
		FileUtils.saveFileFromString(outputFile, featureMan.toString());
	}
}
