/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.IF;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.OVERLAP;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.RULE;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SENTENCES;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.VARIABLES;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.WALS;
import info.jonclark.util.HashUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map.Entry;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.SymbolTable;
import edu.cmu.cs.lti.avenue.trees.smart.TreeGrepper;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;
import edu.cmu.cs.lti.avenue.trees.smart.Variable;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelMode;

// TODO: Add functions:
// target-lex-dependents
// source-lex-dependents,
// x-contains-y (for detecting asymmetry)
// not-present,
// different-word-order
//
// ~ operator before node or value to indicate "NOT"
// 
// note in docs that the logic is "there exists" rather than "for all"
//
// Dealing with percentages of data points firing results

/**
 * A rule that specifies patterns for 1 or more sentences that must then match
 * given constraints (usually constraints on the target language lexical items)
 * in order to fire a specified feature value (such as a WALS universal). TODO:
 * Allow nested if-else-then statements to perform error checking
 * <p>
 * Note to developers of this class: Any method taking a LexicalResult and
 * returning a ComparisonResult should most likely call the method
 * "hasProperVariableUsage()" to avoid very strange bugs.
 */
public class Rule {

	private static final int EXPECTED_MATCHES_PER_VARIABLE = 2;
	private static final int EXPECTED_NUM_VARIABLES = 5;

	private final HashMap<String, Integer> sentenceVariablesByName;
	private final ArrayList<String> sentenceVariablesByIndex;
	private final TreeNode[] sentencePatterns;
	private final ArrayList<SentencePatternMatch>[] ruleMatches;

	private final TreeNode overlapFunction;
	private final Condition[] conditions;
	private final SymbolTable variableData;

	private final FeatureManager featureMan;

	private final String serialized;

	protected Rule(String serialized, TreeNode[] sentencePatterns, Condition[] conditions,
			SymbolTable variableData, ArrayList<String> sentencePatternVars,
			TreeNode overlapFunction, FeatureManager featureMan) {

		this.conditions = conditions;
		this.serialized = serialized;

		this.sentenceVariablesByIndex = sentencePatternVars;
		this.sentenceVariablesByName = new HashMap<String, Integer>(sentencePatternVars.size());
		for (int i = 0; i < sentencePatternVars.size(); i++) {
			this.sentenceVariablesByName.put(sentencePatternVars.get(i), i);
		}

		this.sentencePatterns = sentencePatterns;
		this.ruleMatches = createArray(sentencePatterns.length);
		for (int i = 0; i < ruleMatches.length; i++)
			this.ruleMatches[i] =
					new ArrayList<SentencePatternMatch>(EXPECTED_MATCHES_PER_VARIABLE);

		this.variableData = variableData;
		this.overlapFunction = overlapFunction;
		this.featureMan = featureMan;
	}

	public FeatureManager getFeatureManager() {
		return featureMan;
	}

	/**
	 * Hack to get around Java's generic array deficiency
	 */
	@SuppressWarnings("unchecked")
	private static ArrayList<SentencePatternMatch>[] createArray(int size) {
		return (ArrayList<SentencePatternMatch>[]) new ArrayList[size];
	}

	public static Rule createRule(String serialized, FeatureManager featureMan)
			throws ParseException {

		if (serialized.trim().length() <= 0)
			throw new ParseException("Zero length rule not allowed.", -1);

		SmartTree ruleTree = SmartTree.parse(serialized, "r", LabelMode.LABEL_ALL_NODES);

		// TODO: resolve which sentences and feature nodes we might have to deal
		// with ahead of time

		HashMap<String, Variable> variables = new HashMap<String, Variable>(EXPECTED_NUM_VARIABLES);
		ArrayList<Condition> conditionList = new ArrayList<Condition>();
		ArrayList<TreeNode> sentencePatterns = new ArrayList<TreeNode>();
		ArrayList<String> sentencePatternVariables = new ArrayList<String>();
		TreeNode overlapFunction = null;

		TreeNode root = ruleTree.getRootNode();

		if (root.getValues().get(0) != RULE) {
			throw new ParseException("first node of rule must be \"rule\": " + root, -1);
		}

		for (TreeNode infoNode : root.getChildren()) {
			if (infoNode.getValues().size() > 0) {
				String childName = infoNode.getValues().get(0);
				if (childName == VARIABLES) {
					readVariables(variables, infoNode);
				} else if (childName == SENTENCES) {
					readSentencePatterns(sentencePatterns, sentencePatternVariables, infoNode);
				} else if (childName == IF) {
					readConditions(conditionList, infoNode);
				} else if (childName == OVERLAP) {
					overlapFunction = readOverlapFunction(infoNode);
				} // end if-else for which node
			}
		}

		if (overlapFunction == null) {
			throw new ParseException("No overlap function specified.", -1);
		}

		SymbolTable variableData = new SymbolTable(variables);
		Rule rule =
				new Rule(serialized,
						sentencePatterns.toArray(new TreeNode[sentencePatterns.size()]),
						conditionList.toArray(new Condition[conditionList.size()]), variableData,
						sentencePatternVariables, overlapFunction, featureMan);
		return rule;
	}

	private static TreeNode readOverlapFunction(TreeNode infoNode) throws ParseException {
		TreeNode overlapFunction;
		if (infoNode.getChildren().size() != 1) {
			throw new ParseException("overlap node requires 1 list arguments: " + infoNode, -1);
		}

		overlapFunction = infoNode.getChildren().get(0);
		return overlapFunction;
	}

	private static void readConditions(ArrayList<Condition> conditionList, TreeNode infoNode)
			throws ParseException {
		if (infoNode.getChildren().size() != 2 || infoNode.getValues().size() != 2) {
			throw new ParseException(
					"if node requires 3 arguments (weight as float, condition list, then list): "
							+ infoNode, -1);
		}

		float weight = Float.parseFloat(infoNode.getValues().get(1));
		TreeNode conditionNode = infoNode.getChildren().get(0);
		TreeNode thenNode = infoNode.getChildren().get(1);

		// get implications
		ArrayList<Implication> implicationList = new ArrayList<Implication>();
		for (TreeNode implicationNode : thenNode.getChildren()) {
			if (implicationNode.getValues().size() != 3) {
				throw new ParseException(
						"then node requires 3 arguments (feature set, feature name, feature value): "
								+ implicationNode, -1);
			}
			assert implicationNode.getValues().get(0).equals(WALS);

			ArrayList<String> impValues = implicationNode.getValues();
			Implication imp = new Implication(impValues.get(0), impValues.get(1), impValues.get(2));
			implicationList.add(imp);
		}

		Condition condition =
				new Condition(weight, conditionNode,
						implicationList.toArray(new Implication[implicationList.size()]));
		conditionList.add(condition);
	}

	private static void readSentencePatterns(ArrayList<TreeNode> sentencePatterns,
			ArrayList<String> sentencePatternVariables, TreeNode infoNode) throws ParseException {
		for (TreeNode sentenceNode : infoNode.getChildren()) {

			if (sentenceNode.getValues().size() != 1)
				throw new ParseException("Sentence pattern requires name label: " + sentenceNode,
						-1);
			if (sentenceNode.getChildren().size() != 1)
				throw new ParseException("Sentence pattern must have one list argument: "
						+ sentenceNode, -1);

			// strip of the node with the variable name, then add
			sentencePatternVariables.add(sentenceNode.getValues().get(0));
			sentencePatterns.add(sentenceNode.getChildren().get(0));
		}
	}

	private static void readVariables(HashMap<String, Variable> variables, TreeNode infoNode)
			throws ParseException {

		for (TreeNode variableEntry : infoNode.getChildren()) {

			if (variableEntry.getValues().size() == 0 && variableEntry.getChildren().size() == 0) {

				// allow dummy place holder lists
				continue;
			}

			if (variableEntry.getValues().size() != 1)
				throw new ParseException("A variable must have exactly one name token: "
						+ variableEntry, -1);

			if (variableEntry.getChildren().size() != 1)
				throw new ParseException("A variable must have exactly one list of values: "
						+ variableEntry, -1);

			String variableName = variableEntry.getValues().get(0);
			TreeNode variableValues = variableEntry.getChildren().get(0);
			variables.put(variableName, new Variable(variableValues));
		}
	}

	public TreeNode getOverlapFunction() {
		return overlapFunction;
	}

	public SymbolTable getVariableData() {
		return variableData;
	}

	protected TreeNode[] getSentencePatterns() {
		return sentencePatterns;
	}

	protected HashMap<String, Integer> getSentencePatternVariables() {
		return sentenceVariablesByName;
	}

	/**
	 * Called for every Rule in the system when a new sentence pair is elicited.
	 * Returns true if some sentence pattern for this rule matches. If the pair
	 * matches, this Rule automatically keeps an internal record so that
	 * 
	 * @param pair
	 * @return
	 */
	public ArrayList<SentencePatternMatch> match(SentencePair pair, boolean matchLabeledSubtrees) {

		ArrayList<SentencePatternMatch> matches = new ArrayList<SentencePatternMatch>();
		SmartTree fStruct = pair.getFeatureStructure();

		// iterate over each pattern
		for (int i = 0; i < sentencePatterns.length; i++) {

			String sentenceVariable = sentenceVariablesByIndex.get(i);

			ArrayList<TreeNode> nodesToBeginSearch;
			if (matchLabeledSubtrees) {
				// OLD RESTRICTION: must match a role (labeled node) first
				// this should improve performance dramatically
				nodesToBeginSearch = fStruct.getLabeledNodes();
			} else {
				nodesToBeginSearch = toArrayList(fStruct.getRootNode());
			}

			for (TreeNode fNode : nodesToBeginSearch) {

				Variable[] actualVariableValues =
						TreeGrepper.matchesRecursively(sentencePatterns[i], fNode,
								this.variableData.variablePossibleValues, this.variableData);
				if (actualVariableValues != null) {

					// remember this match
					SentencePatternMatch match =
							new SentencePatternMatch(sentenceVariable, this, i, pair,
									actualVariableValues);
					ruleMatches[i].add(match);
					matches.add(match);

					// we already matched this pattern, so see if the same
					// sentence matches the other pattern too
					break;
				}
			}
		}

		return matches;
	}

	/**
	 * @return True if this rule has at least one match for each specified
	 *         sentence pattern.
	 */
	public boolean isReadyToEvaluate() {
		for (int i = 0; i < ruleMatches.length; i++)
			if (ruleMatches[i].size() < 1)
				return false;
		return true;
	}

	/**
	 * Returns a list of maching sentence pairs if all conditions of this rule
	 * hold for at least one pair. Otherwise, return a list of length zero. It
	 * may be desireable to call ComparisonResult.trimToSize() on each result if
	 * you intend to store many results in memory.
	 * <p>
	 * Note that this method automatically notifies the feature manager of any
	 * implications that fire (this is required for internal coherence of
	 * results). TODO: Take care of order of evaluating implcations.
	 * 
	 * @return
	 * @throws ParseException
	 * @throws CorpusException
	 */
	public ArrayList<ComparisonResult> evaluate() throws ParseException, CorpusException {

		ArrayList<ComparisonResult> allResults = new ArrayList<ComparisonResult>();

		for (final Condition condition : conditions) {
			ArrayList<ComparisonResult> results =
					BooleanFunctionEvaluator.evaluateBoolean(condition.getConditionNode(), this);

			// resolve implications
			for (final ComparisonResult result : results) {
				result.setImplications(condition.getImplications());
				result.setCondition(condition);
			}

			// notify feature manager of new value -- Rule should have no
			// knowledge of feauture manager
			// for (final Implication implication : condition.getImplications())
			// {
			// featureMan.setFeatureValue(implication.getFeatureName(),
			// implication.getFeatureValue());
			// }

			allResults.addAll(results);
		}

		return allResults;
	}

	/**
	 * Statistically disambiguates which condition, and therefore which
	 * implications are the best interpretation of an evaluation.
	 * 
	 * @param evaluationResults
	 * @return
	 */
	public Implication[] resolveFinalImplications(ArrayList<ComparisonResult> evaluationResults) {

		if (evaluationResults.size() == 0) {
			return new Implication[0];
		}

		// first, count occurrances of each implication
		HashMap<Condition, Integer> implicationCounts = new HashMap<Condition, Integer>();

		for (final ComparisonResult result : evaluationResults) {
			HashUtils.increment(implicationCounts, result.getCondition());
		}

		float maxWeight = Float.NEGATIVE_INFINITY;
		Condition maxCondition = null;

		// return the implication with the highest (weight*count)
		for (final Entry<Condition, Integer> entry : implicationCounts.entrySet()) {
			Condition implication = entry.getKey();

			int nCount = entry.getValue();
			float totalWeight = (float) nCount * implication.getWeight();

			if (totalWeight > maxWeight) {
				maxWeight = totalWeight;
				maxCondition = implication;
			}
		}

		return maxCondition.getImplications();
	}

	public static <T> ArrayList<T> toArrayList(T t) {
		ArrayList<T> list = new ArrayList<T>(1);
		list.add(t);
		return list;
	}

	public Condition[] getConditions() {
		return conditions;
	}

	public ArrayList<SentencePatternMatch> getRuleMatchesForVariable(String strSentenceVar)
			throws ParseException {

		Integer iSentenceIndex = sentenceVariablesByName.get(strSentenceVar);
		if (iSentenceIndex == null)
			throw new ParseException("Sentence variable not found for " + strSentenceVar, -1);
		int nSentenceIndex = iSentenceIndex;
		return this.ruleMatches[nSentenceIndex];
	}

	public String getRuleMatchSummary(boolean includeSentenceInstances) {
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < ruleMatches.length; i++) {
			builder.append("Sentence pattern " + sentenceVariablesByIndex.get(i) + " has "
					+ ruleMatches[i].size() + " matches.\n");
			if (includeSentenceInstances) {
				for (SentencePatternMatch match : ruleMatches[i]) {
					builder.append(match.getSentencePair().toString() + "\n");
				}
			}
		}
		return builder.toString();
	}

	public String toString() {
		return serialized;
	}
}
