/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import info.jonclark.util.StringUtils;

import java.util.ArrayList;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

public class LexicalResult implements Result {

	private ArrayList<SentencePatternMatch> patternMatch = new ArrayList<SentencePatternMatch>(1);

	// collect debug info while evaluating rules
	private ArrayList<ArrayList<TreeNode>> matches = new ArrayList<ArrayList<TreeNode>>();
	private ArrayList<TreeNode> patterns = new ArrayList<TreeNode>();
	private ArrayList<String> comments = new ArrayList<String>();
	private ArrayList<TreeNode> curResult;

	/**
	 * Reduces memory consumption. Call after this result has been fully
	 * populated. (Usually called by ComparisonResult).
	 */
	public void trimToSize() {
		for (ArrayList<TreeNode> list : matches) {
			list.trimToSize();
		}
		matches.trimToSize();
		patterns.trimToSize();
		comments.trimToSize();
		curResult.trimToSize();
	}

	public void setPatternMatch(SentencePatternMatch patternMatch) {
		this.patternMatch = Rule.toArrayList(patternMatch);
	}

	public SentencePatternMatch getPatternMatch() {
		return patternMatch.get(0);
	}

	public ArrayList<SentencePatternMatch> getPatternMatches() {
		return patternMatch;
	}

	public void addResult(ArrayList<TreeNode> match, TreeNode pattern, String comment) {
		curResult = match;
		matches.add(match);
		patterns.add(pattern);
		comments.add(comment);
	}

	public ArrayList<TreeNode> getCurrentResult() {
		return curResult;
	}

	public ArrayList<String> getCurrentResultLexicons() throws CorpusException {
		ArrayList<String> lexicons = new ArrayList<String>(curResult.size());
		for (TreeNode node : curResult) {
			if (node.getValues().size() != 2)
				throw new CorpusException("Expected 2 values: POS tag and lexical entry");
			String lex = node.getValues().get(1);
			lexicons.add(lex);
		}
		return lexicons;
	}

	public String getFullTrace() {
		StringBuilder builder = new StringBuilder();

		for (int i = 0; i < matches.size(); i++) {
			builder.append(comments.get(i) + ": ");
			builder.append(patterns.get(i) + " = ");
			builder.append(StringUtils.untokenize(matches.get(i), " :: ") + "\n");
		}

		return builder.toString();
	}

	public String getShortTrace() {
		return StringUtils.untokenize(curResult, " :: ");
	}

	public String getComments() {
		StringBuilder builder = new StringBuilder();
		for (String comment : comments) {
			builder.append(comment + ";");
		}
		return builder.toString();
	}

	public String toString() {
		return getShortTrace();
	}

	public void addComment(String comment) {
		comments.add(comment);
	}
}
