/**
* The AVENUE Project
* Language Technologies Institute
* School of Computer Science
* (c) 2007 Carnegie Mellon University
* 
* Corpus Navigator
* Written by Jonathan Clark
*/
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.text.ParseException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map.Entry;

/**
 * Tracks the state of a set of features, resolving conflicts/ambiguities when
 * multiple values are specified for the same feature type.
 */
public class FeatureManager {

	private final HashMap<String, String> actualFeatureValues;
	private final HashMap<String, HashSet<String>> allowedFeatureValues;

	public static final String NO_DATA = "NO_DATA";

	/**
	 * We do some error checking by requiring the names of the feature ahead of
	 * time
	 */
	public FeatureManager(String[] featureNames, HashMap<String, String[]> allowableFeatureValues) {

		assert featureNames.length == allowableFeatureValues.size() : "featureNames.length != featureValues.length";

		this.actualFeatureValues = new HashMap<String, String>(featureNames.length);
		this.allowedFeatureValues = new HashMap<String, HashSet<String>>(featureNames.length);

		for (Entry<String, String[]> entry : allowableFeatureValues.entrySet()) {

			String featureName = entry.getKey();

			HashSet<String> values = new HashSet<String>(entry.getValue().length);
			for (final String featureValue : entry.getValue()) {
				values.add(featureValue);
			}

			allowedFeatureValues.put(featureName, values);
			actualFeatureValues.put(featureName, NO_DATA);
		}
	}
	
	public static FeatureManager deserialize(File actualFeatureValuesFile, String[] featureNames,
			HashMap<String, String[]> allowableFeatureValues) throws IOException {
		
		FeatureManager man = new FeatureManager(featureNames, allowableFeatureValues);
		
		// load file
		BufferedReader in = new BufferedReader(new FileReader(actualFeatureValuesFile));

		String line;
		while ((line = in.readLine()) != null) {
			String[] tokens = StringUtils.split(line, "|||", 2);
			man.actualFeatureValues.put(tokens[0], tokens[1]);
		}

		in.close();
		
		return man;
	}

	public void setFeatureValue(String featureName, String featureValue) throws ParseException {
		if (!allowedFeatureValues.containsKey(featureName)) {
			throw new ParseException("Invalid Feature Name (not found in config file): \"" + featureName + "\"", -1);
		}

		if (!allowedFeatureValues.get(featureName).contains(featureValue)) {
			throw new ParseException("Invalid Feature Value (not found in config file): " + featureValue, -1);
		}

		actualFeatureValues.put(featureName, featureValue);
	}

	public String getFeatureValue(String featureName) throws ParseException {
		if (!allowedFeatureValues.containsKey(featureName)) {
			throw new ParseException("Invalid Feature Name (not found in config file): " + featureName, -1);
		}

		return actualFeatureValues.get(featureName);
	}

	public String toString() {
		StringBuilder builder = new StringBuilder();
		for (final Entry<String, String> entry : actualFeatureValues.entrySet()) {
			builder.append(entry.getKey() + "|||" + entry.getValue() + "\n");
		}
		return builder.toString();
	}
}



