/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import info.jonclark.properties.SmartProperties;
import info.jonclark.util.ArrayUtils;
import info.jonclark.util.FileUtils;
import info.jonclark.util.FormatUtils;
import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.PrintWriter;

import edu.cmu.cs.lti.avenue.navigation.wals.WALS;

/**
 * Evaluates the precision, recall, and F1 for the output of feature detection,
 * given a gold standard.
 */
public class FeatureDetectionEvaluator {

	public static void main(String[] args) throws Exception {

		if (args.length != 1) {
			System.err.println("Usage: program <properties_config_file>");
			System.exit(1);
		}

		File propsFile = new File(args[0]);
		SmartProperties props = new SmartProperties(propsFile);

		File actualResults = props.getPropertyFile("deduction.evaluation.resultsFile");
		File walsDir = props.getPropertyFile("global.walsDir");
		String language = props.getPropertyString("global.targetLanguage");
		File featureNamesToEvaluateFile =
				props.getPropertyFile("deduction.evaluation.featureNamesToEvaluateFile");
		File outputFile = props.getPropertyFile("deduction.evaluation.outputReport");

		String[] featureNamesToEvaluate =
				StringUtils.tokenize(FileUtils.getFileAsString(featureNamesToEvaluateFile), "\n");
		WALS wals = new WALS(walsDir);

		int languageIndex = wals.getLanguageIndex(language);
		assert languageIndex != -1 : "Language name not recognized: " + language;

		// do we need a calculator for each feature name with the classes being
		// the values?
		// or should we just calculate boolean good or no?

		int nTotal = 0;
		int nFiredCorrect = 0;
		int nBaselineCorrect = 0;
		int nFired = 0;

		PrintWriter out = new PrintWriter(outputFile);
		BufferedReader in = new BufferedReader(new FileReader(actualResults));

		String line;
		while ((line = in.readLine()) != null) {
			if (line.trim().equals(""))
				break;

			String featureName = StringUtils.substringBefore(line, "|||");
			String actualValue = StringUtils.substringAfter(line, "|||");

			if (ArrayUtils.unsortedArrayContains(featureNamesToEvaluate, featureName)) {
				// try {
				int featureIndex = wals.getFeatureIndex(featureName);
				String expectedValue = wals.getFeatureValueForLanguage(languageIndex, featureIndex);
				String baselineValue = wals.getMostCommonFeatureValue(featureIndex);

				System.out.println("FEATURE: " + featureName);
				System.out.println("EXPECT: " + expectedValue);

				if (!expectedValue.equals(FeatureManager.NO_DATA)) {

					if (baselineValue.equals(expectedValue)) {
						nBaselineCorrect++;
						System.out.println("BASELINE: " + baselineValue);
					} else {
						System.out.println("*BASELINE: " + baselineValue);
					}

					if (actualValue.equals(FeatureManager.NO_DATA)) {
						; // do nothing
						System.out.println("#ACTUAL: " + actualValue);
					} else if (actualValue.equals(expectedValue)) {
						nFiredCorrect++;
						nFired++;
						System.out.println("ACTUAL: " + actualValue);
					} else {
						nFired++;
						System.out.println("*ACTUAL: " + actualValue);
					}
					nTotal++;
				}
				System.out.println();
				// } catch (Throwable t) {
				// System.out.println("Could not evaluation feature: " +
				// featureName);
				// t.printStackTrace();
				// }
			}
		}

		// baseline p == r == f1
		double baseline = (double) nBaselineCorrect / (double) nTotal * 100;

		double p = (double) nFiredCorrect / (double) nFired * 100;
		double r = (double) nFiredCorrect / (double) nTotal * 100;
		double f1 = 2 * p * r / (p + r);

		double coverage = (double) nFired / (double) nTotal * 100;

		System.out.println("========================");
		System.out.println("COVERAGE: " + FormatUtils.formatDouble2(coverage) + "% (" + nFired + "/" + nTotal + ")");
		System.out.println("------------------------");
		System.out.println("BASELINE PREC: " + FormatUtils.formatDouble2(baseline) + "% (" + nBaselineCorrect + "/" + nTotal + ")");
		System.out.println("BASELINE RECALL: " + FormatUtils.formatDouble2(baseline) + "% (" + nBaselineCorrect + "/" + nTotal + ")");
		System.out.println("BASELINE F1: " + FormatUtils.formatDouble2(baseline) + "%");
		System.out.println("------------------------");
		System.out.println("PREC: " + FormatUtils.formatDouble2(p) + "% (" + nFiredCorrect + "/" + nFired + ")");
		System.out.println("RECALL: " + FormatUtils.formatDouble2(r) + "% (" + nFiredCorrect + "/" + nTotal + ")");
		System.out.println("F1: " + FormatUtils.formatDouble2(f1) + "%");
		System.out.println("========================");

		in.close();
		out.close();
	}
}
