/**
* The AVENUE Project
* Language Technologies Institute
* School of Computer Science
* (c) 2007 Carnegie Mellon University
* 
* Corpus Navigator
* Written by Jonathan Clark
*/
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.FNODE;
import info.jonclark.util.StringUtils;

import java.text.ParseException;
import java.util.ArrayList;

import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.TreeGrepper;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;
import edu.cmu.cs.lti.avenue.trees.smart.Variable;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelDisplay;

public class FNodeFunctionEvaluator {

	/**
	 * Resolves a sentence variable and f-structure pattern (possibly containing
	 * variables) to an fnode. This should probably be cached before evaluation
	 * 
	 * @param functionNode
	 * @return A TreeNode FROM THE FEATURE STRUCTURE, not from the rules.
	 * @throws ParseException
	 */
	protected static ArrayList<LexicalResult> evaluateFNode(TreeNode functionNode, Rule rule)
			throws ParseException {
		String functionName = functionNode.getValues().get(0);

		// make use of interned strings for == comparison
		assert functionName == functionName.intern() : "functionName not interned";

		if (functionName == FNODE) {

			if (functionNode.getChildren().size() != 1) {
				throw new ParseException(
						"fnode takes a single list describing the sentence as its only argument: "
								+ functionNode, -1);
			}

			TreeNode sentenceNode = functionNode.getChildren().get(0);

			if (sentenceNode.getValues().size() != 1) {
				throw new ParseException(
						"fnode takes one sentence variable as arg 1 in its argument list: "
								+ functionNode, -1);
			}
			if (sentenceNode.getChildren().size() > 1) {
				throw new ParseException(
						"fnode takes an optional list of pattern elements as arg 2: "
								+ functionNode, -1);
			}
			String strSentenceVar = sentenceNode.getValues().get(0);

			// now get the elicited examples that matched the pattern specified
			// for this sentence variable

			ArrayList<SentencePatternMatch> matchesForSentenceVariable = null;
			try {
				matchesForSentenceVariable = rule.getRuleMatchesForVariable(strSentenceVar);
			} catch (ParseException e) {
				throw new ParseException("Error evaluation fNode: " + functionNode + "\n"
						+ StringUtils.getStackTrace(e), -1);
			}

			ArrayList<LexicalResult> results = new ArrayList<LexicalResult>();

			if (sentenceNode.getChildren().size() == 0) {

				for (SentencePatternMatch sentencePatternMatch : matchesForSentenceVariable) {
					// no pattern was specified, so just add the root
					LexicalResult result = new LexicalResult();
					result.setPatternMatch(sentencePatternMatch);

					TreeNode root =
							sentencePatternMatch.getSentencePair().getFeatureStructure().getRootNode();
					result.addResult(Rule.toArrayList(root), functionNode, "evaluateFNode");
					results.add(result);
				}

			} else {
				TreeNode patternNode = sentenceNode.getChildren().get(0);

				for (SentencePatternMatch sentencePatternMatch : matchesForSentenceVariable) {

					// ruleMatch.getFnodeCache().getValue(key);

					// the following code was taken from the matches() method

					// RESTRICTION: must match a role (labeled node) first
					// this should improve performance dramatically
					SmartTree fStruct =
							sentencePatternMatch.getSentencePair().getFeatureStructure();
					ArrayList<TreeNode> labeledNodes = fStruct.getLabeledNodes();
					for (TreeNode fNode : labeledNodes) {

						// attempt to match the parent (the one carrying the
						// label we care about) of each "labeled" node
						if (fNode.getParentNode() != null) {

							Variable[] actualVariableValues =
									TreeGrepper.matchesRecursively(patternNode,
											fNode.getParentNode(),
											sentencePatternMatch.getVariableValues(),
											rule.getVariableData());
							if (actualVariableValues != null) {

								// add the child of the parent node (the labeled
								// node)
								
								// TODO: We need to add the specific node that matched
								// e.g. the child at depth n that matches ((actor ((possessor))))
								
								LexicalResult result = new LexicalResult();
								result.setPatternMatch(sentencePatternMatch);
								result.addResult(Rule.toArrayList(fNode), functionNode,
										"evaluateFNode");
								results.add(result);
							}
						}
					}
				} // for sentencePatternMatch

				if (results.size() == 0) {
					throw new ParseException("Invalid rule -- No matching node found for pattern: "
							+ patternNode.toString(LabelDisplay.NONE) + " within rule: "
							+ patternNode.getParentTree().toString(LabelDisplay.NONE), -1);
				}
			} // end if pattern node exists

			return results;

		} else {

			// this could also be the name of one of our sentences!

			// TODO: Give trace for file and line number
			throw new ParseException("Unrecognized function name: " + functionName, -1);
		}
	}

}

