/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SOURCE_LEX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SOURCE_LEX_IHEAD;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SOURCE_LEX_UHEAD;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.TARGET_LEX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.TARGET_LEX_IHEAD;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.TARGET_LEX_UHEAD;

import java.text.ParseException;
import java.util.ArrayList;

import edu.cmu.cs.lti.avenue.corpus.Alignment;
import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.corpus.PhiPlusMapping;
import edu.cmu.cs.lti.avenue.corpus.SentencePair;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelDisplay;

public class ConstituentFunctionEvaluator {

	/**
	 * Resolves a feature node to a lexical node (provides information on POS,
	 * position, and lexicon).
	 * 
	 * @param functionNode
	 * @return
	 * @throws ParseException
	 * @throws CorpusException
	 */
	protected static ArrayList<LexicalResult> evaluateConstituent(TreeNode functionNode, Rule rule)
			throws ParseException, CorpusException {

		String functionName = functionNode.getValues().get(0);

		// make use of interned strings for == comparison
		assert functionName == functionName.intern() : "functionName not interned";

		if (functionNode.getChildren().size() != 1) {
			throw new ParseException("constituent functions take 1 argument: "
					+ functionNode.toString(LabelDisplay.NONE), -1);
		}

		ArrayList<LexicalResult> results =
				FNodeFunctionEvaluator.evaluateFNode(functionNode.getChildren().get(0), rule);

		if (functionName == SOURCE_LEX) {

			resolveSourceCNodeFromFNode(results, functionNode);
			resolveSourceLexFromSourceCNode(results, functionNode);

		} else if (functionName == TARGET_LEX) {

			// find an alignment group from the sentence pair containing the
			// source index return the all target lexical items associated with
			// that index.

			resolveSourceCNodeFromFNode(results, functionNode);
			resolveSourceLexFromSourceCNode(results, functionNode);
			resolveTargetLexFromSourceLex(results, functionNode);

		} else if (functionName == SOURCE_LEX_IHEAD) {

			resolveSourceCNodeFromFNode(results, functionNode);
			resolveSourceIHeadFromSourceCNode(results, functionNode);
			resolveSourceLexFromSourceCNode(results, functionNode);

		} else if (functionName == TARGET_LEX_IHEAD) {

			resolveSourceCNodeFromFNode(results, functionNode);
			resolveSourceIHeadFromSourceCNode(results, functionNode);
			resolveSourceLexFromSourceCNode(results, functionNode);
			resolveTargetLexFromSourceLex(results, functionNode);

		} else if (functionName == SOURCE_LEX_UHEAD) {

			resolveSourceCNodeFromFNode(results, functionNode);
			resolveSourceUHeadFromSourceCNode(results, functionNode);
			resolveSourceLexFromSourceCNode(results, functionNode);

		} else if (functionName == TARGET_LEX_UHEAD) {

			// then run the target lex algorithm
			resolveSourceCNodeFromFNode(results, functionNode);
			resolveSourceUHeadFromSourceCNode(results, functionNode);
			resolveSourceLexFromSourceCNode(results, functionNode);
			resolveTargetLexFromSourceLex(results, functionNode);

		} else {
			// TODO: Give trace for file and line number
			throw new ParseException("Unrecognized boolean function name: " + functionName, -1);
		}

		return results;
	}

	/**
	 * Resolves the immediate head of each source lexicon in the list
	 * individually *
	 * 
	 * @param results
	 */
	protected static void resolveSourceIHeadFromSourceCNode(ArrayList<LexicalResult> results,
			TreeNode functionNode) {

		for (final LexicalResult result : results) {
			SentencePair pair = result.getPatternMatch().getSentencePair();
			PhiPlusMapping headMapping = pair.getPhiPlusMapping();
			ArrayList<TreeNode> iHeadCNodes = new ArrayList<TreeNode>();

			for (final TreeNode sourceCNode : result.getCurrentResult()) {
				iHeadCNodes.add(headMapping.getImmediateHead(pair, sourceCNode));
			}

			// swap source lexicons for target lexicons
			// and store target lexicons as debug info
			result.addResult(iHeadCNodes, functionNode, "resolveSourceIHeadFromSourceCNode");
		}
	}

	/**
	 * Resolves the ultimate head of each source lexicon in the list
	 * individually
	 * 
	 * @param results
	 * @throws CorpusException
	 */
	protected static void resolveSourceUHeadFromSourceCNode(ArrayList<LexicalResult> results,
			TreeNode functionNode) throws CorpusException {

		for (final LexicalResult result : results) {
			SentencePair pair = result.getPatternMatch().getSentencePair();
			PhiPlusMapping headMapping = pair.getPhiPlusMapping();
			ArrayList<TreeNode> uHeadCNodes = new ArrayList<TreeNode>();

			for (final TreeNode origCNode : result.getCurrentResult()) {
				uHeadCNodes.add(headMapping.getUltimateHead(pair, origCNode));
			}

			// swap source lexicons for target lexicons
			// and store target lexicons as debug info
			result.addResult(uHeadCNodes, functionNode, "resolveSourceUHeadFromSourceCNode");
		}
	}

	protected static void resolveTargetLexFromSourceLex(ArrayList<LexicalResult> results,
			TreeNode functionNode) throws CorpusException {

		for (final LexicalResult result : results) {
			SentencePair pair = result.getPatternMatch().getSentencePair();
			Alignment alignment = pair.getNormalizedAlignment();
			ArrayList<TreeNode> targetLexicons = new ArrayList<TreeNode>();

			
			// NOTE: Changed to normalized from display... Did this break anything?
			
			for (final TreeNode sourceLex : result.getCurrentResult()) {
				targetLexicons.addAll(alignment.getTargetNodes(pair.getNormalizedTargetTokens(),
						sourceLex));
			}

			// swap source lexicons for target lexicons
			// and store target lexicons as debug info
			result.addResult(targetLexicons, functionNode, "resolveTargetLexFromSourceLex");
		}
	}

	/**
	 * Resolves the terminal source lexicons from a list of fnodes
	 * @throws CorpusException 
	 */
	protected static void resolveSourceCNodeFromFNode(ArrayList<LexicalResult> results,
			TreeNode functionNode) throws CorpusException {
		// get the c-node associated with the f-node, then return the
		// lexical terminals
		for (final LexicalResult result : results) {
			PhiPlusMapping phiMapping =
					result.getPatternMatch().getSentencePair().getPhiPlusMapping();

			ArrayList<TreeNode> fNodeList = result.getCurrentResult();
			assert fNodeList.size() == 1 : "There should be exactly one f-node: "
					+ fNodeList.toString();
			TreeNode fNode = fNodeList.get(0);

			assert phiMapping != null : "null phiMapping";
			assert result != null : "null result";
			assert result.getPatternMatch() != null : "null patternMatch";

			TreeNode cNode =
					phiMapping.getPhiInverseTop(result.getPatternMatch().getSentencePair(), fNode);

			result.addResult(Rule.toArrayList(cNode), functionNode, "resolveSourceCNodeFromFNode");
		}
	}

	/**
	 * Resolves the terminal source lexicons from a list of fnodes
	 */
	protected static void resolveSourceLexFromSourceCNode(ArrayList<LexicalResult> results,
			TreeNode functionNode) {
		// get the c-node associated with the f-node, then return the
		// lexical terminals
		for (final LexicalResult result : results) {
			ArrayList<TreeNode> cNodes = result.getCurrentResult();
			ArrayList<TreeNode> sourceLex = new ArrayList<TreeNode>();

			for (final TreeNode cNode : cNodes) {
				sourceLex.addAll(cNode.getLexicalTerminals());
			}

			result.addResult(sourceLex, functionNode, "resolveSourceLexFromSourceCNode");
		}
	}

}
