/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import info.jonclark.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;

import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelDisplay;

public class ComparisonResult implements Result, Cloneable {

	private ArrayList<SentencePatternMatch> sentencePatternMatches =
			new ArrayList<SentencePatternMatch>();

	private Condition condition;
	private TreeNode conditionNode;
	private ArrayList<Result> operands;
	private ArrayList<String> comments = new ArrayList<String>();

	private Implication[] implications; // RHS "then"
	private boolean onlyCopy = true;

	public ComparisonResult(int nOperands) {
		this.operands = new ArrayList<Result>(nOperands);
	}

	public ArrayList<SentencePatternMatch> getPatternMatches() {
		return sentencePatternMatches;
	}

	protected void setConditionNode(TreeNode conditionNode) {
		this.conditionNode = conditionNode;
	}

	protected void setCondition(Condition condition) {
		this.condition = condition;
	}

	public Condition getCondition() {
		return condition;
	}

	protected void addOperand(Result operand) {
		this.operands.add(operand);
		sentencePatternMatches.addAll(operand.getPatternMatches());
	}

	protected void setImplications(Implication[] implications) {
		this.implications = implications;
	}

	public TreeNode getConditionNode() {
		return conditionNode;
	}

	public ArrayList<Result> getOperands() {
		return operands;
	}

	public Implication[] getImplications() {
		return implications;
	}

	/**
	 * Same behavior as clone except that the first call returns the existing
	 * object;
	 */
	public ComparisonResult uniqueCopy() {
		if (onlyCopy) {
			return this;
		} else {
			onlyCopy = false;
			try {
				return (ComparisonResult) super.clone();
			} catch (CloneNotSupportedException e) {
				throw new Error(e);
			}
		}
	}

	/**
	 * Gives a set of lexicons that were used as evidence for this implication.
	 * 
	 * @return
	 */
	public String getFullTrace() {
		StringBuilder builder = new StringBuilder();

		assert conditionNode != null : "Null condition detected for ComparisonResult: "
				+ this.toString();
		builder.append("TRACE OF " + getComments() + " for " + conditionNode.toString(LabelDisplay.NONE));

		int n = 1;
		for (final Result result : operands) {
			builder.append("\n" + "OP #" + n + ":\n");
			for (SentencePatternMatch match : result.getPatternMatches()) {
				builder.append(match.getSentencePair().serialize() + "\n");
			}
			builder.append(result.getFullTrace());
			n++;
		}

		return builder.toString();
	}

	public String getShortTrace() {
		StringBuilder builder = new StringBuilder();

		assert conditionNode != null : "Null condition detected for ComparisonResult: "
				+ this.toString();
		for (SentencePatternMatch match : getPatternMatches()) {
			builder.append(StringUtils.untokenize(match.getSentencePair().getNormalizedSourceTokens())
					+ " / "
					+ StringUtils.untokenize(match.getSentencePair().getNormalizedTargetTokens())
					+ "\n");
		}
		// builder.append(condition.toString(false) + ":\n");
		// builder.append("comments: " + getComments() + "\n");

		for (final Result result : operands) {
			builder.append(result.getShortTrace() + " :: ");
		}

		return builder.toString();
	}

	public String toString() {
		return Arrays.toString(implications);
	}

	/**
	 * Reduces memory consumption. Call after this result has been fully
	 * populated.
	 */
	public void trimToSize() {
		operands.trimToSize();
		for (final Result op : operands) {
			op.trimToSize();
		}
	}

	public String getComments() {
		StringBuilder builder = new StringBuilder();
		for (String comment : comments) {
			builder.append(comment + ";");
		}
		for (Result result : operands) {
			builder.append(result.getComments());
		}
		return builder.toString();
	}

	public void addComment(String comment) {
		comments.add(comment);
	}

}
