/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive;

import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.AND;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.ASSERTED;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.DIFFERENT;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.DIFFERENT_AFFIX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.DIFFERENT_PREFIX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.DIFFERENT_SUFFIX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.EQUALS;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.GREATER;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.IN_ORDER;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.LESS;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.NOT_ASSERTED;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.NOT_PRESENT;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.OR;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.PRESENT;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SAME;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SAME_AFFIX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SAME_PREFIX;
import static edu.cmu.cs.lti.avenue.navigation.featuredetection.deductive.RuleConstants.SAME_SUFFIX;
import info.jonclark.util.DebugUtils;
import info.jonclark.util.StringUtils;

import java.text.ParseException;
import java.util.ArrayList;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree.LabelDisplay;

public class BooleanFunctionEvaluator {

	public static int VERBOSITY = 3;

	protected static ArrayList<ComparisonResult> evaluateBoolean(TreeNode condition, Rule rule)
			throws ParseException, CorpusException {

		if (VERBOSITY == 3) {
			System.out.println("BOOL: Evaluating: " + condition.toString());
		}

		String functionName = condition.getValues().get(0);

		// make use of interned strings for == comparison
		assert functionName == functionName.intern() : "functionName not interned";

		if (condition.getChildren().size() < 1) {
			// TODO: Provide trace and line number
			throw new ParseException("Invalid number of arguments for function " + functionName
					+ ": " + condition, -1);
		}

		if (functionName == DIFFERENT || functionName == DIFFERENT_SUFFIX
				|| functionName == DIFFERENT_PREFIX || functionName == DIFFERENT_AFFIX
				|| functionName == SAME || functionName == SAME_SUFFIX
				|| functionName == SAME_PREFIX || functionName == SAME_AFFIX) {

			return evaluateDifferent(condition, functionName, rule);

		} else if (functionName == IN_ORDER) {
			return evaluateInOrder(condition, rule);

		} else if (functionName == PRESENT || functionName == NOT_PRESENT) {
			return evaluatePresent(condition, functionName, rule);

		} else if (functionName == EQUALS || functionName == GREATER || functionName == LESS) {
			return evaluateNumeric(condition, functionName, rule);

		} else if (functionName == ASSERTED || functionName == NOT_ASSERTED) {
			return evaluateAsserted(condition, functionName, rule);

		} else if (functionName == AND) {
			return evaluateAnd(condition, rule);

		} else if (functionName == OR) {
			return evaluateOr(condition, rule);

		} else {
			// TODO: Give trace for file and line number
			throw new ParseException("Unrecognized boolean function name: " + functionName, -1);
		}
	}

	protected static ArrayList<ComparisonResult> evaluateOr(TreeNode condition, Rule rule)
			throws ParseException, CorpusException {

		ArrayList<TreeNode> childConditions = condition.getChildren();
		ArrayList<ComparisonResult>[] childResults = createArray2(childConditions.size());
		ArrayList<ComparisonResult> orResults = new ArrayList<ComparisonResult>();

		ComparisonResult orResult = new ComparisonResult(childResults.length);

		boolean gotOne = false;

		// first solve all child results
		for (int i = 0; i < childResults.length; i++) {
			childResults[i] =
					BooleanFunctionEvaluator.evaluateBoolean(childConditions.get(i), rule);

			// if there were no results, this function will not evaluate to true
			if (childResults[i].size() == 0) {
				; // wait for another match
			} else {

				// group each set of results into a result just for cleanliness
				ComparisonResult andGroup = new ComparisonResult(childResults[i].size());
				andGroup.addComment("andGroup");
				andGroup.setConditionNode(condition);
				for (final ComparisonResult child : childResults[i])
					orResult.addOperand(child);
				orResult.addOperand(andGroup);
				gotOne = true;
			}
		}

		if (!gotOne) {
			assert orResults.size() == 0 : "Expected to return false with zero-length arraylist";
			return orResults;
		}

		// if we got here, the function was successfully evaluated
		orResult.setConditionNode(condition);
		orResult.addComment("and");
		orResults.add(orResult);
		return orResults;

	}

	protected static ArrayList<ComparisonResult> evaluateAnd(TreeNode condition, Rule rule)
			throws ParseException, CorpusException {

		ArrayList<TreeNode> childConditions = condition.getChildren();
		ArrayList<ComparisonResult>[] childResults = createArray2(childConditions.size());
		ArrayList<ComparisonResult> andResults = new ArrayList<ComparisonResult>();

		ComparisonResult andResult = new ComparisonResult(childResults.length);

		// first solve all child results
		for (int i = 0; i < childResults.length; i++) {
			childResults[i] =
					BooleanFunctionEvaluator.evaluateBoolean(childConditions.get(i), rule);

			// if there were no results, this function will not evaluate to true
			if (childResults[i].size() == 0) {
				assert andResults.size() == 0 : "Expected to return false with zero-length arraylist";
				return andResults;
			} else {

				// group each set of results into a result just for cleanliness
				ComparisonResult andGroup = new ComparisonResult(childResults[i].size());
				andGroup.addComment("andGroup");
				andGroup.setConditionNode(condition);
				for (final ComparisonResult child : childResults[i])
					andResult.addOperand(child);
				andResult.addOperand(andGroup);
			}
		}

		// if we got here, the function was successfully evaluated
		andResult.setConditionNode(condition);
		andResult.addComment("and");
		andResults.add(andResult);
		return andResults;

		// We have to check for correct usage of variables here
		// for all possible combinations of the andResults
		// (This is basically a confusion network search)
		//
		// The proposedResult is a volatile array that will, over the
		// course of its life, contain all possible combinations of
		// andResults
		// ComparisonResult[] proposedResult = new
		// ComparisonResult[childResults.length];
		// evaluateAndRecursively(andResults, rule, 0, childResults,
		// childConditions, condition,
		// proposedResult);
		// return andResults;
	}

	static long xx = 0;

	protected static void evaluateAndRecursively(ArrayList<ComparisonResult> andResults, Rule rule,
			int nResult, ArrayList<ComparisonResult>[] childResults,
			ArrayList<TreeNode> childConditions, TreeNode condition,
			ComparisonResult[] proposedResult) throws ParseException {

		xx++;
		if (xx % 1000000 == 0)
			System.out.println("Evaluated " + xx + " calls to \"and\"...");

		// if (andResults.size() > 0) {
		// // only find one result: short circuit AND
		// return;
		// } else
		if (nResult == childResults.length) {
			if (OverlapEvaluator.isProperPair(rule, true, proposedResult)) {
				// this is a keeper
				ComparisonResult andResult = new ComparisonResult(proposedResult.length);
				andResult.addComment("and");
				andResult.setConditionNode(condition);
				for (ComparisonResult operand : proposedResult)
					andResult.addOperand(operand);
				andResults.add(andResult);
			}
		} else {

			// preemptively prune bad variable usage
			// if (OverlapEvaluator.isProperPair(rule, false, proposedResult)) {
			for (int i = 0; i < childResults[nResult].size(); i++) {
				proposedResult[nResult] = childResults[nResult].get(i);
				evaluateAndRecursively(andResults, rule, nResult + 1, childResults,
						childConditions, condition, proposedResult);
			}
			proposedResult[nResult] = null;
			// }
		}
	}

	protected static ArrayList<ComparisonResult> evaluatePresent(TreeNode condition,
			String functionName, Rule rule) throws ParseException, CorpusException {

		if (condition.getChildren().size() != 1) {
			throw new ParseException(
					"function " + functionName + " takes 1 argument: " + condition, -1);
		}

		ArrayList<LexicalResult> lexResults =
				ConstituentFunctionEvaluator.evaluateConstituent(condition.getChildren().get(0),
						rule);
		ArrayList<ComparisonResult> compResults = new ArrayList<ComparisonResult>();
		for (final LexicalResult lexResult : lexResults) {
			if (functionName == PRESENT && lexResult.getCurrentResult().size() > 0) {
				ComparisonResult c = new ComparisonResult(1);
				c.addOperand(lexResult);
				c.setConditionNode(condition);
				c.addComment(functionName);
				compResults.add(c);
			} else if (functionName == NOT_PRESENT && lexResult.getCurrentResult().size() == 0) {
				ComparisonResult c = new ComparisonResult(1);
				c.addOperand(lexResult);
				c.setConditionNode(condition);
				c.addComment(functionName);
				compResults.add(c);
			}
		}

		return compResults;
	}

	/**
	 * Hack to get around Java's generic array deficiency.
	 */
	@SuppressWarnings("unchecked")
	private static ArrayList<ComparisonResult>[] createArray2(int nSize) {
		return (ArrayList<ComparisonResult>[]) new ArrayList[nSize];
	}

	protected static ArrayList<ComparisonResult> evaluateInOrder(TreeNode condition, Rule rule)
			throws ParseException, CorpusException {

		if (condition.getChildren().size() < 2) {
			throw new ParseException("in-order requires at least 2 arguments: "
					+ condition.toString(LabelDisplay.NONE), -1);
		}

		ArrayList<ComparisonResult> hypotheses = new ArrayList<ComparisonResult>();

		// begin our hypotheses vector for results by populating it with the
		// first position
		ArrayList<LexicalResult> firstPosition =
				ConstituentFunctionEvaluator.evaluateConstituent(condition.getChildren().get(0),
						rule);
		for (LexicalResult lexicalResult : firstPosition) {
			ComparisonResult comparisonResult = new ComparisonResult(1);
			comparisonResult.addOperand(lexicalResult);
			comparisonResult.setConditionNode(condition);
			comparisonResult.addComment("in-order");
			hypotheses.add(comparisonResult);
		}

		// TODO: This method could be made more efficient if we constrain our
		// search to like variables pre-emptively rather than filtering out bad
		// variable usage as we go

		// iterate over each position, branching some hypotheses and
		// removing implausible hypotheses
		for (int iPosition = 1; iPosition < condition.getChildren().size(); iPosition++) {

			ArrayList<LexicalResult> nextPositions =
					ConstituentFunctionEvaluator.evaluateConstituent(condition.getChildren().get(
							iPosition), rule);

			// the number of hypotheses can change after each positional
			// iteration
			ArrayList<ComparisonResult> survivingHypotheses =
					new ArrayList<ComparisonResult>(hypotheses.size());

			// try all combinations of hypotheses with the next set of positions
			for (int j = 0; j < hypotheses.size(); j++) {

				// the following is mostly a lot of syntactic garbage to do
				// something simple: make sure the position of the last word
				// from the previous match is actually less than the first word
				// from the current match

				// grab a hypothesis and the associated operands for each
				// iPosition
				ComparisonResult hypothesis = hypotheses.get(j);
				ArrayList<Result> lexicalOperands = hypothesis.getOperands();

				// get the position of the lexicon with the maximum position
				assert lexicalOperands.get(iPosition - 1) instanceof LexicalResult : "Expected lexical result";
				LexicalResult prevPosition = (LexicalResult) lexicalOperands.get(iPosition - 1);
				ArrayList<TreeNode> prevLexicons = prevPosition.getCurrentResult();
				if(prevLexicons.size() == 0) {
					if (VERBOSITY == 3) {
						System.out.println("BOOL: Trying []. No lexical items found. Skipping entry.");
					}
					continue;
				}
				TreeNode prevLexicon = prevLexicons.get(prevLexicons.size() - 1);

				// this could contain multiple indices if this is a phrase (e.g.
				// NP)
				ArrayList<Integer> nPrevPositions = prevLexicon.getTerminalIndices();
				int nPrevPosition = nPrevPositions.get(nPrevPositions.size() - 1);

				for (LexicalResult nextPosition : nextPositions) {

					if (VERBOSITY == 3) {
						System.out.println("BOOL: Trying PREV from Sent#"
								+ prevPosition.getPatternMatch().getSentencePair().getId() + ": "
								+ prevPosition.toString() + " WITH NEXT from Sent#"
								+ nextPosition.getPatternMatch().getSentencePair().getId() + ": "
								+ nextPosition.toString());
					}

					// make sure we satisfy variable conditions
					if (!OverlapEvaluator.isProperPair(rule, true, prevPosition, nextPosition)) {
						if (VERBOSITY == 3) {
							System.out.println("BOOL: Rejected hypothesis due to sentence overlap.");
						}
						continue;
					}

					// grab the position of the first lexicon in the next
					// position
					ArrayList<TreeNode> nextLexicons = nextPosition.getCurrentResult();
					if(nextLexicons.size() == 0) {
						if (VERBOSITY == 3) {
							System.out.println("BOOL: No lexical items found for nextPosition. Skipping entry.");
						}
						continue;
					}
					TreeNode nextLexicon = nextLexicons.get(nextLexicons.size() - 1);

					// this could contain multiple indices if this is a phrase
					// (e.g. NP)
					ArrayList<Integer> nNextPositions = nextLexicon.getTerminalIndices();
					int nNextPosition = nNextPositions.get(nNextPositions.size() - 1);

					if(nPrevPosition == nNextPosition) {
						System.out.println("BOOL: NOTIFICATION: Two lexical items claim "
							+ "to be in the same position ("
							+ nPrevPosition
							+ "): "
							+ prevLexicon
							+ " AND "
							+ nextLexicon
							+ "\nTRACE OP #1:\n"
							+ prevPosition.getFullTrace()
							+ "\nTRACE OP #2:\n"
							+ nextPosition.getFullTrace()
							+ "\nFor c-struct: "
							+ nextPosition.getPatternMatch().getSentencePair().getSourceConstituentStructure()
							+ "\nWhile evaluating: " + condition.toString(LabelDisplay.NONE));
					}
					
					if (nPrevPosition > nNextPosition) {
						// this hypothesis is not compatible with this position,
						; // so do nothing
						if (VERBOSITY == 3) {
							System.out.println("BOOL: Rejected hypothesis due to wrong ordering.");
						}
					} else {
						if (VERBOSITY == 3) {
							System.out.println("BOOL: Hypothesis survived.");
						}
						// we found a match for this position
						ComparisonResult survivingHypothesis = hypothesis.uniqueCopy();
						survivingHypothesis.addOperand(nextPosition);
						survivingHypotheses.add(survivingHypothesis);
					}
				}
			} // for each hypothesis
			hypotheses = survivingHypotheses;
		} // for each ordering position

		if (DebugUtils.isAssertEnabled()) {
			for (int i = 0; i < hypotheses.size(); i++) {
				assert hypotheses.get(i).getConditionNode() != null : "Null condition detected for result at index "
						+ i;
			}
		}

		// all remaining hypotheses are true
		return hypotheses;
	}

	protected static ArrayList<ComparisonResult> evaluateDifferent(TreeNode condition,
			String functionName, Rule rule) throws ParseException, CorpusException {

		if (condition.getChildren().size() != 2) {
			throw new ParseException("function " + functionName + " takes 2 arguments: "
					+ condition, -1);
		}

		// first resolve which two constituents we're comparing
		ArrayList<LexicalResult> resultsA =
				ConstituentFunctionEvaluator.evaluateConstituent(condition.getChildren().get(0),
						rule);
		ArrayList<LexicalResult> resultsB =
				ConstituentFunctionEvaluator.evaluateConstituent(condition.getChildren().get(1),
						rule);

		ArrayList<ComparisonResult> mergedResults =
				new ArrayList<ComparisonResult>(resultsA.size() * resultsB.size());

		// take the cross product of the results
		for (final LexicalResult resultA : resultsA) {
			for (final LexicalResult resultB : resultsB) {

				// never compare a sentence to itself if the sentences are
				// represented by different variables
				if (!OverlapEvaluator.isProperPair(rule, true, resultA, resultB)) {
					continue;
				}

				ArrayList<TreeNode> lexiconsA = resultA.getCurrentResult();
				ArrayList<TreeNode> lexiconsB = resultB.getCurrentResult();

				// first see if we even have the same number of tokens
				// (lexicons)
				if (lexiconsA.size() != lexiconsB.size()) {

					// we don't have the same number of tokens
					if (functionName == DIFFERENT) {

						// yes! we can say they are different
						ComparisonResult comparisonResult = new ComparisonResult(2);
						comparisonResult.addOperand(resultA);
						comparisonResult.addOperand(resultB);
						comparisonResult.setConditionNode(condition);
						comparisonResult.addComment(functionName);
						mergedResults.add(comparisonResult);

					} else if (functionName == DIFFERENT_SUFFIX || functionName == DIFFERENT_PREFIX
							|| functionName == DIFFERENT_AFFIX) {
						// we do not know how to properly match the words in
						// this case
						; // add nothing
					} else if (functionName == SAME || functionName == SAME_SUFFIX
							|| functionName == SAME_PREFIX || functionName == SAME_AFFIX) {
						// these arguments cannot be the same
						; // add nothing
					} else {
						throw new ParseException("Unrecognized difference function name: "
								+ functionName, -1);
					}

				} else {

					boolean isSatisfied = true;

					// we have the same number of lexicons
					for (int i = 0; i < lexiconsA.size(); i++) {
						ArrayList<String> valuesA = lexiconsA.get(i).getValues();
						ArrayList<String> valuesB = lexiconsB.get(i).getValues();

						if (valuesA.size() != 2)
							throw new CorpusException(
									"Expected 2 values: POS tag and lexical entry");
						if (valuesB.size() != 2)
							throw new CorpusException(
									"Expected 2 values: POS tag and lexical entry");

						String lexA = valuesA.get(1);
						String lexB = valuesB.get(1);

						assert lexA == lexA.intern() : "lexA not interned: " + lexA;
						assert lexB == lexB.intern() : "lexB not interned: " + lexB;

						assert lexA.length() > 0 : "Zero length words are not allowed";
						assert lexB.length() > 0 : "Zero length words are not allowed";

						char firstCharA = lexA.charAt(0);
						char lastCharA = lexA.charAt(lexA.length() - 1);
						char firstCharB = lexB.charAt(0);
						char lastCharB = lexB.charAt(lexB.length() - 1);

						if (functionName == DIFFERENT) {
							isSatisfied = (lexA != lexB);
						} else if (functionName == DIFFERENT_SUFFIX) {
							isSatisfied = (firstCharA == firstCharB && lastCharA != lastCharB);
						} else if (functionName == DIFFERENT_PREFIX) {
							isSatisfied = (firstCharA != firstCharB && lastCharA == lastCharB);
						} else if (functionName == DIFFERENT_AFFIX) {
							boolean hasDifferentSuffix =
									(firstCharA == firstCharB && lastCharA != lastCharB);
							boolean hasDifferentPrefix =
									(firstCharA != firstCharB && lastCharA == lastCharB);
							isSatisfied = (hasDifferentSuffix || hasDifferentPrefix);
						} else if (functionName == SAME) {
							isSatisfied = (lexA == lexB && lexA.length() > 0);
						} else if (functionName == SAME_PREFIX) {
							isSatisfied = (firstCharA == firstCharB);
						} else if (functionName == SAME_SUFFIX) {
							isSatisfied = (lastCharA == lastCharB);
						} else if (functionName == SAME_AFFIX) {
							boolean hasSamePrefix = (firstCharA == firstCharB);
							boolean hasSameSuffix = (lastCharA == lastCharB);
							isSatisfied = (hasSamePrefix || hasSameSuffix);
						} else {
							throw new ParseException("Unrecognized difference function name: "
									+ functionName, -1);
						}

						if (!isSatisfied) {
							break;
						}

					} // for i in lexiconsA/B

					if (isSatisfied) {
						// yes! we can say they are different
						ComparisonResult comparisonResult = new ComparisonResult(2);
						comparisonResult.addOperand(resultA);
						comparisonResult.addOperand(resultB);
						comparisonResult.setConditionNode(condition);
						comparisonResult.addComment(functionName);
						mergedResults.add(comparisonResult);
					} else {
						// add nothing
					}

				} // if lexicons.length same

			} // for resultB
		} // for resultA

		if (DebugUtils.isAssertEnabled()) {
			for (int i = 0; i < mergedResults.size(); i++) {
				assert mergedResults.get(i).getConditionNode() != null : "Null condition detected for result at index "
						+ i;
			}
		}

		return mergedResults;
	}

	/**
	 * Evaluates numeric comparisons such as >, <, =
	 */
	protected static ArrayList<ComparisonResult> evaluateNumeric(TreeNode condition,
			String functionName, Rule rule) throws ParseException, CorpusException {

		if (condition.getChildren().size() != 1 && condition.getValues().size() != -1) {
			throw new ParseException("function " + functionName
					+ " takes 2 arguments (required integer, numeric function): " + condition, -1);
		}

		int nValue = Integer.parseInt(condition.getValues().get(1));

		ArrayList<ComparisonResult> accepted = new ArrayList<ComparisonResult>();
		ArrayList<NumericResult> numResults =
				CountFunctionEvaluator.evaluateCount(condition.getChildren().get(0), rule);

		for (final NumericResult numResult : numResults) {

			if ((functionName == EQUALS && numResult.getCountValue() == nValue)
					|| (functionName == LESS && numResult.getCountValue() < nValue)
					|| (functionName == GREATER && numResult.getCountValue() > nValue)) {

				ComparisonResult c = new ComparisonResult(1);
				c.addOperand(numResult);
				c.setConditionNode(condition);
				c.addComment(functionName);
				accepted.add(c);
			}
		}

		return accepted;
	}

	protected static ArrayList<ComparisonResult> evaluateAsserted(TreeNode condition,
			String functionName, Rule rule) throws ParseException {

		if (true)
			throw new Error("Deprecated function: " + functionName);

		TreeNode featureNode = condition.getChildren().get(0);

		FeatureManager featureMan = rule.getFeatureManager();
		// String featureSetName = featureNode.getValues().get(0);
		String featureName = featureNode.getValues().get(1);
		String desiredFeatureValue = featureNode.getValues().get(2);

		// strip quotes, if any
		featureName = StringUtils.substringBetween(featureName, "\"", "\"");
		desiredFeatureValue = StringUtils.substringBetween(desiredFeatureValue, "\"", "\"");

		String currentFeatureValue = featureMan.getFeatureValue(featureName);

		if (functionName == ASSERTED) {
			if (currentFeatureValue.equals(desiredFeatureValue)) {
				ComparisonResult result = new ComparisonResult(0);
				result.addComment(functionName + ": " + currentFeatureValue);
				result.setConditionNode(condition);
				return Rule.toArrayList(result);
			} else {
				return new ArrayList<ComparisonResult>(0);
			}
		} else if (functionName == NOT_ASSERTED) {
			if (!currentFeatureValue.equals(desiredFeatureValue)) {
				ComparisonResult result = new ComparisonResult(0);
				result.addComment(functionName + ": " + currentFeatureValue);
				result.setConditionNode(condition);
				return Rule.toArrayList(result);
			} else {
				return new ArrayList<ComparisonResult>(0);
			}
		} else {
			throw new ParseException("Unrecognized asserted function name: " + functionName, -1);
		}
	}
}
