package edu.cmu.cs.lti.avenue.featurespecification;

import info.jonclark.log.LogUtils;
import info.jonclark.util.StringUtils;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import edu.cmu.cs.lti.avenue.corpus.CorpusException;
import edu.cmu.cs.lti.avenue.navigation.search.generation1.tables.ConfigurationException;
import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

/**
 * Tracks the allowable values and relationships in the feature set and can
 * extract specified portions of feature structure trees accordingly.
 */
public class FeatureStructureManager {

	public enum Type {
		CLAUSES, PARTICIPANTS, FEATURES, CONJUNCTIONS, MODIFIERS
	};

	private ArrayList<Section> sections = new ArrayList<Section>();
	private HashMap<String, FeatureSpec> featureSpecs = new HashMap<String, FeatureSpec>();

	private HashSet<String> featureNames = new HashSet<String>();
	private HashSet<String> clauseTypes = new HashSet<String>();
	private HashSet<String> participantTypes = new HashSet<String>();
	private HashSet<String> conjunctionSet = new HashSet<String>();
	private HashSet<String> modifierSet = new HashSet<String>();
	private HashMap<String, FeatureContext> featureContexts = null;

	private ArrayList<String> clauseList;
	private ArrayList<String> participantList;
	private ArrayList<String> conjunctionList = new ArrayList<String>();
	private ArrayList<String> modifierList = new ArrayList<String>();

	public static final String CONJUNCTION_TYPE = "conj";
	public static final String MODIFIER_TYPE = "mod";

	private int nFeatureSpecs = 0;
	private int nClauseSpecs = 0;
	private int nNounSpecs = 0;
	private int nModSpecs = 0;
	private int nConjSpecs = 0;

	private int maxFeatureValues = 0;
	private int maxClauseValues = 0;
	private int maxNounValues = 0;
	private int maxModValues = 0;
	private int maxConjValues = 0;

	private static final Logger log = LogUtils.getLogger();

	/**
	 * @param file
	 *            The "LDC Corpus" XML file containing the specifications for
	 *            the feature set.
	 * @throws IOException
	 * @throws SAXException
	 * @throws ParserConfigurationException
	 * @throws ConfigurationException
	 */
	public FeatureStructureManager(File file) throws IOException, SAXException,
			ParserConfigurationException, ConfigurationException {

		// use Java's DOM XML parser to parse the document
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		DocumentBuilder db = dbf.newDocumentBuilder();
		Document doc = db.parse(file);

		NodeList sectionNodes = doc.getElementsByTagName("section");
		for (int a = 0; a < sectionNodes.getLength(); a++) {
			Node sectionNode = sectionNodes.item(a);
			NodeList sectionNodeChildren = sectionNode.getChildNodes();

			Section section = new Section();
			ArrayList<String> sectionNotes = new ArrayList<String>();
			ArrayList<FeatureSpec> sectionFeatureSpecs = new ArrayList<FeatureSpec>();

			for (int i = 0; i < sectionNodeChildren.getLength(); i++) {

				Node sectionNodeChild = sectionNodeChildren.item(i);
				String sectionNodeName = sectionNodeChild.getNodeName();

				if (sectionNodeName.equals("section-name")) {
					String sectionName = sectionNodeChild.getTextContent();
					section.name = sectionName;
				} else if (sectionNodeName.equals("note")) {
					String note = sectionNodeChild.getTextContent();
					sectionNotes.add(note);
				} else if (sectionNodeName.equals("feature")) {
					NodeList featureNodeChildren = sectionNodeChild.getChildNodes();

					FeatureSpec featureSpec = new FeatureSpec();
					featureSpec.section = section;
					ArrayList<FeatureValueSpec> featureValueSpecs =
							new ArrayList<FeatureValueSpec>();

					for (int j = 0; j < featureNodeChildren.getLength(); j++) {
						Node featureNodeChild = featureNodeChildren.item(j);
						String featureNodeName = featureNodeChild.getNodeName();

						if (featureNodeName.equals("feature-name")) {
							String strFeatureName = featureNodeChild.getTextContent();
							featureSpec.name = strFeatureName;
							// System.out.println("FEATURE: " + strFeatureName);

						} else if (featureNodeName.equals("value")) {
							NodeList valueNodeChildren = featureNodeChild.getChildNodes();

							FeatureValueSpec featureValueSpec = new FeatureValueSpec();
							featureValueSpec.setParent(featureSpec);
							featureValueSpecs.add(featureValueSpec);

							ArrayList<FeatureValueRestriction> restrictions =
									new ArrayList<FeatureValueRestriction>();

							for (int k = 0; k < valueNodeChildren.getLength(); k++) {
								Node valueNodeChild = valueNodeChildren.item(k);
								String valueNodeName = valueNodeChild.getNodeName();

								if (valueNodeName.equals("value-name")) {
									String strValueName = valueNodeChild.getTextContent();
									// System.out.println("VALUE: " +
									// strValueName);
									featureValueSpec.setName(strValueName);

								} else if (valueNodeName.equals("restriction")) {
									String strRestriction = valueNodeChild.getTextContent();
									// System.out.println("RESTRICTION: " +
									// strRestriction);

									FeatureValueRestriction restriction =
											new FeatureValueRestriction();
									restriction.polarity = strRestriction.startsWith("~");
									strRestriction =
											StringUtils.removeLeadingString(strRestriction, "~");
									strRestriction =
											StringUtils.substringBetween(strRestriction, "(", ")");

									restriction.feature =
											StringUtils.substringBefore(strRestriction, " ");
									restriction.value =
											StringUtils.substringAfter(strRestriction, " ");
									restrictions.add(restriction);
								} else if (valueNodeName.equals("note")) {
									featureValueSpec.setNotes(valueNodeChild.getTextContent());
								} else if (valueNodeName.equals("ontRef")) {
									// TODO
								} else if (valueNodeName.equals("#text")
										|| valueNodeName.equals("#comment")) {
									// ignore
								} else {
									log.warning("Unexpected value tag: " + valueNodeName);
								}
							} // for value node children

							featureValueSpec.setRestrictions(restrictions.toArray(new FeatureValueRestriction[restrictions.size()]));

							assert featureValueSpec.getName() != null : "null name";
							assert featureValueSpec.getRestrictions() != null : "null restrictions";
							assert featureValueSpec.getParent() != null : "null parent";
						} else if (featureNodeName.equals("note")) {
							featureSpec.notes = featureNodeChild.getTextContent();
						} else if (featureNodeName.equals("ontRef")) {
							// TODO
						} else if (featureNodeName.equals("#text")
								|| featureNodeName.equals("#comment")) {
							continue; // ignore
						} else {
							log.warning("Unexpected feature tag: " + featureNodeName);
						} // if-else for possible values
					} // for featureNodeChildren

					featureSpec.values =
							featureValueSpecs.toArray(new FeatureValueSpec[featureValueSpecs.size()]);

					featureNames.add(featureSpec.name);
					featureSpecs.put(featureSpec.name, featureSpec);
					nFeatureSpecs++;
					maxFeatureValues = Math.max(maxFeatureValues, featureSpec.values.length);

					if (featureSpec.name.startsWith("np-")) {
						nNounSpecs += featureSpec.values.length;
						maxNounValues = Math.max(maxNounValues, featureSpec.values.length);
					} else if (featureSpec.name.startsWith("c-")) {
						nClauseSpecs += featureSpec.values.length;
						maxClauseValues = Math.max(maxClauseValues, featureSpec.values.length);
					} else if (featureSpec.name.startsWith("mod-")) {
						nModSpecs += featureSpec.values.length;
						maxModValues = Math.max(maxModValues, featureSpec.values.length);
						modifierList = featureSpec.getStringValues();
						modifierSet.addAll(modifierList);
					} else if (featureSpec.name.startsWith("conj-")) {
						nConjSpecs += featureSpec.values.length;
						maxConjValues = Math.max(maxConjValues, featureSpec.values.length);
						conjunctionList = featureSpec.getStringValues();
						conjunctionSet.addAll(conjunctionList);
					} else {
						log.warning("Invalid feature name: " + featureSpec.name);
					}

					// store names of valid participants
					if (featureSpec.name.equals("np-function")) {
						for (final FeatureValueSpec v : featureSpec.values) {
							String strName = StringUtils.substringAfter(v.getName(), "fn-");
							participantTypes.add(strName);
						}
					}

					// store names of valid clause types
					if (featureSpec.name.equals("c-function")) {
						for (final FeatureValueSpec v : featureSpec.values) {
							String strName = StringUtils.substringAfter(v.getName(), "fn-");
							clauseTypes.add(strName);
						}
					}

					sectionFeatureSpecs.add(featureSpec);
				} else if (sectionNodeName.equals("#text") || sectionNodeName.equals("#comment")) {
					// ignore
				} else {
					log.warning("Unexpected section tag: " + sectionNodeName);
				} // section node children if-else
			} // for section node children

			assert section.name != null : "null feature spec name";
			section.features =
					sectionFeatureSpecs.toArray(new FeatureSpec[sectionFeatureSpecs.size()]);
			section.notes = sectionNotes.toArray(new String[sectionNotes.size()]);
			sections.add(section);

		} // for sections

		clauseList = new ArrayList<String>(clauseTypes);
		participantList = new ArrayList<String>(participantTypes);

		log.info("FEATURE MANAGER: Loaded " + this.getNumClauses() + " clause types");
		log.info("FEATURE MANAGER: Loaded " + this.getNumParticipants() + " participant types");
		log.info("FEATURE MANAGER: Loaded " + this.getNumClauseFeatures() + " clause features"
				+ " (" + this.getMaximumClauseFeatureValues() + " max)");
		log.info("FEATURE MANAGER: Loaded " + this.getNumNounFeatures() + " noun features" + " ("
				+ this.getMaximumNounFeatureValues() + " max)");
		log.info("FEATURE MANAGER: Loaded " + this.getNumConjunctionFeatures()
				+ " conjunction features" + " (" + this.getMaximumConjunctionValues() + " max)");
		log.info("FEATURE MANAGER: Loaded " + this.getNumModifierFeatures() + " modifier features"
				+ " (" + this.getMaximumModifierValues() + " max)");

	}// end init

	/***************************************************************************
	 * Determining membership
	 **************************************************************************/

	public boolean isParticipantFeature(String str) {
		return str.startsWith("np-");
	}

	public boolean isClauseFeature(String str) {
		return str.startsWith("c-");
	}

	/***************************************************************************
	 * Extracting particular children
	 **************************************************************************/

	public ArrayList<TreeNode> getChildClauses(TreeNode featureStructure) {
		ArrayList<TreeNode> result = new ArrayList<TreeNode>();
		getChildClauses(result, featureStructure);
		return result;
	}

	private void getChildClauses(ArrayList<TreeNode> result, TreeNode parent) {
		for (final TreeNode child : parent.getChildren()) {
			if (child.getValues().size() == 1 && clauseTypes.contains(child.getValues().get(0))) {
				result.add(child);
			}
			getChildClauses(result, child);
		}
	}

	public ArrayList<TreeNode> getChildParticipants(TreeNode featureStructure) {
		ArrayList<TreeNode> result = new ArrayList<TreeNode>();
		getChildParticipants(result, featureStructure);
		return result;
	}

	private void getChildParticipants(ArrayList<TreeNode> result, TreeNode parent) {
		for (final TreeNode child : parent.getChildren()) {
			if (child.getValues().size() == 1
					&& participantTypes.contains(child.getValues().get(0))) {
				result.add(child);
			}
			getChildParticipants(result, child);
		}
	}

	/**
	 * Gets feature nodes for the specified clause ONLY.
	 * 
	 * @param featureStructure
	 * @return
	 */
	public ArrayList<TreeNode> getFeatureNodes(TreeNode featureStructure) {
		ArrayList<TreeNode> result = new ArrayList<TreeNode>();
		getFeatureNodes(result, featureStructure);
		return result;
	}

	private void getFeatureNodes(ArrayList<TreeNode> result, TreeNode clause) {
		for (final TreeNode child : clause.getChildren()) {
			if (child.getValues().size() == 2 && featureNames.contains(child.getValues().get(0))) {
				result.add(child);
			}
			// do not descent recursively
			// getFeatureNodes(result, child);
		}
	}

	public ArrayList<TreeNode> getChildConjunctions(TreeNode featureStructure) {
		ArrayList<TreeNode> result = new ArrayList<TreeNode>();
		getChildWithString(result, featureStructure, CONJUNCTION_TYPE);
		return result;
	}

	public ArrayList<TreeNode> getChildModifiers(TreeNode featureStructure) {
		ArrayList<TreeNode> result = new ArrayList<TreeNode>();
		getChildWithString(result, featureStructure, MODIFIER_TYPE);
		return result;
	}

	private void getChildWithString(ArrayList<TreeNode> result, TreeNode parent, String name) {
		for (final TreeNode child : parent.getChildren()) {
			if (child.getValues().size() == 1 && child.getValues().get(0).equals(name)) {
				result.add(child);
			}
			getChildWithString(result, child, name);
		}
	}

	/***************************************************************************
	 * Finding compatibilities
	 **************************************************************************/

	public ArrayList<String> getCompatibleClauses(String currentValue, SmartTree featureStructure) {
		return clauseList;
	}

	public ArrayList<String> getCompatibleParticipants(String currentValue,
			SmartTree featureStructure) {
		return participantList;
	}

	public ArrayList<String> getCompatibleConjunctions(SmartTree featureStructure) {
		return conjunctionList;
	}

	public ArrayList<String> getCompatibleModifiers(SmartTree featureStructure) {
		return modifierList;
	}

	public ArrayList<String> getCompatibleFeatureValues(String feature, SmartTree featureStructure) {
		// TODO: Compatibility checking.
		// Do we want to upgrade to feature bitmaps for this?
		// probably not. that would take a long time
		// try implementing it as a tree-handling program for now
		throw new Error();
	}

	public ArrayList<String> getAllValuesForFeature(String feature) {
		return featureSpecs.get(feature).getStringValues();
	}

	public FeatureSpec getSpecForFeature(String feature) throws FeatureStructureException {
		FeatureSpec s = featureSpecs.get(feature);
		if (s == null)
			throw new FeatureStructureException("No spec found for feature: " + feature);
		return s;
	}

	public ArrayList<String> getCompatibleElements(TreeNode key, SmartTree featureStructure, Type t) {
		return getCompatibleElements(key.getValues().get(0), featureStructure, t);
	}

	/**
	 * @param key
	 *            Either the feature name or the current value for any other
	 *            type
	 * @param featureStructure
	 * @param t
	 * @return
	 */
	public ArrayList<String> getCompatibleElements(String key, SmartTree featureStructure, Type t) {

		if (t == Type.CLAUSES) {
			return getCompatibleClauses(key, featureStructure);
		} else if (t == Type.PARTICIPANTS) {
			return getCompatibleParticipants(key, featureStructure);
		} else if (t == Type.FEATURES) {
			return getCompatibleFeatureValues(key, featureStructure);
		} else if (t == Type.CONJUNCTIONS) {
			return getCompatibleConjunctions(featureStructure);
		} else if (t == Type.MODIFIERS) {
			return getCompatibleModifiers(featureStructure);
		} else {
			throw new RuntimeException("Unknown type: " + t);
		}
	}

	public ArrayList<TreeNode> getChildren(TreeNode featureStructure, Type t) {
		if (t == Type.CLAUSES) {
			return getChildClauses(featureStructure);
		} else if (t == Type.PARTICIPANTS) {
			return getChildParticipants(featureStructure);
		} else if (t == Type.FEATURES) {
			return getFeatureNodes(featureStructure);
		} else if (t == Type.CONJUNCTIONS) {
			return getChildConjunctions(featureStructure);
		} else if (t == Type.MODIFIERS) {
			return getChildModifiers(featureStructure);
		} else {
			throw new RuntimeException("Unknown type: " + t);
		}
	}

	public FeatureContext getFeatureContextFor(String str) throws CorpusException {
		if (featureContexts == null) {
			initFeatureContexts();
		}
		FeatureContext f = featureContexts.get(str);
		if (f == null) {
			throw new CorpusException("No feature context found for: " + str);
		}
		return f;
	}

	public Collection<FeatureContext> getFeatureContexts() {
		if (featureContexts == null) {
			initFeatureContexts();
		}
		return featureContexts.values();
	}

	private void initFeatureContexts() {
		featureContexts = new HashMap<String, FeatureContext>();
		featureContexts.put(FeatureContext.ROOT.name, FeatureContext.ROOT);
		int id = FeatureContext.FIRST_FEATURE_CONTEXT_ID;

		// handle base case of root clause
		String root = FeatureContext.ROOT.name;
		for (String p : participantList) {
			String s = root + "/" + p;
			featureContexts.put(s, new FeatureContext(s, id));
			id++;
		}

		for (String c : clauseList) {
			featureContexts.put(c, new FeatureContext(c, id));
			id++;
			for (String p : participantList) {
				String s = c + "/" + p;
				featureContexts.put(s, new FeatureContext(s, id));
				id++;
			}
		}
		for (String s : participantList) {
			featureContexts.put(s, new FeatureContext(s, id));
			id++;
		}
		for (String s : conjunctionList) {
			s = StringUtils.substringAfter(s, "conjunction-");
			s = StringUtils.substringBefore(s, "-conjunction");
			featureContexts.put(s, new FeatureContext(s, id));
			id++;
		}
		featureContexts.put("modifier", new FeatureContext("modifier", id));
		id++;
		// for(final String s : modifierList)
		// featureContexts.add(new FeatureContext(s));
	}

	public Collection<String> getFeatureNames() {
		return featureSpecs.keySet();
	}
	
	public Collection<String> getParticipants() {
		return participantList;
	}
	
	public Collection<String> getClauseTypes() {
		return clauseList;
	}

	public Collection<FeatureSpec> getFeatureSpecs() {
		return featureSpecs.values();
	}

	public ArrayList<Section> getSections() {
		return sections;
	}

	public int getNumFeatures() {
		return featureSpecs.size();
	}

	/***************************************************************************
	 * Counts of high-level feature structure elements
	 **************************************************************************/

	public int getNumClauses() {
		return clauseTypes.size();
	}

	public int getNumParticipants() {
		return participantTypes.size();
	}

	/***************************************************************************
	 * Counts of features in each category
	 **************************************************************************/

	public int getNumNounFeatures() {
		return nNounSpecs;
	}

	public int getNumClauseFeatures() {
		return nClauseSpecs;
	}

	public int getNumConjunctionFeatures() {
		return nConjSpecs;
	}

	public int getNumModifierFeatures() {
		return nModSpecs;
	}

	/***************************************************************************
	 * Max counts of simultaneous feature structure elements
	 **************************************************************************/

	public int getMaximumCompatibleParticipants() {
		return getNumParticipants();
	}

	public int getMaximumCompatibleClauses() {
		// max clauses that can be together including main clause
		return getNumClauses();
	}

	/***************************************************************************
	 * Max counts of values in each category
	 **************************************************************************/

	public int getMaximumeFeatureValues() {
		return maxFeatureValues;
	}

	public int getMaximumClauseFeatureValues() {
		return maxClauseValues;
	}

	public int getMaximumNounFeatureValues() {
		return maxNounValues;
	}

	public int getMaximumConjunctionValues() {
		return maxConjValues;
	}

	public int getMaximumModifierValues() {
		return maxModValues;
	}

	/**
	 * Provides a more detailed error report than the isValid method.
	 * 
	 * @param featureStructure
	 * @throws FeatureStructureException
	 */
	public void validateFeatureStructure(SmartTree featureStructure)
			throws FeatureStructureException {

		// give a detailed error message of why the FS is invalid!
		for (final TreeNode node : featureStructure.getUnlabeledNodes()) {
			ArrayList<String> nodeValues = node.getValues();
			if (nodeValues.size() == 1) {
				String nodeValue = nodeValues.get(0);
				if (!clauseTypes.contains(nodeValue) && !participantTypes.contains(nodeValue)
						&& !conjunctionSet.contains(nodeValue) && !modifierSet.contains(nodeValue)) {
					// throw new FeatureStructureException("Unknown
					// clause/participant type: "
					// + nodeValue);
				}
			} else if (nodeValues.size() == 2) {
				String featureName = nodeValues.get(0);
				String featureValue = nodeValues.get(1);

				if (!featureNames.contains(featureName)) {
					throw new FeatureStructureException("Unknown feature name: " + featureName);
				} else {
					FeatureSpec spec = this.getSpecForFeature(featureName);
					if (!spec.getStringValues().contains(featureValue)) {
						throw new FeatureStructureException("Unknown value for feature: "
								+ featureValue + " in node " + node);
					}
				}
			} else {
				throw new FeatureStructureException("A expected one or two values: " + node);
			}
		}

		for (final TreeNode node : featureStructure.getLabeledNodes()) {
			ArrayList<String> nodeValues = node.getValues();
			if (nodeValues.size() != 0) {
				throw new FeatureStructureException(
						"A child of a clause/participant node may not have any values: " + node);
			}
		}
	}

	/**
	 * TESTING METHOD
	 * 
	 * @param args
	 * @throws Exception
	 */
	public static void main(String... args) throws Exception {
		new FeatureStructureManager(new File(args[0]));
	}
}
