/**
 * The AVENUE Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 * 
 * Corpus Navigator
 * Written by Jonathan Clark
 */
package edu.cmu.cs.lti.avenue.corpus;

import info.jonclark.log.LogUtils;
import info.jonclark.util.StringUtils;

import java.util.ArrayList;
import java.util.logging.Logger;

import edu.cmu.cs.lti.avenue.trees.smart.SmartTree;
import edu.cmu.cs.lti.avenue.trees.smart.TreeNode;

public class PhiPlusMapping {

	private static final Logger log = LogUtils.getLogger();

	// private static final String PHI = "\u03D5";
	// private static final String PHI_INVERSE = "\u03D5\u207B\u00B9";
	public static final String PHI = "phi";
	public static final String PHI_INVERSE = "phiInv";
	public static final String HEAD = "h";

	public static final int NO_CONSTITUENT = 1000;

	private final int[][] phiInverse;
	private final int[] phiInverseTop;
	private final int[] phiMapping;
	private final int[] headMapping;
	private final int[][] headInverse;

	// public PhiPlusMapping(int[] phiMapping, int[] headMapping) {
	//
	// this.phiMapping = phiMapping;
	// this.headMapping = headMapping;
	//
	// // get the dimensions for the phiInverse
	// int nMaxFeatureIndex = 0;
	// for (int i = 0; i < phiMapping.length; i++)
	// nMaxFeatureIndex = Math.max(phiMapping[i], nMaxFeatureIndex);
	//		
	// this.phiInverse = new int[nMaxFeatureIndex][];
	//		
	// // TODO: create arraylists of features indices and build up each list
	// }

	/**
	 * @param phiInverse
	 *            A two-dimensional array in which the first dimension is the
	 *            zero-based index of the feature node and the second dimension
	 *            is a list of zero-based indices of constituent nodes
	 *            associated with the feature node specified by the first
	 *            dimension.
	 * @throws CorpusException
	 */
	public PhiPlusMapping(int[][] phiInverse, int[] headMapping) throws CorpusException {

		this.phiInverse = phiInverse;
		this.headMapping = headMapping;

		// construct phiInverseTop
		this.phiInverseTop = new int[phiInverse.length];
		for (int i = 0; i < phiInverse.length; i++) {
			this.phiInverseTop[i] = Integer.MAX_VALUE;

			for (int j = 0; j < phiInverse[i].length; j++) {
				phiInverseTop[i] = Math.min(phiInverse[i][j], phiInverseTop[i]);
			}

			if (this.phiInverseTop[i] == Integer.MAX_VALUE) {
				throw new CorpusException("Expected to find phiInverse value for " + (i + 1));
			}
		}

		// get the dimensions for the phiMapping
		int nMaxConstituentIndex = 0;
		for (int i = 0; i < phiInverse.length; i++)
			for (int j = 0; j < phiInverse[i].length; j++)
				nMaxConstituentIndex = Math.max(phiInverse[i][j], nMaxConstituentIndex);

		this.phiMapping = new int[nMaxConstituentIndex + 1];

		// initialize all elements of phi mapping to -1
		for (int i = 0; i < phiMapping.length; i++)
			phiMapping[i] = -1;

		// now set elements of phiInverse
		for (int i = 0; i < phiInverse.length; i++)
			for (int j = 0; j < phiInverse[i].length; j++)
				phiMapping[phiInverse[i][j]] = i;

		// assert phiMapping.length == headMapping.length : "Phi mapping (" +
		// phiMapping.length
		// + ") and head mapping (" + headMapping.length + ") have different
		// lengths: "
		// + this.toString();

		this.headInverse = createHeadInverse(headMapping);
	}

	private static int[][] createHeadInverse(int[] headMapping) {

		ArrayList<Integer>[] headInverse = (ArrayList<Integer>[]) new ArrayList[headMapping.length];
		for (int i = 0; i < headInverse.length; i++) {
			headInverse[i] = new ArrayList<Integer>();
		}

		for (int i = 0; i < headMapping.length; i++) {
			if (headMapping[i] != -1) {
				headInverse[headMapping[i]].add(i);
			}
		}

		int[][] result = new int[headMapping.length][];
		for (int i = 0; i < headMapping.length; i++) {
			result[i] = new int[headInverse[i].size()];
			for (int j = 0; j < result[i].length; j++) {
				result[i][j] = headInverse[i].get(j);
			}
		}

		return result;
	}

	/**
	 * Get the feature structure node associated with the specified constituent
	 * structure node.
	 * 
	 * @param constituentNodeIndex
	 * @return
	 */
	public int getPhi(int constituentNodeIndex) {
		return phiMapping[constituentNodeIndex];
	}

	public TreeNode getPhi(SentencePair pair, TreeNode constituentNode) {
		int fIndex = getPhi(constituentNode.getTreeLabelingIndex());
		TreeNode fNode = pair.getFeatureStructure().getByTreeLabelingIndex(fIndex);
		return fNode;
	}

	/**
	 * Returns the 1-based constituent structure node index corresponding to the
	 * lexical head of the specified 1-based constituent node index.
	 * 
	 * @param constituentNodeIndex
	 * @return
	 */
	public int getImmediateHead(int constituentNodeIndex) {
		return headMapping[constituentNodeIndex - 1] + 1;
	}

	public TreeNode getImmediateHead(SentencePair pair, TreeNode constituentNode) {
		int cIndex = getImmediateHead(constituentNode.getTreeLabelingIndex());
		TreeNode cNode = pair.getSourceConstituentStructure().getByTreeLabelingIndex(cIndex);
		return cNode;
	}

	/**
	 * Returns the 1-based constituent structure node index corresponding to the
	 * lexical head of the specified 1-based constituent node index.
	 * 
	 * @param constituentNodeIndex
	 * @return
	 */
	public int[] getImmediateDependents(int constituentNodeIndex) throws CorpusException {
		return headInverse[constituentNodeIndex];
	}

	/**
	 * Given a 1-based constituentNodeIndex, returns the corresponding 1-based
	 * constituentNodeIndex of the given node's ultimate head.
	 * 
	 * @param constituentNodeIndex
	 * @return
	 * @throws CorpusException
	 */
	public int getUltimateHead(int constituentNodeIndex) throws CorpusException {

		assert constituentNodeIndex <= headMapping.length : "Constituent node index out of bounds: "
				+ constituentNodeIndex;

		int prevIndex = constituentNodeIndex - 1;
		int headIndex = headMapping[constituentNodeIndex - 1];

		int nSafety = 0;

		while (headIndex != prevIndex) {
			if (headMapping[headIndex] == -1)
				throw new CorpusException("No head for node " + headIndex);

			prevIndex = headIndex;
			headIndex = headMapping[headIndex];

			nSafety++;
			if (nSafety == 100) {

//				log.warning("Entered infinite loop while trying to find uhead in: " + this + "(" + x + ")");
				 throw new CorpusException(
				 "Entered infinite loop while trying to find uhead.");

				// we're in an infinite loop, attempt to break the cycle by
				// choosing item lowest in the tree (closest to a terminal)
//				while (prevIndex > headIndex) {
//					prevIndex = headIndex;
//					headIndex = headMapping[headIndex];
//				}
			}
		}
		return headIndex + 1;
	}
	
	public TreeNode getUltimateHead(SentencePair pair, TreeNode constituentNode)
			throws CorpusException {

		try {
			int cIndex = getUltimateHead(constituentNode.getTreeLabelingIndex());
			TreeNode cNode = pair.getSourceConstituentStructure().getByTreeLabelingIndex(cIndex);
			return cNode;
		} catch (CorpusException e) {
			throw new CorpusException("Could not get ultimate head for sentence pair: "
					+ pair.getMyLine() + "\n" + e.getMessage(), e);
		}
	}

	/**
	 * Get the the ***ZERO-BASED*** constituent structure nodes associated with
	 * the specified ***ONE-BASED*** feature structure node.
	 * 
	 * @param featureNodeIndex
	 * @return
	 */
	public int[] getPhiInverse(int featureNodeIndex) {
		return phiInverse[featureNodeIndex - 1];
	}

	public TreeNode[] getPhiInverse(SentencePair pair, TreeNode featureNode) {
		int[] cIndices = getPhiInverse(featureNode.getTreeLabelingIndex());
		TreeNode[] cNodes = new TreeNode[cIndices.length];
		for (int i = 0; i < cIndices.length; i++)
			cNodes[i] = pair.getSourceConstituentStructure().getByTreeLabelingIndex(cIndices[i]);
		return cNodes;
	}

	/**
	 * Gets the 1-based index of the constituent structure node with the minimum
	 * index (that is, the mother of all the c-structure nodes) associated with
	 * the specified 1-based feature node index.
	 * 
	 * @param featureNodeIndex
	 * @return
	 */
	public int getPhiInverseTop(int featureNodeIndex) {
		return phiInverseTop[featureNodeIndex - 1] + 1;
	}

	public TreeNode getPhiInverseTop(SentencePair pair, TreeNode featureNode)
			throws CorpusException {
		try {
			assert featureNode.getTreeLabelingIndex() > 0 : "Negative tree labeling index detected.";
			int cIndex = getPhiInverseTop(featureNode.getTreeLabelingIndex()); // +
			// 1
			// ;
			assert cIndex > 0 : "cIndex (" + cIndex + ") should be 1-based for node (label: "
					+ featureNode.getNodeLabel() + "; treeIndex: "
					+ featureNode.getTreeLabelingIndex() + "): " + featureNode;
			TreeNode cNode = pair.getSourceConstituentStructure().getByTreeLabelingIndex(cIndex);
			return cNode;
		} catch (Throwable t) {
			throw new CorpusException("Error getting node: " + featureNode.toString()
					+ "For sentence at " + pair.getMyLine() + "\n" + StringUtils.getStackTrace(t),
					t);
		}
	}

	public String toString() {

		StringBuilder builder = new StringBuilder();
		builder.append("[ ");
		builder.append("len(" + PHI + ")=" + phiMapping.length + "; ");
		builder.append("len(" + PHI_INVERSE + ")=" + phiInverse.length + "; ");
		builder.append("len(" + HEAD + ")=" + headMapping.length + "; ");

		// first print phi mapping
		for (int i = 0; i < phiMapping.length; i++) {
			if (phiMapping[i] != -1)
				builder.append(PHI + "(" + SmartTree.SOURCE_C_STRUCT_LABEL + (i + 1) + ")="
						+ SmartTree.F_STRUCT_LABEL + (phiMapping[i] + 1) + "; ");
		}

		// now give inverse mapping
		for (int i = 0; i < phiInverse.length; i++) {

			if (phiInverse[i].length > 0) {
				builder.append(PHI_INVERSE + "(" + SmartTree.F_STRUCT_LABEL + (i + 1) + ")={");
				for (int j = 0; j < phiInverse[i].length; j++) {
					if (j == phiInverse[i].length - 1)
						builder.append(SmartTree.SOURCE_C_STRUCT_LABEL + (phiInverse[i][j] + 1));
					else
						builder.append(SmartTree.SOURCE_C_STRUCT_LABEL + (phiInverse[i][j] + 1)
								+ ",");
				}
				builder.append("}; ");
			}

		}

		// finally show head relationships
		for (int i = 0; i < headMapping.length; i++) {
			if (headMapping[i] != -1) {
				builder.append(HEAD + "(" + SmartTree.SOURCE_C_STRUCT_LABEL + (i + 1) + ")="
						+ SmartTree.SOURCE_C_STRUCT_LABEL + (headMapping[i] + 1) + "; ");
			}
		}

		builder.append("]");

		return builder.toString();
	}

	public static PhiPlusMapping deserialize(String serialized) throws RuntimeException {

		try {
			String strPhiLength =
					StringUtils.substringBetween(serialized, "len(" + PHI + ")=", ";");
			String strPhiInvLength =
					StringUtils.substringBetween(serialized, "len(" + PHI_INVERSE + ")=", ";");
			String strHeadLength =
					StringUtils.substringBetween(serialized, "len(" + HEAD + ")=", ";");
			// int phiLength = Integer.parseInt(strPhiLength);
			int phiInvLength = Integer.parseInt(strPhiInvLength);
			int headLength = Integer.parseInt(strHeadLength);

			int[] headMapping = new int[headLength];
			int[][] phiInverse = new int[phiInvLength][];

			// initialize phiInverse with 0-length arrays
			for (int i = 0; i < phiInverse.length; i++) {
				phiInverse[i] = new int[0];
			}

			// initialize head mapping with -1
			for (int i = 0; i < headMapping.length; i++) {
				headMapping[i] = -1;
			}

			String strNoBrackets = StringUtils.substringBetween(serialized, "[", "]");
			String[] tokens = StringUtils.tokenize(strNoBrackets, "; ");

			for (final String token : tokens) {

				if (token.startsWith(PHI_INVERSE)) {

					String strFNodeIndex =
							StringUtils.substringBetween(token, "(" + SmartTree.F_STRUCT_LABEL, ")");
					int fNodeIndex = Integer.parseInt(strFNodeIndex);
					String strCNodeList = StringUtils.substringBetween(token, "={", "}");
					String[] strCNodes =
							StringUtils.tokenize(strCNodeList, ","
									+ SmartTree.SOURCE_C_STRUCT_LABEL);
					int[] cNodes = StringUtils.toIntArray(strCNodes);

					// System.out.println("PI-Mapping " + fNodeIndex + " ==> ");

					for (int i = 0; i < cNodes.length; i++)
						cNodes[i]--;

					phiInverse[fNodeIndex - 1] = cNodes;

				} else if (token.startsWith(PHI)) {
					;
				} else if (token.startsWith(HEAD)) {

					String strKey =
							StringUtils.substringBetween(token, "("
									+ SmartTree.SOURCE_C_STRUCT_LABEL, ")");
					int key = Integer.parseInt(strKey);
					String strValue =
							StringUtils.substringAfter(token, "=" + SmartTree.SOURCE_C_STRUCT_LABEL);
					int value = Integer.parseInt(strValue);

					// System.out.println("H-Mapping " + key + " ==> " + value);

					headMapping[key - 1] = value - 1;
				}
			}

			PhiPlusMapping mapping = new PhiPlusMapping(phiInverse, headMapping);
			return mapping;
		} catch (Exception e) {
			throw new RuntimeException("Exception while parsing: " + serialized + ": "
					+ StringUtils.getStackTrace(e));
		}
	}

	/**
	 * For testing only...
	 */
	public static void main(String[] args) throws Exception {

		int[] headMapping = new int[] { -1, 3, -1, -1 };
		int[][] phiInverse = new int[][] { {}, { 0, 1, 3 }, {} };

		PhiPlusMapping map = new PhiPlusMapping(phiInverse, headMapping);
		System.out.println(map.toString());
		System.out.println(deserialize(map.toString()).toString());
	}
}
