// This source file defines the class
//    CParamDef
//
// Attention:
//       - when printing usage, no difference is made neither 
//         between different integer types ("n") nor between 
//         different floating point types ("x")

// 1996 - original version by Sonja Niessen
//        extension by Gregor Leusch

// 2005.10.31 - sv: added getValue


#ifndef __paramdef_hh__
#define __paramdef_hh__

//#include <sstream>
//#include <iostream>


//________________________________________________________
//
// some typedefs
//_________________________________________________________

typedef enum {Char, String, 
	      Int, Unsignedint, 
	      Shortint, Unsignedshort,
	      Longint, Unsignedlong, 
	      Double, Float, 
	      Bool} types;

//typedef enum {False, True} boolean;


/*
_________________________________________________________

 CParamDef                                           
_________________________________________________________

 class for the definition of a command line parameter

A parameter description would be NICE here ...

Note that paramFileNameInit is NOT strcpy'd but linked. So
be careful not to delete the string before deleting the
CParamDef object. Also, the object will not free memory
associated by the paramFileNameInit parameter!

_________________________________________________________
*/

class CParamDef
{
 public:
  // Construction and destruction
  CParamDef(char *keywordInit,
	    char flagInit,
	    types partypeInit,
	    char *partypenameInit,
	    char *descriptionInit,
	    char *errordescInit,
	    bool optInit,
	    int acceptWithoutKeywordOrder);
  virtual ~CParamDef();
  
  int optional(){return opt;};
  int isSatisfied(){return satisfied;};

  const char * getSource() const { return paramFileName; }
  // note that newSource is NOT strcpy'd
  void setSource(char * newSource) { paramFileName = newSource; }

  char getFlag() {return flag;};
  int understands (char c); 
  int understands (char *ptr);

  virtual int dealWith (char *token, char *source = NULL);
  virtual void usageInfo ();
  virtual void writeDesc ();
  virtual bool skipAfter () {return true;}; // go ahead to next token
  virtual void writeDataType ();
  virtual char *getKeyword() {return keyword;}
  virtual char *getPartypeName() {return partypename;}
  virtual bool myKWless_param(int nr);       // is it my turn to get this
                                             // keywordless parameter?
  virtual bool no_keyword();                 // "true" if this is a keywordless parameter 

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  inline virtual void display(std::ostream & os = std::clog) const {}

 protected:
  virtual int getVarFromToken (char *token){return token[0];};
  virtual void writeDefault (){};

  // the class variables
  bool satisfied;  // true, if the parameter is either optional or 
                      // already encountered on runtime
  bool opt;        // true, if the parameter is optional

  char  *keyword;     // corresponding keyword in the parameter file
  char  flag;         // character indicating the command line parameter 
                      // (with leading '-')
  types partype;      // type of the parameter
  char *partypename;  // the string representing the value in showUsage() etc.
                      // e.g. "file" in "-f file", "date" in "-d date"...
  char  *description; // descriptive text for usage information
  char  *errordesc;   // (e.g.: "should be an integer")

  void **var;          // associated variable
  void *deflt;         // default value
  int acceptOrder;    // place in order of parameters that should be given without
                      // a keyword
  char *paramFileName; // pointer to name of param file, NULL if specified by command-line
};

//_________________________________________________________
//
// CCharParamDef
//_________________________________________________________
//
//
// class for the definition of a command line character parameter
//

class CCharParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CCharParamDef (char *keywordInit,
		 char flagInit,
		 char  *varInit,
		 char  defltInit,
		 char *partypenameInit,
		 char *descriptionInit,
		 bool optInit,
		 int acceptWithoutKeywordOrder);

  virtual ~CCharParamDef () {};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  char *var;          // associated variable
  char deflt;         // default value
};

//_________________________________________________________
//
// CIntParamDef
//_________________________________________________________
//
//
// class for the definition of a command line integer parameter
//
// Bugs:
//       - if the string, which is to be interpreted as an 
//         integer parameter, contains non digit characters,
//         the latter are not recognized: Only the leading
//         digit characters are converted into an integer.
//         In this case, no runtime error message is produced.

class CIntParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CIntParamDef (char *keywordInit,
		char flagInit,
		int  *varInit,
		int  defltInit,
		char *partypenameInit,
		char *descriptionInit,
		bool optInit,
		int acceptWithoutKeywordOrder);

  virtual ~CIntParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  int *var;          // associated variable
  int deflt;         // default value
};

//_________________________________________________________
//
// CUnsignedintParamDef
//_________________________________________________________
//
//
// class for the definition of a command line Unsigned integer parameter
//
// Bugs:
//       - if the string, which is to be interpreted as an 
//         integer parameter, contains non digit characters,
//         the latter are not recognized: Only the leading
//         digit characters are converted into an integer.
//         In this case, no runtime error message is produced.
//

class CUnsignedintParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CUnsignedintParamDef (char *keywordInit,
			char flagInit,
			unsigned int  *varInit,
			unsigned int  defltInit,
			char *partypenameInit,
			char *descriptionInit,
			bool optInit,
			int acceptWithoutKeywordOrder);

  virtual ~CUnsignedintParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  unsigned int *var;          // associated variable
  unsigned int deflt;         // default value
};

//_________________________________________________________
//
// CShortintParamDef
//_________________________________________________________
//
//
// class for the definition of a command line short integer parameter
//
// Bugs:
//       - if the string, which is to be interpreted as an 
//         integer parameter, contains non digit characters,
//         the latter are not recognized: Only the leading
//         digit characters are converted into an integer.
//         In this case, no runtime error message is produced.

class CShortintParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CShortintParamDef (char *keywordInit,
		     char flagInit,
		     short int  *varInit,
		     short int  defltInit,
		     char *partypenameInit,
		     char *descriptionInit,
		     bool optInit,
		     int acceptWithoutKeywordOrder);

  virtual ~CShortintParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  short int *var;          // associated variable
  short int deflt;         // default value
};

//_________________________________________________________
//
// CUnsignedshortParamDef
//_________________________________________________________
//
//
// class for the definition of a command line Unsigned integer parameter
//
// Bugs:
//       - if the string, which is to be interpreted as an 
//         integer parameter, contains non digit characters,
//         the latter are not recognized: Only the leading
//         digit characters are converted into an integer.
//         In this case, no runtime error message is produced.

class CUnsignedshortParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CUnsignedshortParamDef (char *keywordInit,
			  char flagInit,
			  unsigned short int  *varInit,
			  unsigned short int  defltInit,
			  char *partypenameInit,
			  char *descriptionInit,
			  bool optInit,
			  int acceptWithoutKeywordOrder);

  virtual ~CUnsignedshortParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  unsigned short int *var;          // associated variable
  unsigned short int deflt;         // default value
};

//_________________________________________________________
//
// CLongintParamDef
//_________________________________________________________
//
//
// class for the definition of a command line Long integer parameter
//
// Bugs:
//       - if the string, which is to be interpreted as an 
//         integer parameter, contains non digit characters,
//         the latter are not recognized: Only the leading
//         digit characters are converted into an integer.
//         In this case, no runtime error message is produced.

class CLongintParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CLongintParamDef (char *keywordInit,
		    char flagInit,
		    long int  *varInit,
		    long int  defltInit,
		    char *partypenameInit,
		    char *descriptionInit,
		    bool optInit,
		    int acceptWithoutKeywordOrder);

  virtual ~CLongintParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  long int *var;          // associated variable
  long int deflt;         // default value
};

//_________________________________________________________
//
// CUnsignedlongParamDef
//_________________________________________________________
//
//
// class for the definition of a command line unsigned long integer parameter
//
// Bugs:
//       - if the string, which is to be interpreted as an 
//         integer parameter, contains non digit characters,
//         the latter are not recognized: Only the leading
//         digit characters are converted into an integer.
//         In this case, no runtime error message is produced.

class CUnsignedlongParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CUnsignedlongParamDef (char *keywordInit,
			 char flagInit,
			 unsigned long int  *varInit,
			 unsigned long int  defltInit,
			 char *partypenameInit,
			 char *descriptionInit,
			 bool optInit,
			 int acceptWithoutKeywordOrder);

  virtual ~CUnsignedlongParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  unsigned long int *var;          // associated variable
  unsigned long int deflt;         // default value
};

//_________________________________________________________
//
// CStringParamDef
//_________________________________________________________
//
// class for the definition of a command line string parameter
//
// Bugs: 

class CStringParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CStringParamDef (char *keywordInit,
		   char flagInit,
		   char  **varInit,
		   char  *defltInit,
		   char *partypenameInit,
		   char *descriptionInit,
		   bool optInit,
		   int acceptWithoutKeywordOrder);

  virtual ~CStringParamDef ();

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){if ( deflt ) {std::cerr << deflt;} else {std::cerr << "No Default given";} };

  char **var;          // associated variable
  char *deflt;         // default value

  bool varToBeDeleted; // true, if memory of *var is allocated here
};

//_________________________________________________________
//
// CDoubleParamDef
//_________________________________________________________
//
// class for the definition of a command line double parameter
//
// Bugs: 

class CDoubleParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CDoubleParamDef (char *keywordInit,
		   char flagInit,
		   double *varInit,
		   double  defltInit,
		   char *partypenameInit,
		   char *descriptionInit,
		   bool optInit,
		   int acceptWithoutKeywordOrder);

  virtual ~CDoubleParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  double *var;          // associated variable
  double deflt;         // default value
};
//_________________________________________________________
//
// CFloatParamDef
//_________________________________________________________
//
// class for the definition of a command line Float parameter
//
// Bugs: 

class CFloatParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CFloatParamDef (char *keywordInit,
		  char flagInit,
		  float *varInit,
		  float  defltInit,
		  char *partypenameInit,
		  char *descriptionInit,
		  bool optInit,
		  int acceptWithoutKeywordOrder);

  virtual ~CFloatParamDef (){};

  virtual void *getValue () { return var; };

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  virtual int getVarFromToken (char *token);
  virtual void writeDefault (){std::cerr << deflt;};

  float *var;          // associated variable
  float deflt;         // default value
};

//_________________________________________________________
//
// CBoolParamDef
//_________________________________________________________
//
// class for the definition of a command line bool parameter
//
// Bugs: 


class CBoolParamDef : public CParamDef
{
 public:
  // Construction and destruction
  CBoolParamDef (char *keywordInit,
		 char flagInit,
		 bool *varInit,
		 bool  defltInit,
		 char *partypenameInit,
		 char *descriptionInit,
		 bool optInit,
		 int acceptWithoutKeywordOrder);

  virtual ~CBoolParamDef (){};

  virtual void *getValue () { return var; };

  virtual int dealWith (char *token, char *source);
  virtual void usageInfo ();
  virtual void writeDesc ();

  virtual bool skipAfter () {return false;}; // can be followed by another 
                                                // parameter in the same token
  virtual bool myKWless_param(int nr);
  virtual bool CBoolParamDef::no_keyword();

  //! Print keyword=value to output stream.
  virtual void display(std::ostream & os = std::clog) const;

 protected:
  bool *var;          // associated variable
  bool deflt;         // default value
  virtual void writeDefault () {std::cerr << deflt? "on":"off";}
};


#endif









