// This source file defines the class
//    CParam
//


// ********************************************************************************
// **                                                                            **
// ** The class CParam (def. in param.hh, sources etc. in param.cc, paramdef.cc, **
// ** paramdef.hh) parses command line parameters and parameter files.           **
// **                                                                            **
// ********************************************************************************
//
//                               ***************
//                               ** Features: **
//                               ***************
//
// - char, *char, bool, float/double and all int data types are implemeted
// - error checking
// - parameters may be optional or required; default can be given
// - command line parameters can be given as "-pValue", "-p Value",
//      "--parameter=value" or "--parameter value" resp. 
// - bool switches can be given as "-s" or "--switch", several switches can be 
//      given at once: "-abcd").
// - parameter files are implemented ("-f"); they may contain:
//      - "# remark" 
//      - "switch"          (cp. command line: "--switch")
//      - "parameter value" (cp. command line "--parameter=value")
// - several parameters and parameter files can be given at once, and
//      are interpreted "left-to-right", so parameters given in a file can
//      be "overwritten" by giving the new value after "-f file"
//      (except for bool switches)
//
// - "-h" and "--help" implemented
//
//*******************************************************************************
//
//                           ************************
//                           ** How to use CParam: **
//                           ************************
//
// ** Include param.hh (requires paramdef.hh), link param.o and paramdef.o.    **
// ** Instanciate a variable of the suitable type for each possible parameter. **
// ** Instanciate one instance of CParam.                                      **
// ** For each variable call method AddParamDef(...) of Cparam as follows:     **
//
// AddParamDef(char *keyword,  // --keyword=value, Parameterfile: keyword value
//
//             char flag,      // -flag value; '\0' if you don't want this para-
//                             // meter to have a flag (useful for dangerous
//                             // parameters) or if you run out of letters...
//
//             ??? *variable,  // variable to set, type: see above
//
//             ??? default,    // default value for var.
//
//             char *description,
//                             // will be printed out when called with "--help"
//
//             bool optional,  // is this parameter optional?
//
//             [char *representation], 
//                             // the representation for a value at the usage info.
//                             // optional, default is "n" / "x" / "s" / "c"  
//
//             [int pos_of_kwdless_param]);
//                             // if this parameter is to be given WITHOUT a leading key-
//                             // word/flag: The position in the command line, starting
//                             // with 1. Default is "-1", meaning "flag/keyword is
//                             // required". "representation" must be given when using; 
//                             // it will be shown in help screen. The parameter can be
//                             // given in a parameter file using the given keyword;
//                             // NOTE that either none or all non-kwd-parameters must be
//                             // given this way, as the order is mixed up by this.
//
//
//
//
// ** After this, call CParam's method **
//
// interpret(argv, argc);  
//
//
// ** with the argv and argc given to main(...). **
//
// ** Return Value is OK (1) if ok, NOT_OK (0) if an error has occurred. Except    **
// ** for some special errors (e.g. error parsing integer etc.), the program       **
// ** will not be termitated by interpret(...)!                                    **
//
// ** If the program should be terminated on any error, use Interpret_exitOnError  **
// ** instead; if you want to get a more detailed error description, use           **
// ** Interpret_returnError. (Both just warn, too, if there are errors/warnings    **
// ** regarding definitions of parameters)                                         **
//
//
// ** That's all! **
//
//
// ** All other methods are not needed but internally or in special cases. Read **
// ** the comments in the sources for additional information.                   **
//
// *******************************************************************************
//
// ** Known Bugs / Things to do: **
// - When trying to give integer numbers containing wrong characters,
//   there is NO WARNING! (eg. "123xx45" is interpreted as "123")
// - You cannot give strings or chars starting with ' ' or '=' in parameter
//   files
// - You cannot unset switches once given in the command line
// - When trying to give a value to a switch (like "--bool=123"), there
//   is NO WARNING.
// - Lists of values for one parameter (cp. "ls *.cc") are not supported yet
// - The order of the keyword-less parameters shown at "--help" need not match
//   the order to use in the command line
//
// uncomment the next line or give it as compiler option and recompile
// param.cc if CParam should exit on warnings.

// #define CPARAM_EXIT_ON_WARNINGS


#ifndef __param_hh__
#define __param_hh__

#include <iostream>

#include <ParamDef.hh>

//_________________________________________________________
//
// CParam
//_________________________________________________________
//
// class to handle command line parameters
//
// Bugs: 

class CParam
{
public:
  // Construction and destruction
  CParam();
  ~CParam();

  int AddParamDef ( CParamDef *def );    // already instantiated CParamDef
  int AddParamDef (char *keywordInit,    // general
		   char flagInit,
		   types partypeInit,
		   char *descriptionInit,
		   char *errordescInit,
		   bool optInit, char *partypenameInit,
		   int acceptWithoutKeywordOrder);
  int AddParamDef (char *keywordInit,    // char
		   char flagInit,
		   char  *varInit,
		   char  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="c",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // int
		   char flagInit,
		   int  *varInit,
		   int  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="n",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // unsigned int
		   char flagInit,
		   unsigned int  *varInit,
		   unsigned int  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="n",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // short int
		   char flagInit,
		   short int  *varInit,
		   short int  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="n",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // unsigned short int
		   char flagInit,
		   unsigned short int  *varInit,
		   unsigned short int  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="n",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // long int
		   char flagInit,
		   long int  *varInit,
		   long int  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="n",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // unsigned long int
		   char flagInit,
		   unsigned long int  *varInit,
		   unsigned long int  defltInit,
		   char *descriptionInit,
		   bool optIni, char *partypenameInit="n",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // string
		   char flagInit,
		   char  **varInit,
		   char  *defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="s",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // double
		   char flagInit,
		   double *varInit,
		   double  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="x",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // float
		   char flagInit,
		   float *varInit,
		   float  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="x",
		   int acceptWithoutKeywordOrder=-1);
  int AddParamDef (char *keywordInit,    // bool
		   char flagInit,
		   bool *varInit,
		   bool  defltInit,
		   char *descriptionInit,
		   bool optInit, char *partypenameInit="",
		   int acceptWithoutKeywordOrder=-1);

  int Interpret (char **argv,
		 int argc); /* returns OK (1) / NOT_OK (0)                    */
 
  int Interpret_returnError (char **argv,
	 	 int argc); /* for compatibility reasons: returns errors      */
			    /* unsing the PROBLEM_... constants (see below)   */
 
  int Interpret_exitOnError (char **argv,
		 int argc); /* exits on error                                 */

  int read_parameter_file (const char *Filename);

  /**
     @brief Find out where parameter was specified
     @return NULL if parameter
     - was specified on commandline
     - has not been seen yet
     - has not been specified
     otherwise, the filename of the parameter file is returned
  */
  const char* getSourceFromKey(const char *key);

  //! Print out parameters to stream.
  virtual void Display(std::ostream& os = std::clog) const;

  void * getValueFromKey( const char *key );


  /* Return values for Interpret                */
  static const int OK;
  static const int NOT_OK;
 
  /* Return values for Interpret_returnError    */
  static const int PROBLEM_UNDEFINED_PROBLEM; 
  static const int PROBLEM_MISSING_HYPHEN;
  static const int PROBLEM_UNKNOWN_KEYWORD;
  static const int PROBLEM_UNKNOWN_FLAG;
  static const int PROBLEM_INVALID_VALUE;
  static const int PROBLEM_MISSING_VALUE;
  static const int PROBLEM_MISSING_PARAMETER;
  static const int PROBLEM_CANNOT_OPEN_FILE;
  static const int PROBLEM_MISSING_VALUE_IN_FILE;
  static const int PROBLEM_UNKNOWN_KEYWORD_IN_FILE;
  static const int PROBLEM_GENERAL_PROBLEM_IN_FILE;
  static const int PROBLEM_PARAM_FILE_LOOP;


private:

  //! Pointer type for parameter definitions.
  typedef CParamDef* TParamDefPtr;
  //! Wrapper type to organize parameter definitions in a list.
  typedef struct ParamDefElem
  {
    ParamDefElem *next;       //!< Next element in list.
    TParamDefPtr definition;  //!< Pointer to parameter definition itself.
  } ParamDefElem;


  //! Element of list of all parsed parameter files.
  typedef struct TParamFileElem {
    TParamFileElem *next;    //!< Next element in list.
    char *name;              //!< Name of parameter file.
  } TParamFileElem;
  //! Pointer on list element of list of all parameter files.
  typedef TParamFileElem* PParamFileElem;

  // the class variables
  unsigned int NParamDef;    //!< number of parameter definitions in list
  /**
     @brief List of parameter definitions.
     This list contains one parameter definition object for each parameter
     the program supports.
   */
  ParamDefElem *ParamList;
  ParamDefElem *Tail;            //!< Last element of the parameter definition list.

  PParamFileElem ParamFileList;  //!< List of the names of the parsed param files

    char *argv0;
    int no_of_kwdless_par;     // Number of keyword-less parameters

    void usageInfo(char *myName); // particular proc. for better readability
    int showError(int errorNo, const char *WrongKeyword, char *WrongValue);
    bool CheckParameters();
};
#endif
















