package edu.cmu.cs.lti.letras.trees;

import java.util.ArrayList;

public class TreeNode {

	private final ArrayList<TreeNode> children = new ArrayList<TreeNode>();
	private final ArrayList<String> values = new ArrayList<String>();
	private final SmartTree parentTree;
	private final TreeNode parentNode;
	private int treeIndex;
	protected int terminalIndex;

	protected TreeNode(SmartTree parentTree, TreeNode parentNode, int treeIndex) {
		this.parentTree = parentTree;
		this.parentNode = parentNode;
		this.treeIndex = treeIndex;
		this.terminalIndex = -1;
	}

	protected void addChild(TreeNode child) {
		children.add(child);
	}

	/**
	 * Returns all nodes in this subtree that have a label (index). Order is not
	 * guaranteed.
	 */
	protected void getLabeledNodes(ArrayList<TreeNode> nodes) {

		if (this.hasTreeIndex()) {
			nodes.add(this);
		}

		if (!this.isTerminal()) {
			for (final TreeNode child : children) {
				child.getLabeledNodes(nodes);
			}
		}

	}

	public ArrayList<TreeNode> getChildren() {
		return children;
	}

	/**
	 * Returns all nodes in this subtree that have a label (index). Order is not
	 * guaranteed.
	 */
	public ArrayList<TreeNode> getLabeledNodes() {
		ArrayList<TreeNode> nodes = new ArrayList<TreeNode>();
		getLabeledNodes(nodes);
		return nodes;
	}

	public void addValue(String value) {
		this.values.add(value);
	}

	public ArrayList<String> getValues() {
		return values;
	}

	public String getValuesAsString() {
		StringBuilder builder = new StringBuilder();
		for (String value : values)
			builder.append(value.toString() + " ");
		if (builder.length() > 0)
			builder.deleteCharAt(builder.length() - 1);
		return builder.toString();
	}

	public final boolean isTerminal() {
		return (children.size() == 0);
	}

	public int getTreeIndex() {
		return treeIndex;
	}

	public boolean hasTreeIndex() {
		return (treeIndex != -1);
	}

	/**
	 * Gets the index of this node relative to the sequence of terminals. This
	 * is useful for tasks such as going from a constituent index to a lexical
	 * index.
	 * 
	 * @return
	 */
	public ArrayList<Integer> getTerminalIndices() {

		if (this.isTerminal()) {
			ArrayList<Integer> list = new ArrayList<Integer>(1);
			list.add(terminalIndex);
			return list;
		} else {
			ArrayList<Integer> list = new ArrayList<Integer>();
			for (TreeNode child : this.children) {
				child.getTerminalIndices(list);
			}
			return list;
		}

	}

	private void getTerminalIndices(ArrayList<Integer> list) {
		if (this.isTerminal()) {
			list.add(terminalIndex);
		} else {
			for (TreeNode child : this.children) {
				child.getTerminalIndices(list);
			}
		}
	}

	public String toString() {
		return toString(true);
	}

	public String getNodeLabel() {
		return parentTree.getName() + treeIndex;
	}

	public TreeNode getParentNode() {
		return parentNode;
	}

	public ArrayList<TreeNode> getLexicalTerminals() {
		ArrayList<TreeNode> results = new ArrayList<TreeNode>();
		getLexicalTerminals(results);
		return results;
	}

	/**
	 * This method should only be applied to constituent structures.
	 */
	private void getLexicalTerminals(ArrayList<TreeNode> results) {
		if (this.isTerminal()) {
			assert this.values.size() == 2 : "A lexical terminal node must contain exactly 2 values!";
			results.add(this);
		} else {
			for (TreeNode child : children) {
				child.getLexicalTerminals(results);
			}
		}
	}

	public String toString(boolean includeNodeLabels) {

		String strIndex;
		if (includeNodeLabels && hasTreeIndex())
			strIndex = " [" + getNodeLabel() + "]";
		else
			strIndex = "";

		if (isTerminal()) {
			return strIndex + "(" + getValuesAsString() + ")";
		} else {

			StringBuilder childString = new StringBuilder();
			for (TreeNode child : children) {
				childString.append(child.toString(includeNodeLabels));
			}

			return strIndex + "(" + getValuesAsString() + childString.toString() + ") ";
		}
	}

	public String toLatexString(boolean includeNodeLabels) {

		String strIndex;
		if (includeNodeLabels && hasTreeIndex())
			strIndex = "\\emph{" + getNodeLabel() + "}\\\\";
		else
			strIndex = "";

		if (isTerminal()) {
			return " [." + strIndex + getValuesAsString() + " ] ";
		} else {

			StringBuilder childString = new StringBuilder();
			for (TreeNode child : children) {
				childString.append(child.toLatexString(includeNodeLabels));
			}

			return " [." + strIndex + getValuesAsString() + childString.toString() + " ] ";
		}
	}
}
