package edu.cmu.cs.lti.letras.trees;

import info.jonclark.util.StringUtils;

import java.lang.ref.SoftReference;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Vector;

/**
 * In the current implementation, the tree structure should remain immutable
 * upon creation. There might still be a few architectual leaks, but the user
 * should still take care not to violate this constraint.
 */
public class SmartTree {

	private TreeNode root = null;
	private int nextTreeIndex = 0;
	private int nextTerminalIndex = 0;
	private Vector<TreeNode> nodeList = new Vector<TreeNode>();
	private final String name;
	private SoftReference<ArrayList<TreeNode>> labeledNodes;

	public enum LabelMode {
		LABEL_ALL_NODES, LABEL_ODD_NODES
	}

	private final LabelMode labelMode;

	private SmartTree(String name, LabelMode labelMode) {
		this.name = name;
		this.labelMode = labelMode;
		// Remember: The root MUST be set!
	}

	public String getName() {
		return name;
	}

	public static SmartTree parse(String strTree, String name, LabelMode labelMode)
			throws ParseException {

		// TODO: Handle input not starting with a paren
		// TODO: Handle input containing multiple root nodes

		assert strTree.charAt(0) == '(';
		assert strTree.charAt(strTree.length() - 1) == ')';

		SmartTree tree = new SmartTree(name, labelMode);
		tree.nextTreeIndex++;
		tree.root = parseRecursively(strTree.substring(1, strTree.length() - 1), tree, null,
				tree.nextTreeIndex, 2, labelMode);

		if (tree.nodeList.size() < 1)
			tree.nodeList.setSize(1);
		tree.nodeList.add(0, tree.root);
		
		labelTerminals(tree, tree.root);
		
		return tree;
	}

	/**
	 * Responsible for intern()ing all values
	 * 
	 * @param currentNode
	 * @param tree
	 * @param parentNode
	 * @param treeIndex
	 * @param depth
	 * @param labelMode
	 * @return
	 * @throws ParseException
	 */
	private static TreeNode parseRecursively(String currentNode, SmartTree tree,
			TreeNode parentNode, int treeIndex, int depth, LabelMode labelMode)
			throws ParseException {

		// System.out.println("parsing: " + currentNode);

		TreeNode treeNode = new TreeNode(tree, parentNode, treeIndex);

		int nChildStart;
		int nChildEnd = -1;
		while ((nChildStart = currentNode.indexOf('(', nChildEnd)) != -1) {

			if (nChildStart != nChildEnd + 1) {
				String values = currentNode.substring(nChildEnd + 1, nChildStart);
				for (String value : StringUtils.tokenize(values)) {

					treeNode.addValue(value.intern());

					assert !value.contains("(") : "Value contains leading paren: " + value;
					assert !value.contains(")") : "Value contains trailing paren: " + value;

					// System.out.println("Added value: " + value);
				}
			}

			String nestedNode = StringUtils.substringBetweenMatching(currentNode, '(', ')');

			// only assign node index labels at odd depths
			if (labelMode == LabelMode.LABEL_ODD_NODES && depth % 2 == 0) {

				// do not give children node labels
				TreeNode child = parseRecursively(nestedNode, tree, treeNode, -1, depth + 1,
						labelMode);
				treeNode.addChild(child);

			} else {

				tree.nextTreeIndex++;
				TreeNode child = parseRecursively(nestedNode, tree, treeNode, tree.nextTreeIndex,
						depth + 1, labelMode);
				treeNode.addChild(child);

				// grow vector if needed
				if (tree.nodeList.size() <= child.getTreeIndex()) {
					tree.nodeList.setSize(child.getTreeIndex() - 1);
				}

				tree.nodeList.setElementAt(child, child.getTreeIndex() - 2);
			}

		}

		if (nChildEnd + 1 < currentNode.length()) {
			String remainingValues = currentNode.substring(nChildEnd + 1, currentNode.length());
			// System.out.println("remaining: " + remainingValues);
			for (String value : StringUtils.tokenize(remainingValues)) {
				treeNode.addValue(value.intern());

				assert !value.contains("(") : "Value contains leading paren: " + value;
				assert !value.contains(")") : "Value contains trailing paren: " + value;

				// System.out.println("Added value: " + value);
			}
		}

		return treeNode;
	}
	
	private static void labelTerminals(SmartTree tree, TreeNode node) {
		if(node.isTerminal()) {
			node.terminalIndex = tree.nextTerminalIndex;
			tree.nextTerminalIndex++;
		} else {
			for(TreeNode child : node.getChildren()) {
				labelTerminals(tree, child);
			}
		}
	}

	public TreeNode getByIndex(int treeIndex) {
		TreeNode node = nodeList.elementAt(treeIndex);
		assert node.getTreeIndex() == treeIndex : "Index/element# mismatch.";
		return node;
	}

	public TreeNode getRoot() {
		return root;
	}

	public ArrayList<TreeNode> getLabeledNodes() {

		// make use of memory-sensitive caching to improve performance
		if (labeledNodes.get() == null) {
			labeledNodes = new SoftReference<ArrayList<TreeNode>>(root.getLabeledNodes());
		}

		return labeledNodes.get();
	}

	public String toString() {
		return toString(true);
	}

	public String toString(boolean includeNodeLabels) {
		return root.toString(includeNodeLabels).trim();
	}

	public String toLatexString(boolean includeNodeLabels) {
		return "\\Tree " + root.toLatexString(includeNodeLabels);
	}

	public static void main(String[] args) throws Exception {
		// String strTest = "((agent ((feature value) (another-feature 2))))";
		String strTest = "((actor ((np-function fn-actor) (np-general-type proper-noun-type)(np-person person-third) (np-biological-gender bio-gender-female)(np-animacy anim-human) (np-identifiability identifiable)(np-pronoun-antecedent antecedent-n/a)(np-number num-sg)(np-specificity specific)  (np-distance distance-neutral)(np-pronoun-exclusivity inclusivity-n/a)))    (undergoer ((np-function fn-undergoer)(np-person person-third)(np-identifiability unidentifiable) (np-number num-pl)(np-specificity non-specific)(np-animacy anim-inanimate)(np-biological-gender bio-gender-n/a)(np-general-type common-noun-type)(np-pronoun-exclusivity inclusivity-n/a)(np-pronoun-antecedent antecedent-n/a)(np-distance distance-neutral)))  (c-polarity polarity-positive) (c-v-absolute-tense past) (c-v-lexical-aspect activity-accomplishment)(c-general-type declarative-clause)(c-my-causer-intentionality intentionality-n/a)(c-comparison-type comparison-n/a)(c-relative-tense relative-n/a)(c-our-boundary boundary-n/a)(c-comparator-function comparator-n/a)(c-causee-control control-n/a)(c-our-situations situations-n/a)(c-comparand-type comparand-n/a)(c-causation-directness directness-n/a)(c-source source-neutral)(c-causee-volitionality volition-n/a)(c-assertiveness assertiveness-neutral)(c-solidarity solidarity-neutral)(c-v-grammatical-aspect gram-aspect-neutral)(c-adjunct-clause-type adjunct-clause-type-n/a)(c-v-phase-aspect phase-aspect-neutral)(c-secondary-type secondary-neutral)(c-event-modality event-modality-none)(c-function fn-main-clause)(c-minor-type minor-n/a)(c-copula-type copula-n/a)(c-power-relationship power-peer)(c-our-shared-subject shared-subject-n/a)(c-question-gap gap-n/a))";
		SmartTree tree = parse(strTest, "n", LabelMode.LABEL_ODD_NODES);

		System.out.println(tree.toString());

		for (int i = 0; i < tree.nodeList.size(); i++)
			System.out.println(i + ": " + tree.nodeList.get(i));
	}
}
