package edu.cmu.cs.lti.letras.tools;

import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;

import edu.cmu.cs.lti.letras.corpus.PhiPlusMapping;
import edu.cmu.cs.lti.letras.corpus.SentencePair;
import edu.cmu.cs.lti.letras.corpus.Serializer;
import edu.cmu.cs.lti.letras.trees.SmartTree;
import edu.cmu.cs.lti.letras.trees.TreeNode;

/**
 * A tool to aide in creating "Phi Plus" mappings. LFG Phi mappings that are
 * augmented with information about the lexical head.
 */
public class PhiPlusMapper {

	public static void main(String[] args) throws Exception {

		if (args.length != 2) {
			System.err.println("Usage: <input_cf_struct_file> <output_cf_struct_phi_file>");
			System.exit(1);
		}

		File inputFile = new File(args[0]);
		File outputFile = new File(args[1]);
		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		PrintWriter out = new PrintWriter(outputFile);

		ArrayList<SentencePair> inputPairs = Serializer.loadSentencePairs(inputFile);

		// for each CF-struct in file
		for (SentencePair pair : inputPairs) {

			SmartTree fStruct = pair.getFeatureStructure();
			SmartTree cStruct = pair.getConstituentStructure();

			assert fStruct != null;
			assert cStruct != null;

			// show user the labeled c-struct and f-struct
			System.out.println(fStruct.toString(true));
			System.out.println(cStruct.toString(true));
			System.out.println(pair.getMyLine());

			// for each node of the f-struct, ask the user to name the
			// corresponding
			// c-struct nodes

			ArrayList<TreeNode> fNodes = fStruct.getLabeledNodes();
			ArrayList<TreeNode> cNodes = cStruct.getLabeledNodes();

			int[][] phiInverse = new int[fNodes.size()][];
			int[] headMapping = new int[cNodes.size()];

			// initialize arrays
			for (int i = 0; i < phiInverse.length; i++) {
				phiInverse[i] = new int[0];
			}
			for (int i = 0; i < headMapping.length; i++) {
				headMapping[i] = -1;
			}

			// TODO: Try to automatically determine the phi mapping:
			// determine whether first NP is actor or undergoer based on what
			// type of actor there is

			// ask for phi mapping
			int i = 0;
			for (final TreeNode fNode : fNodes) {
				int[] indices = new int[1];
				indices[0] = findMatchingNode(fNode, cNodes);

				if (indices[0] == -1) {
					// get the space-delimited numbers from the user
					System.out.print(PhiPlusMapping.PHI + "(" + fNode.getNodeLabel() + ")=? ");
					String line = in.readLine();
					String[] tokens = StringUtils.tokenize(line);
					indices = StringUtils.toIntArray(tokens);
				}

				// convert to zero-base from one-base
				for (int j = 0; j < indices.length; j++)
					indices[j]--;

				phiInverse[i] = indices;
				i++;
			}

			// ask for head mapping
			for (final TreeNode cNode : cNodes) {
				if (!cNode.isTerminal()) {
					ArrayList<TreeNode> children = cNode.getChildren();
					if (children.size() == 1) {
						// we know what the head MUST be

						// convert one-base to zero-base
						headMapping[cNode.getTreeIndex() - 1] = children.get(0).getTreeIndex() - 1;
					} else {

						// attempt to automatically determine the head
						int nHead = findHead(cNode);

						// we must ask the user for the head
						if (nHead == -1) {
							System.out.print(PhiPlusMapping.HEAD + "(" + cNode.getNodeLabel()
									+ ")=? ");
							String line = in.readLine();
							nHead = Integer.parseInt(line);
						}

						// convert one-base to zero-base
						headMapping[cNode.getTreeIndex() - 1] = nHead - 1;
					}
				}
			}

			// build our phi-plus mapping
			PhiPlusMapping mapping = new PhiPlusMapping(phiInverse, headMapping);
			pair.setPhiPlusMapping(mapping);
			System.out.println(mapping.toString() + "\n\n");
			// System.out.println("Press enter to continue...\n");
			// in.readLine();

			// save incrementally
			out.println(pair.toString());
			out.flush();
		}

		in.close();
		System.out.println("Done.");

	}

	public static int findMatchingNode(TreeNode fNode, ArrayList<TreeNode> cNodes) {
		TreeNode parent = fNode.getParentNode();

		// TODO: HAS_SAID and HAS_WANTED constructions (...but not all?)

		if (parent != null) {
			String fValue = parent.getValues().get(0);

			// check if we're nested
			TreeNode grandparent = parent.getParentNode();
			if (grandparent != null && grandparent.getValues().size() > 0) {

				String grandparentValue = grandparent.getValues().get(0);
				if (grandparentValue.equals("comp")) {
					// we're in a complement, so zoom in to the complement
					// clause of
					// the c-structure
					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// remember to update criteria for children in
						// non-recursive
						// rule below!
						if (cValue.startsWith("SBAR")) {
							return findMatchingNode(fNode, cNode.getChildren());
						}
					}
				} else if (fValue.equals("causee") || fValue.equals("undergoer")
						|| fValue.equals("stimulus") || fValue.equals("pred")) {

					// we're nested in something like an undergoer
					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("VP") || cValue.equals("SQ")) {
							for (TreeNode cNode2 : cNode.getLabeledNodes()) {
								String cValue2 = cNode2.getValues().get(0);
								// ADJP is specifically for pred
								if (cValue2.equals("NP") || cValue2.equals("ADJP")) {
									return findMatchingNode(fNode, cNode2.getChildren());
								}
							}
						}
					}
				} else if (fValue.equals("rel-clause")) {

					// we're nested in a rel-clause
					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("SBAR")) {
							return findMatchingNode(fNode, cNode.getChildren());
						}
					}
				} else {
					System.out.println("Uknown type of nested clause: " + grandparentValue);
				} // checking what type of clause we're in

			} else {

				if (fValue.equals("instrumental")) {
					// match to the phrase starting with "with"

					for (TreeNode cNode : cNodes) {
						if (cNode.getValues().size() > 1 && cNode.getValues().get(1).equals("with")) {
							return cNode.getParentNode().getTreeIndex();
						}
					}

				} else if (fValue.equals("beneficiary")) {
					// match to the phrase starting with "for"

					for (TreeNode cNode : cNodes) {
						if (cNode.getValues().size() > 1 && cNode.getValues().get(1).equals("for")) {
							return cNode.getParentNode().getTreeIndex();
						}
					}

				} else if (fValue.equals("non-focus-actor")) {
					// match to the phrase starting with "by"

					for (TreeNode cNode : cNodes) {
						if (cNode.getValues().size() > 1 && cNode.getValues().get(1).equals("by")) {
							return cNode.getParentNode().getTreeIndex();
						}
					}

				} else if (fValue.equals("focus-undergoer") || fValue.equals("actor")
						|| fValue.equals("perceiver")) {
					// TODO: this will probably work for all "focus" features
					// find the first NP and hope for the best

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("NP") || cValue.equals("WHNP")) {
							return cNode.getTreeIndex();
						}
					}

				} else if (fValue.equals("causer")) {
					// TODO: this will probably work for all "focus" features
					// find the first NP and hope for the best

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("NP") || cValue.equals("WHNP")) {
							return cNode.getTreeIndex();
						}
					}

				} else if (fValue.equals("causee") || fValue.equals("undergoer")
						|| fValue.equals("stimulus") || fValue.equals("pred")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("VP") || cValue.equals("SQ")) {
							for (TreeNode cNode2 : cNode.getLabeledNodes()) {
								String cValue2 = cNode2.getValues().get(0);
								// ADJP is specifically for pred
								if (cValue2.equals("NP") || cValue2.equals("ADJP")) {
									return cNode2.getTreeIndex();
								}
							}
						}
					}

				} else if (fValue.equals("causative-undergoer")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);
						if (cValue.equals("VP") || cValue.equals("SQ")) {
							for (TreeNode cNode2 : cNode.getLabeledNodes()) {
								String cValue2 = cNode2.getValues().get(0);
								if (cValue.equals("VP") || cValue.equals("SQ")) {
									for (TreeNode cNode3 : cNode.getLabeledNodes()) {
										String cValue3 = cNode2.getValues().get(0);
										if (cValue3.equals("NP")) {
											return cNode3.getTreeIndex();
										}
									}
								}
							}
						}
					}

				} else if (fValue.equals("comp")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// remember to update criteria for children in recursive
						// rule above!
						if (cValue.startsWith("SBAR")) {
							return cNode.getTreeIndex();
						}
					}

				} else if (fValue.equals("modifier")) {

					String modRole = fNode.getValues().get(1);
					
					if (modRole.equals("mod-ordinal-first")) {
						for (TreeNode cNode : cNodes) {
							String cLex = cNode.getValues().size() > 1 ? cNode.getValues().get(1)
									: "";
							if (cLex.equals("first")) {
								return cNode.getTreeIndex();
							}
						}
					} else if(modRole.equals("role-addition")) {
						for (TreeNode cNode : cNodes) {
							String cLex = cNode.getValues().size() > 1 ? cNode.getValues().get(1)
									: "";
							
							// XXX: TODO: Search for "in addition"
							
							if (cLex.equals("first")) {
								return cNode.getTreeIndex();
							}
						}
					}
					
//					// fall back to matching non-article determiners
//					for (TreeNode cNode : cNodes) {
//						String cValue = cNode.getValues().get(0);
//						if (cValue.equals("DT") && cNode.getValues().size() > 1
//								&& !cNode.getValues().get(1).equals("a")
//								&& !cNode.getValues().get(1).equals("an")
//								&& !cNode.getValues().get(1).equals("the")) {
//							return cNode.getIndex();
//						}
//					}

				} else if (fValue.equals("rel-clause")) {

					for (TreeNode cNode : cNodes) {
						String cValue = cNode.getValues().get(0);

						// match non-article determiners
						if (cValue.equals("SBAR")) {
							return cNode.getTreeIndex();
						}
					}

				} // end massive fValue matching if-else

			} // end if parent == comp

		} else {
			// this is a root node, so match it to S1
			return 1;
		} // end if parent != null

		return -1;
	}

	public static int findHead(TreeNode cNode) {
		String value = cNode.getValues().get(0);

		if (value.equals("S1")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).startsWith("SBARQ"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("SQ") || value.equals("S")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).equals("VP"))
					return child.getTreeIndex();
			}
			// fallback to AUX
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).equals("AUX"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("SBARQ")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).startsWith("SQ")
						|| child.getValues().get(0).startsWith("S"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("SBAR")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).startsWith("S"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("NP")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).startsWith("NN"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("VP")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).startsWith("VB")
						|| child.getValues().get(0).equals("VP")
						|| child.getValues().get(0).equals("S"))
					return child.getTreeIndex();
			}
			// fallback to AUX
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).startsWith("AUX"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("PP")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).equals("NP"))
					return child.getTreeIndex();
			}
			return -1;
		} else if (value.equals("ADJP")) {
			for (TreeNode child : cNode.getChildren()) {
				if (child.getValues().get(0).equals("JJ"))
					return child.getTreeIndex();
			}
			return -1;
		}

		return -1;
	}
}
