/**
 * The LETRAS Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 */
package edu.cmu.cs.lti.letras.navigation;

import info.jonclark.clientserver.ConnectionException;
import info.jonclark.clientserver.SimpleClient;
import info.jonclark.clientserver.SimpleServer;
import info.jonclark.log.LogUtils;
import info.jonclark.properties.PropertyUtils;
import info.jonclark.properties.SmartProperties;
import info.jonclark.util.ArrayUtils;
import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Properties;
import java.util.Queue;
import java.util.logging.Logger;

import edu.cmu.cs.lti.letras.corpus.SentencePair;
import edu.cmu.cs.lti.letras.corpus.SentenceTest;
import edu.cmu.cs.lti.letras.corpus.Serializer;
import edu.cmu.cs.lti.letras.featurebitmaps.Feature;
import edu.cmu.cs.lti.letras.featurebitmaps.FeatureGroup;
import edu.cmu.cs.lti.letras.filtering.ImplicationFinder;
import edu.cmu.cs.lti.letras.filtering.SentenceAnalyzer;
import edu.cmu.cs.lti.letras.filtering.SentenceSelector;
import edu.cmu.cs.lti.letras.filtering.TestFilter;

/**
 * A navigator that applies implicational universals to find the most relevant
 * examples from the elicitation corpus.
 */
public class ElicitationNavigator {

	// TODO: keep a list of sentences the user has already completed
	// and NEVER remove one of these sentences
	//
	// Talk to Erik about implementing this on the client side
	// so that we don't annoy our poor user

	private final File elicitationCorpusFile;
	private final File universalsFile;
	private final File testsFile;

	// the minimum probability an implicational universal must have to be used
	// in calculations
	private final double minProb;

	// the four major components of the navigation system
	private final SentenceAnalyzer sentenceAnalyzer;
	private final ImplicationFinder implicationFinder;
	private final TestFilter testFilter;
	private final SentenceSelector sentenceSelector;

	private ArrayList<SentenceTest> remainingTests = new ArrayList<SentenceTest>();

	private static final Logger log = LogUtils.getLogger();

	// our connection to the elicitation tool
	private final SimpleServer server;
	private final int port;

	public ElicitationNavigator(Properties props) throws NumberFormatException, IOException,
			ParseException {

		log.info("Initializing...");

		// extract configuration from properties file
		SmartProperties smartProps = new SmartProperties(props);
		this.elicitationCorpusFile = new File(props.getProperty("elicitationCorpusFile"));
		this.universalsFile = new File(props.getProperty("universalsFile"));
		this.testsFile = new File(props.getProperty("testsFile"));

		this.minProb = Double.parseDouble(props.getProperty("minProb"));

		this.port = smartProps.getPropertyInt("port");
		int nMaxConnections = smartProps.getPropertyInt("maxConnections");
		String serverEncoding = props.getProperty("serverEncoding");

		// load data files
		ArrayList<SentencePair> sentencePairsList = Serializer.loadSentencePairs(elicitationCorpusFile);
		HashSet<SentencePair> sentencePairsHash = new HashSet<SentencePair>(sentencePairsList);
		ArrayList<FeatureGroup<Feature>> implicationalUniversals = Serializer.loadImplicationalUniversals(
				universalsFile, minProb);
		ArrayList<FeatureGroup<SentenceTest>> tests = Serializer.loadSentenceTests(testsFile,
				sentencePairsHash);

		// initialize remaining tests with all
		for (FeatureGroup<SentenceTest> group : tests) {
			remainingTests.add(group.getValue());
		}

		// instantiate major navigation components
		this.sentenceAnalyzer = new SentenceAnalyzer(tests);
		this.implicationFinder = new ImplicationFinder(implicationalUniversals);
		this.testFilter = new TestFilter(tests);
		this.sentenceSelector = new SentenceSelector();

		// start a "feature finder" server for the elicitation tool to connect
		// to
		this.server = new SimpleServer(port, nMaxConnections, serverEncoding) {
			@Override
			public void handleClientRequest(SimpleClient client) {
				try {
					provideElicitationSentences(client);
				} catch (ConnectionException e) {
					log.severe("ConnectionException while communicating with client: "
							+ StringUtils.getStackTrace(e));
				} catch (IOException e) {
					log.severe("IOException while communicating with client: "
							+ StringUtils.getStackTrace(e));
				}
			}
		};
	}

	/**
	 * Starts the ElicitationNavigator in server mode for use with the
	 * Elicitation Tool
	 * 
	 * @throws IOException
	 */
	public void runServer() throws IOException {
		server.runServer();
		log.info("Server is now running on port " + port);
	}

	/**
	 * Run the ElicitationNavigator in console mode (as opposed to server mode)
	 * 
	 * @throws IOException
	 */
	public void runConsoleDemo() throws IOException {

		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		SentencePair sentencePair;

		while ((sentencePair = getNextSentencePair()) != null) {
			System.err.println("Context: " + sentencePair.getContext());
			System.err.println("Please translate: "
					+ StringUtils.untokenize(sentencePair.getSourceSentence()));
			System.err.print("Translation: ");

			String line = in.readLine();
			sentencePair.setFSentence(StringUtils.tokenize(line));
			submitElicitedSentencePair(sentencePair);
		}
	}

	/**
	 * A thin wrapper around the sentence selector that returns the next
	 * sentence (hopefully the current most valuable sentence) for elicitation.
	 * 
	 * @return
	 */
	public SentencePair getNextSentencePair() {
		log.info("There are now " + remainingTests.size() + " tests remaining.");

		SentencePair sentencePair = sentenceSelector.getNextSentencePair(remainingTests);
		return sentencePair;
	}

	/**
	 * Pass along a sentence elicited from the user to the various components of
	 * the system.
	 * 
	 * @param elicitedSentencePair
	 */
	public void submitElicitedSentencePair(SentencePair elicitedSentencePair) {

		// analyze sentence
		sentenceAnalyzer.addElicitedSentencePair(elicitedSentencePair);
		ArrayList<Feature> discoveredFeatures = sentenceAnalyzer.getDiscoveredFeatures();

		// debug output
		StringBuilder discoveredBuilder = new StringBuilder();
		int i = 0;
		for (Feature feature : discoveredFeatures)
			discoveredBuilder.append(++i + ". " + feature.getName() + "\n");
		log.info("The following features were discovered:\n" + discoveredBuilder.toString());

		// see if we can gather any knowledge from out IU's
		implicationFinder.addDiscoveredFeatures(discoveredFeatures);
		ArrayList<Feature> impliedFeatures = implicationFinder.getImplications();

		// debug output
		StringBuilder impliedBuilder = new StringBuilder();
		i = 0;
		for (Feature feature : impliedFeatures)
			impliedBuilder.append(++i + ". " + feature.getName() + "\n");
		log.info("The following features were found in the implicational universals:\n"
				+ impliedBuilder.toString());

		// now combine all of our known features
		ArrayList<Feature> allKnownFeatures = new ArrayList<Feature>();
		allKnownFeatures.addAll(discoveredFeatures);
		allKnownFeatures.addAll(impliedFeatures);

		// turn our eliminated features into the features that we still care
		// about
		FeatureGroup<Void> allKnownFeaturesGroup = new FeatureGroup<Void>(allKnownFeatures, null);
		FeatureGroup<Void> unknownFeaturesGroup = allKnownFeaturesGroup.complement();

		// update how many tests are remaining
		remainingTests = testFilter.getRemainingTests(unknownFeaturesGroup);
	}

	/**
	 * This method is the main point of integration between the server, client,
	 * and all navigational components.
	 * 
	 * @param client
	 * @throws ConnectionException
	 * @throws IOException
	 */
	public void provideElicitationSentences(SimpleClient client) throws ConnectionException,
			IOException {

		log.info("New incoming connection from " + client.getHost());

		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		SentencePair outgoingSentencePair;
		SentencePair incomingSentencePair = null;

		// the sentences that we have sent to the client, but not yet received
		// back.
		Queue<SentencePair> outstandingSentences = new LinkedList<SentencePair>();

		while ((outgoingSentencePair = getNextSentencePair()) != null
				|| incomingSentencePair != null) {

			if (outgoingSentencePair != null) {
				log.info("Requesting translation of sentence: '"
						+ StringUtils.untokenize(outgoingSentencePair.getSourceSentence())
						+ "' With context: '" + outgoingSentencePair.getContext() + "'");

				client.sendMessage("newpair");
				client.sendMessage("srcsent: "
						+ StringUtils.untokenize(outgoingSentencePair.getSourceSentence()));
				client.sendMessage("tgtsent: "
						+ StringUtils.untokenize(outgoingSentencePair.getTargetSentence()));
				client.sendMessage("aligned: " + outgoingSentencePair.getAlignment());
				client.sendMessage("context: " + outgoingSentencePair.getContext());
				client.sendMessage("comment: " + outgoingSentencePair.getComment());
				client.sendMessage("");
				client.sendMessage("");

				outstandingSentences.add(outgoingSentencePair);
			}

			// get the group back from the client, do some processing, and send
			// more
			incomingSentencePair = null;
			String clientLine;
			while ((clientLine = client.getMessage()) != null && !(clientLine.equals(""))) {

				String value = StringUtils.substringAfter(clientLine, ":");
				value = value.trim();

				if (clientLine.startsWith("newpair")) {
					incomingSentencePair = outstandingSentences.poll();
				} else if (clientLine.startsWith("srcsent")) {
					incomingSentencePair.setESentence(StringUtils.tokenize(value));
				} else if (clientLine.startsWith("tgtsent")) {
					incomingSentencePair.setFSentence(StringUtils.tokenize(value));
				} else if (clientLine.startsWith("aligned")) {
					incomingSentencePair.setAlignments(value);
				} else if (clientLine.startsWith("context")) {
					incomingSentencePair.setContext(value);
				} else if (clientLine.startsWith("comment")) {
					incomingSentencePair.setComment(value);
				} else {
					log.warning("Unknown tag in response from elicitation tool: " + clientLine);
				}

			} // end while

			// kill off extra newline
			String extraNewline = client.getMessage();
			assert extraNewline.equals("");

			if (incomingSentencePair != null) {
				log.info("For sentence '"
						+ StringUtils.untokenize(outgoingSentencePair.getSourceSentence())
						+ "' got translation '"
						+ StringUtils.untokenize(outgoingSentencePair.getTargetSentence()) + "'");
				submitElicitedSentencePair(outgoingSentencePair);
			} else {
				log.info("No sentence received this iteration.");
			}
		}

		in.close();

		client.sendMessage("endcorpus");
		log.info("Done. Terminating ElicitationNavigator.");
		System.exit(0);
	}

	public static void main(String[] args) throws Exception {

		if (args.length != 1) {
			System.err.println("Usage: program <navigator_properties_file>");
			System.exit(1);
		}

		Properties props = PropertyUtils.getProperties(args[0]);
		ElicitationNavigator nav = new ElicitationNavigator(props);
		assert nav != null;

		if (ArrayUtils.unsortedArrayContains(args, "--console")) {
			nav.runConsoleDemo();
		} else {
			nav.runServer();
		}
		// wait until the user kills this process
		while (true)
			Thread.sleep(Long.MAX_VALUE);
	}
}
