package edu.cmu.cs.lti.letras.featuredetection;

import info.jonclark.log.LogUtils;
import info.jonclark.util.StringUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.logging.Logger;

import edu.cmu.cs.lti.letras.corpus.Alignment;
import edu.cmu.cs.lti.letras.corpus.PhiPlusMapping;
import edu.cmu.cs.lti.letras.corpus.SentencePair;
import edu.cmu.cs.lti.letras.trees.SmartTree;
import edu.cmu.cs.lti.letras.trees.TreeNode;
import edu.cmu.cs.lti.letras.trees.SmartTree.LabelMode;

/**
 * A rule that specifies patterns for 1 or more sentences that must then match
 * given constraints (usually constraints on the target language lexical items)
 * in order to fire a specified feature value (such as a WALS universal). TODO:
 * Allow nested if-else-then statements to perform error checking
 */
public class Rule {

	public static final String RULE = "rule";
	public static final String VARIABLES = "variables";
	public static final String SENTENCES = "sentences";
	public static final String IF = "if";
	public static final String THEN = "then";
	public static final String WALS = "WALS";

	public static final String DIFFERENT = "different";
	public static final String DIFFERENT_SUFFIX = "different-suffix";
	public static final String DIFFERENT_PREFIX = "different-prefix";
	public static final String IN_ORDER = "in-order";

	public static final String SOURCE_LEX = "source-lex";
	public static final String TARGET_LEX = "target-lex";
	public static final String SOURCE_LEX_IHEAD = "source-lex-ihead";
	// public static final String TARGET_LEX_IHEAD = "target-lex-ihead";
	public static final String SOURCE_LEX_UHEAD = "source-lex-uhead";
	public static final String TARGET_LEX_UHEAD = "target-lex-uhead";
	public static final String FNODE = "fnode";
	public static final String POS = "pos";

	private static final int EXPECTED_MATCHES_PER_VARIABLE = 2;
	private static final int EXPECTED_NUM_VARIABLES = 5;

	private final HashMap<String, Integer> sentencePatternVariables;
	private final TreeNode[] sentencePatterns;

	private final TreeNode[] conditions;
	private final String[] implications;

	private static class LexicalResult {
		public SentencePatternMatch patternMatch;
		public TreeNode fNode;
		public ArrayList<TreeNode> lexicons;
		
		// used to collect debug info when going from source to target
		public ArrayList<TreeNode> intermediateLexicons; 
	}

	private static class ComparisonResult implements Cloneable {
		private TreeNode condition;
		private ArrayList<LexicalResult> lexicalOperands;
		private String implication; // RHS "then"
		private boolean onlyCopy = true;

		protected void setCondition(TreeNode condition) {
			this.condition = condition;
		}

		protected void setLexicalOperands(ArrayList<LexicalResult> lexicalOperands) {
			this.lexicalOperands = lexicalOperands;
		}

		protected void setImplication(String implication) {
			this.implication = implication;
		}

		public TreeNode getCondition() {
			return condition;
		}

		public ArrayList<LexicalResult> getLexicalOperands() {
			return lexicalOperands;
		}

		public String getImplication() {
			return implication;
		}

		/**
		 * Much like clone except that the first call returns the existing
		 * object;
		 */
		public ComparisonResult uniqueCopy() {
			if (onlyCopy) {
				return this;
			} else {
				onlyCopy = false;
				try {
					return (ComparisonResult) super.clone();
				} catch (CloneNotSupportedException e) {
					throw new Error(e);
				}
			}
		}

	}

	// array.length = numSentencePatterns
	// ArrayList.size() = numMatchesForVariable
	private final ArrayList<SentencePatternMatch>[] ruleMatches;

	// BEGIN "SYMBOL TABLE"

	private final HashMap<String, Integer> variableIndices;

	// String[numVariables][numPossibleValues]
	private final String[][] variablePossibleValues;

	// END "SYMBOL TABLE"

	private static final Logger log = LogUtils.getLogger();

	protected Rule(TreeNode[] sentencePatterns, TreeNode[] conditions, String[] implications,
			HashMap<String, String[]> variables, ArrayList<String> sentencePatternVars) {

		this.conditions = conditions;
		this.implications = implications;

		this.sentencePatternVariables = new HashMap<String, Integer>(sentencePatternVars.size());
		for (int i = 0; i < sentencePatternVars.size(); i++) {
			this.sentencePatternVariables.put(sentencePatternVars.get(i), i);
		}

		this.sentencePatterns = sentencePatterns;
		this.ruleMatches = createArray(variables.size());
		for (int i = 0; i < ruleMatches.length; i++)
			this.ruleMatches[i] = new ArrayList<SentencePatternMatch>(EXPECTED_MATCHES_PER_VARIABLE);

		int i = 0;
		this.variableIndices = new HashMap<String, Integer>(variables.size());
		this.variablePossibleValues = new String[variables.size()][];
		for (Entry<String, String[]> variable : variables.entrySet()) {
			this.variableIndices.put(variable.getKey(), i);
			this.variablePossibleValues[i] = variable.getValue();
			i++;
		}
	}

	/**
	 * Hack to get around Java's generic array deficiency
	 */
	@SuppressWarnings("unchecked")
	private static ArrayList<SentencePatternMatch>[] createArray(int size) {
		return (ArrayList<SentencePatternMatch>[]) new ArrayList[size];
	}

	public static Rule createRule(String serialized, ArrayList<SentencePair> allSentences)
			throws ParseException {

		SmartTree ruleTree = SmartTree.parse(serialized, "r", LabelMode.LABEL_ALL_NODES);

		// TODO: resolve which sentences and feature nodes we might have to deal
		// with ahead of time

		HashMap<String, String[]> variables = new HashMap<String, String[]>(EXPECTED_NUM_VARIABLES);
		ArrayList<String> implicationList = new ArrayList<String>();
		ArrayList<TreeNode> conditionList = new ArrayList<TreeNode>();
		ArrayList<TreeNode> sentencePatterns = new ArrayList<TreeNode>();
		ArrayList<String> sentencePatternVariables = new ArrayList<String>();

		TreeNode root = ruleTree.getRoot();
		assert root.getValues().get(0).equals(RULE);
		assert root.getChildren().size() == 4;
		for (TreeNode infoNode : root.getChildren()) {
			if (infoNode.getValues().size() > 0) {
				String childName = infoNode.getValues().get(0);
				if (childName.equals(VARIABLES)) {

					// TODO: add possible names of variables first
					// but later resolve to the actual value of that variable
					// variables will actually take on different values for
					// different matched sentences.

					assert infoNode.getChildren().size() == 1 : "A variable may have only 1 list argument";
					TreeNode variableData = infoNode.getChildren().get(0);
					String variableName = variableData.getValues().get(0);
					TreeNode variableValues = variableData.getChildren().get(0);
					String[] arrVariableValues = StringUtils.tokenize(variableValues.getValues().get(
							0));
					variables.put(variableName, arrVariableValues);

				} else if (childName.equals(SENTENCES)) {

					for (TreeNode sentenceNode : infoNode.getChildren()) {

						assert sentenceNode.getValues().size() == 1 : "Sentence pattern requires name label";
						assert sentenceNode.getChildren().size() == 1 : "Sentence pattern must have one list argument";

						// strip of the node with the variable name, then add
						sentencePatternVariables.add(sentenceNode.getValues().get(0));
						sentencePatterns.add(sentenceNode.getChildren().get(0));
					}

				} else if (childName.equals(IF)) {

					for (TreeNode conditionNode : infoNode.getChildren()) {
						conditionList.add(conditionNode);
					}

				} else if (childName.equals(THEN)) {

					for (TreeNode implicationNode : infoNode.getChildren()) {
						assert implicationNode.getValues().size() == 2;
						assert implicationNode.getValues().get(0).equals(WALS);
						String strImplication = implicationNode.getValues().get(0);
						implicationList.add(strImplication);
					}

				}

			}
		}

		Rule rule = new Rule(sentencePatterns.toArray(new TreeNode[sentencePatterns.size()]),
				conditionList.toArray(new TreeNode[conditionList.size()]),
				implicationList.toArray(new String[implicationList.size()]), variables,
				sentencePatternVariables);
		return rule;
	}

	protected TreeNode[] getSentencePatterns() {
		return sentencePatterns;
	}

	protected HashMap<String, Integer> getSentencePatternVariables() {
		return sentencePatternVariables;
	}

	/**
	 * Called for every Rule in the system when a new sentence pair is elicited.
	 * Returns true if some sentence pattern for this rule matches. If the pair
	 * matches, this Rule automatically keeps an internal record so that
	 * 
	 * @param pair
	 * @return
	 */
	public boolean matches(SentencePair pair) {
		SmartTree fStruct = pair.getFeatureStructure();

		boolean foundMatch = false;

		// iterate over each pattern
		for (int i = 0; i < sentencePatterns.length; i++) {

			// the following code is also in the evaluateFNode() method

			// RESTRICTION: must match a role (labeled node) first
			// this should improve performance dramatically
			ArrayList<TreeNode> labeledNodes = fStruct.getLabeledNodes();
			for (TreeNode fNode : labeledNodes) {
				if (matchesRecursively(sentencePatterns[i], fNode)) {

					// remember this match
					ruleMatches[i] = new SentencePatternMatch(sentenceVariable, rule,
							nSentencePattern, pair, variableValues);

					// we already matched this pattern, so see if the same
					// sentence matches the other pattern too
					break;
				}
			}
		}

		return foundMatch;
	}

	/**
	 * Recursive helper for matches()
	 */
	private boolean matchesRecursively(TreeNode patternNode, TreeNode fNode) {

		if (fNode.getValues().size() == 0) {

			// check that the pattern also specifies no values
			if (patternNode.getValues().size() != 0) {
				return false;
			}

		} else {

			// TODO: Use hashing here? Or does == still make more sense since
			// we're using intern()?
			ArrayList<String> valuesToMatch = patternNode.getValues();
			ArrayList<String> fNodeValues = fNode.getValues();

			// TODO: XXX: Replace variable values
			log.warning("Not replacing variable values.");

			for (String valueToMatch : valuesToMatch) {
				boolean matched = false;
				for (String fNodeValue : fNodeValues) {
					assert fNodeValue == fNodeValue.intern() : "fNodeValue not interned";
					assert valueToMatch == valueToMatch.intern() : "valueToMatch not interned";
					if (fNodeValue == valueToMatch) {
						matched = true;
						break;
					}
				}
				if (!matched) {
					return false;
				}
			}

		} // if fNode has label

		// we matched everything we needed to at this tree level, so continue
		// recursive matching if needed
		if (patternNode.getChildren().size() > 0) {

			// we have more pattern left to match, but nothing to match it to
			if (fNode.getChildren().size() < 1)
				return false;

			// now we have to try the cross-product of node subtree pairings
			// generally, the cross-product should be relatively small
			for (TreeNode patternChild : patternNode.getChildren()) {
				for (TreeNode fChild : fNode.getChildren()) {
					if (matchesRecursively(patternChild, fChild)) {
						return true;
					}
				}
			}

			// no combination in the cross-product matched
			return false;
		} else {
			// we matched everything there was to match
			return true;
		}
	}

	/**
	 * @return True if this rule has at least one match for each specified
	 *         sentence pattern.
	 */
	public boolean isReadyToEvaluate() {
		for (int i = 0; i < ruleMatches.length; i++)
			if (ruleMatches[i].size() < 1)
				return false;
		return true;
	}

	/**
	 * Returns a list of maching sentence pairs if all conditions of this rule
	 * hold for at least one pair. Otherwise, return a list of length zero.
	 * (shortcircuit AND behavior)
	 * 
	 * @return
	 */
	public ArrayList<ComparisonResult> evaluate(SentencePatternMatch match) {

		// How do we handle all of this junk?
		// retain shortcircuit AND

		ArrayList<ComparisonResult> allResults = null;

		for (TreeNode condition : conditions) {
			ArrayList<ComparisonResult> boolResults = evaluateBoolean(condition);

			if (boolResults.size() == 0) {
				// evaluation failed; return "false"
				return boolResults;
			} else {
				if (allResults == null)
					allResults = boolResults;
				else
					allResults.addAll(boolResults);
			}
		}

		assert allResults != null : "Some results should have been returned from above if we got here.";

		// TODO: resolve implications of evaluation results

		return allResults;
	}

	private ArrayList<ComparisonResult> evaluateBoolean(TreeNode condition) {
		String functionName = condition.getValues().get(0);

		// make use of interned strings for == comparison
		assert functionName == functionName.intern() : "functionName not interned";

		if (condition.getChildren().size() != 2) {
			// TODO: Provide trace and line number
			throw new RuntimeException("Invalid number of arguments for function " + functionName);
		}

		if (functionName == DIFFERENT) {
			return evaluateDifferent(condition, functionName);
		} else if (functionName == DIFFERENT_SUFFIX) {
			return evaluateDifferent(condition, functionName);
		} else if (functionName == DIFFERENT_PREFIX) {
			return evaluateDifferent(condition, functionName);
		} else if (functionName == IN_ORDER) {
			return evaluateInOrder(condition);
		} else {
			// TODO: Give trace for file and line number
			throw new RuntimeException("Unrecognized boolean function name: " + functionName);
		}
	}

	/**
	 * Ensure that if each variable corresponds to a unique sentence instance.
	 * (We don't want to compare a sentence to itself unless the user intended
	 * that. Likewise, we want to compare a sentence only to itself if that's
	 * what the user intended).
	 * 
	 * @return
	 */
	private boolean hasProperVariableUsage(SentencePatternMatch... matches) {

		for (int i = 0; i < matches.length; i++) {
			for (int j = i; j < matches.length; j++) {
				if (matches[i].getSentenceVariable() == matches[j].getSentenceVariable()) {
					// same variable => sentences must be the same
					if (!matches[i].getSentencePair().equals(matches[j].getSentencePair())) {
						return false;
					}
				} else {
					// different variable => sentences must not be the same
					if (matches[i].getSentencePair().equals(matches[j].getSentencePair())) {
						return false;
					}
				}
			}
		}

		return true;
	}

	private ArrayList<ComparisonResult> evaluateDifferent(TreeNode condition, String functionName) {
		// first resolve which two constituents we're comparing
		ArrayList<LexicalResult> resultsA = evaluateConstituent(condition.getChildren().get(0));
		ArrayList<LexicalResult> resultsB = evaluateConstituent(condition.getChildren().get(1));

		ArrayList<ComparisonResult> mergedResults = new ArrayList<ComparisonResult>(resultsA.size()
				* resultsB.size());

		// take the cross product of the results
		for (final LexicalResult resultA : resultsA) {
			for (final LexicalResult resultB : resultsB) {

				// never compare a sentence to itself if the sentences are
				// represented by different variables
				if (!hasProperVariableUsage(resultA.patternMatch, resultB.patternMatch)) {
					continue;
				}

				ArrayList<TreeNode> lexiconsA = resultA.lexicons;
				ArrayList<TreeNode> lexiconsB = resultB.lexicons;

				// first see if we even have the same number of tokens
				// (lexicons)
				if (lexiconsA.size() != lexiconsB.size()) {

					// we don't have the same number of tokens
					if (functionName == DIFFERENT) {

						// yes! we can say they are different
						ComparisonResult comparisonResult = new ComparisonResult();
						comparisonResult.lexicalOperands = new ArrayList<LexicalResult>(2);
						comparisonResult.lexicalOperands.add(resultA);
						comparisonResult.lexicalOperands.add(resultB);
						comparisonResult.condition = condition;
						mergedResults.add(comparisonResult);

					} else if (functionName == DIFFERENT_SUFFIX) {

						// we do not know how to properly match the words in
						// this case
						; // add nothing

					} else if (functionName == DIFFERENT_PREFIX) {

						// we do not know how to properly match the words in
						// this case
						; // add nothing

					} else {
						throw new RuntimeException("Unrecognized difference function name: "
								+ functionName);
					}

				} else {

					// we have the same number of lexicons
					for (int i = 0; i < lexiconsA.size(); i++) {
						ArrayList<String> valuesA = lexiconsA.get(i).getValues();
						ArrayList<String> valuesB = lexiconsB.get(i).getValues();

						assert valuesA.size() == 2 : "Expected 2 values: POS tag and lexical entry";
						assert valuesB.size() == 2 : "Expected 2 values: POS tag and lexical entry";

						String lexA = valuesA.get(1);
						String lexB = valuesB.get(1);

						assert lexA == lexA.intern();
						assert lexB == lexB.intern();

						assert lexA.length() > 0 : "Zero length words are not allowed";
						assert lexB.length() > 0 : "Zero length words are not allowed";

						boolean isDifferent;

						if (functionName == DIFFERENT) {
							isDifferent = (lexA != lexB);
						} else if (functionName == DIFFERENT_SUFFIX) {
							isDifferent = (lexA != lexB && lexA.charAt(0) == lexB.charAt(0));
						} else if (functionName == DIFFERENT_PREFIX) {
							isDifferent = (lexA != lexB && lexA.charAt(lexA.length() - 1) == lexB.charAt(lexB.length() - 1));
						} else {
							throw new RuntimeException("Unrecognized difference function name: "
									+ functionName);
						}

						if (isDifferent) {
							// yes! we can say they are different
							ComparisonResult comparisonResult = new ComparisonResult();
							comparisonResult.lexicalOperands = new ArrayList<LexicalResult>(2);
							comparisonResult.lexicalOperands.add(resultA);
							comparisonResult.lexicalOperands.add(resultB);
							comparisonResult.condition = condition;
							mergedResults.add(comparisonResult);
						} else {
							// add nothing
						}

					} // for i in lexiconsA/B

				} // if lexicons.length same

			} // for resultB
		} // for resultA

		return mergedResults;
	}

	private ArrayList<ComparisonResult> evaluateInOrder(TreeNode condition) {
		assert condition.getChildren().size() > 1 : "in-order requires at least 2 arguments";

		ArrayList<ComparisonResult> hypotheses = new ArrayList<ComparisonResult>();

		// begin our hypotheses vector for results by populating it with the
		// first position
		ArrayList<LexicalResult> firstPosition = evaluateConstituent(condition.getChildren().get(0));
		for (LexicalResult lexicalResult : firstPosition) {
			ComparisonResult comparisonResult = new ComparisonResult();
			comparisonResult.lexicalOperands = new ArrayList<LexicalResult>();
			comparisonResult.lexicalOperands.add(lexicalResult);
			hypotheses.add(comparisonResult);
		}

		// TODO: This method could be made more efficient if we constrain our
		// search to like variables pre-emptively rather than filtering out bad
		// variable usage as we go

		// iterate over each position, branching some hypotheses and
		// removing implausible hypotheses
		for (int iPosition = 1; iPosition < condition.getChildren().size(); iPosition++) {

			ArrayList<LexicalResult> nextPositions = evaluateConstituent(condition.getChildren().get(
					iPosition));

			// the number of hypotheses can change after each positional
			// iteration
			ArrayList<ComparisonResult> survivingHypotheses = new ArrayList<ComparisonResult>(
					hypotheses.size());

			// try all combinations of hypotheses with the next set of positions
			for (int j = 0; j < hypotheses.size(); j++) {

				// the following is mostly a lot of syntactic garbage to do
				// something simple: make sure the position of the last word
				// from the previous match is actually less than the first word
				// from the current match

				// grab a hypothesis and the associated operands for each
				// iPosition
				ComparisonResult hypothesis = hypotheses.get(j);
				ArrayList<LexicalResult> lexicalOperands = hypothesis.getLexicalOperands();

				// get the position of the lexicon with the maximum position
				LexicalResult operandForCurrentPosition = lexicalOperands.get(iPosition - 1);
				ArrayList<TreeNode> hypothesisLexicons = operandForCurrentPosition.lexicons;
				TreeNode lastLexicon = hypothesisLexicons.get(hypothesisLexicons.size() - 1);

				// this could contain multiple indices if this is a phrase (e.g.
				// NP)
				ArrayList<Integer> nPrevPositions = lastLexicon.getTerminalIndices();
				int nPrevPosition = nPrevPositions.get(nPrevPositions.size() - 1);

				for (LexicalResult nextPosition : nextPositions) {

					// grab the position of the first lexicon in the next
					// position
					ArrayList<TreeNode> nextLexicons = nextPosition.lexicons;
					TreeNode nextLexicon = nextLexicons.get(nextLexicons.size() - 1);

					// this could contain multiple indices if this is a phrase
					// (e.g. NP)
					ArrayList<Integer> nNextPositions = nextLexicon.getTerminalIndices();
					int nNextPosition = nNextPositions.get(nNextPositions.size() - 1);

					assert nPrevPosition != nNextPosition : "Two lexical items claim to be in the same position";
					if (nPrevPosition > nNextPosition) {
						// this hypothesis is not compatible with this position,
						; // so do nothing
					} else {
						// we found a match for this position
						ComparisonResult survivingHypothesis = hypothesis.uniqueCopy();
						survivingHypothesis.lexicalOperands.add(nextPosition);
						survivingHypotheses.add(survivingHypothesis);
					}
				}

				hypotheses = survivingHypotheses;
			}
		}

		// all remaining hypotheses are true
		return hypotheses;
	}

	/**
	 * Resolves a feature node to a lexical node (provides information on POS,
	 * position, and lexicon).
	 * 
	 * @param node
	 * @return
	 */
	private ArrayList<LexicalResult> evaluateConstituent(TreeNode node) {
		String functionName = node.getValues().get(0);

		// make use of interned strings for == comparison
		assert functionName == functionName.intern() : "functionName not interned";
		assert node.getChildren().size() == 1 : "constituent functions take 1 argument";

		ArrayList<LexicalResult> results = evaluateFNode(node.getChildren().get(0));

		if (functionName == SOURCE_LEX) {

			resolveSourceLex(results);

		} else if (functionName == TARGET_LEX) {

			// find an alignment group from the sentence pair containing the
			// source index return the all target lexical items associated with
			// that index.
			
			resolveSourceLex(results);

			for (final LexicalResult result : results) {
				SentencePair pair = result.patternMatch.getSentencePair();
				Alignment alignment = pair.getAlignment();
				ArrayList<TreeNode> targetLexicons = new ArrayList<TreeNode>();
				
				for(final TreeNode sourceLex : result.lexicons) {
					targetLexicons.addAll(alignment.getTargetLexicons(pair, sourceLex));
				}
				
				// swap source lexicons for target lexicons
				// and store target lexicons as debug info
				result.intermediateLexicons = result.lexicons;
				result.lexicons = targetLexicons;
			}

		} else if (functionName == SOURCE_LEX_IHEAD) {

			// use the head mapping to get the immediate head
			// then run the source lex algorithm

		} else if (functionName == SOURCE_LEX_UHEAD) {

			//

		} else if (functionName == TARGET_LEX_UHEAD) {

			// then run the target lex algorithm

		} else {
			// TODO: Give trace for file and line number
			throw new RuntimeException("Unrecognized boolean function name: " + functionName);
		}
	}

	private void resolveSourceLex(ArrayList<LexicalResult> results) {
		// get the c-node associated with the f-node, then return the
		// lexical terminals
		for (final LexicalResult result : results) {
			PhiPlusMapping phiMapping = result.patternMatch.getSentencePair().getPhiPlusMapping();
			TreeNode cNode = phiMapping.getPhiInverseTop(result.patternMatch.getSentencePair(),
					result.fNode);
			result.lexicons = cNode.getLexicalTerminals();
		}
	}

	/**
	 * Resolves a sentence variable and f-structure pattern (possibly containing
	 * variables) to an fnode. This should probably be cached before evaluation
	 * 
	 * @param node
	 * @return A TreeNode FROM THE FEATURE STRUCTURE, not from the rules.
	 */
	private ArrayList<LexicalResult> evaluateFNode(TreeNode node) {
		String functionName = node.getValues().get(0);

		// make use of interned strings for == comparison
		assert functionName == functionName.intern() : "functionName not interned";

		if (functionName == FNODE) {

			// TODO: resolve variable names

			// only value is sentence variable
			// only child is the fnode pattern to match
			assert node.getValues().size() == 1 : "fnode takes one sentence variable as arg 1";
			assert node.getChildren().size() == 1 : "fnode takes a list of pattern elements as arg 2";
			String strSentenceVar = node.getValues().get(0);
			TreeNode patternNode = node.getChildren().get(0);

			Integer iSentenceIndex = this.sentencePatternVariables.get(strSentenceVar);
			if (iSentenceIndex == null)
				throw new RuntimeException("Sentence variable not found: " + strSentenceVar);
			int nSentenceIndex = iSentenceIndex;

			// now get the elicited examples that matched the pattern specified
			// for this sentence variable
			ArrayList<SentencePatternMatch> matchesForSentenceVariable = this.ruleMatches[nSentenceIndex];
			ArrayList<LexicalResult> results = new ArrayList<LexicalResult>();

			for (SentencePatternMatch sentencePatternMatch : matchesForSentenceVariable) {

				// ruleMatch.getFnodeCache().getValue(key);

				// the following code was taken from the matches() method

				// RESTRICTION: must match a role (labeled node) first
				// this should improve performance dramatically
				SmartTree fStruct = sentencePatternMatch.getSentencePair().getFeatureStructure();
				ArrayList<TreeNode> labeledNodes = fStruct.getLabeledNodes();
				for (TreeNode fNode : labeledNodes) {
					if (matchesRecursively(patternNode, fNode)) {

						// we found a matching fnode
						// and there was great rejoicing
						LexicalResult result = new LexicalResult();
						result.patternMatch = sentencePatternMatch;
						result.fNode = fNode;
						results.add(result);
					}
				}
			} // for ruleMatches

			if (results.size() == 0) {
				throw new RuntimeException("Invalid rule -- No matching node found for pattern: "
						+ patternNode.toString(false));
			} else {
				return results;
			}

		} else {

			// this could also be the name of one of our sentences!

			// TODO: Give trace for file and line number
			throw new RuntimeException("Unrecognized function name: " + functionName);
		}
	}

	public String[] getImplications() {
		return implications;
	}
}
