/**
 * The LETRAS Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 */
package edu.cmu.cs.lti.letras.featurebitmaps;

import java.util.ArrayList;
import java.util.HashMap;

/**
 * Allows a set of many FeatureGroups to be quickly searched for subsets.
 */
// TODO: use arraylist instead of hash (test performance difference)
// TODO: use smaller slices of bits when doing comparisons
public class FeatureGroupSet<T> {

	private class FeatureGroupNode {
		public final boolean isTerminal;
		public final long maskBits;
		public final int arrayIndex;

		public final HashMap<Long, FeatureGroupNode> nonTerminals;
		public final HashMap<Long, FeatureGroup<T>> terminals;

		public FeatureGroupNode(int arrayIndex, long maskBits) {

			this.isTerminal = (arrayIndex == 1);
			this.maskBits = maskBits;
			this.arrayIndex = arrayIndex;

			if (isTerminal) {
				terminals = new HashMap<Long, FeatureGroup<T>>();
				nonTerminals = null;
			} else {
				nonTerminals = new HashMap<Long, FeatureGroupNode>();
				terminals = null;
			}

		}

		public <X> boolean containsSubsetOf(FeatureGroup<X> superset) {
			if (arrayIndex < superset.compositeMask.length) {
				return ((superset.compositeMask[arrayIndex] & maskBits) == superset.compositeMask[arrayIndex]);
			} else {
				return (maskBits == 0x0000000000000000L);
			}
		}
	}

	private FeatureGroupNode root;

	public FeatureGroupSet() {
		root = new FeatureGroupNode(1, 0x0000000000000000L);
	}

	public void add(FeatureGroup<T> group) {

		// see if we need to deepen our tree and add a new root
		while (group.compositeMask.length > root.arrayIndex + 1) {

			FeatureGroupNode oldRoot = root;
			root = new FeatureGroupNode(root.arrayIndex + 1, 0x0000000000000000L);
			root.nonTerminals.put(oldRoot.maskBits, oldRoot);
		}

		// add the node
		addToExisting(group, root);
	}

	private void addToExisting(FeatureGroup<T> group, FeatureGroupNode parentNode) {

		if (parentNode.isTerminal) {

			assert parentNode.arrayIndex == 1;
			assert group.compositeMask.length >= parentNode.arrayIndex - 1;

			parentNode.terminals.put(group.compositeMask[0], group);
		} else {

			long childMask;
			if (parentNode.arrayIndex - 1 >= group.compositeMask.length) {
				childMask = 0x0000000000000000L;
			} else {
				childMask = group.compositeMask[parentNode.arrayIndex - 1];
			}

			FeatureGroupNode childNode = parentNode.nonTerminals.get(childMask);

			if (childNode == null) {
				childNode = new FeatureGroupNode(parentNode.arrayIndex - 1, childMask);
				parentNode.nonTerminals.put(childMask, childNode);
			}

			addToExisting(group, childNode);
		}
	}

	public <X> ArrayList<FeatureGroup<T>> getSubsetsOf(FeatureGroup<X> superset) {
		ArrayList<FeatureGroup<T>> matches = new ArrayList<FeatureGroup<T>>();
		findSubsets(superset, matches, root);
		return matches;
	}

	private <X> void findSubsets(FeatureGroup<X> superset, ArrayList<FeatureGroup<T>> matches,
			FeatureGroupNode parentNode) {
		if (parentNode.isTerminal) {
			for (FeatureGroup<T> group : parentNode.terminals.values()) {
				if (group.isSubsetOf(superset)) {
					matches.add(group);
				}
			}
		} else {
			for (FeatureGroupNode childNode : parentNode.nonTerminals.values()) {
				if (childNode.containsSubsetOf(superset)) {
					findSubsets(superset, matches, childNode);
				}
			}
		}
	}
}
