/**
 * The LETRAS Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 */
package edu.cmu.cs.lti.letras.corpus;

import java.text.ParseException;

import edu.cmu.cs.lti.letras.trees.SmartTree;

/**
 * Represents a single sentence pair that either will or has been presented to
 * the user during elicitation.
 */
public class SentencePair implements Comparable<SentencePair> {

	private final int id;
	private String[] eSentence;
	private String[] fSentence;
	private Alignment alignment;
	private String context;
	private String comment;

	// file:line pairs where components of this sentence pair originated (for
	// debugging)
	private String fstLine;
	private String cstLine;
	private String myLine;

	private SmartTree featureStructure;
	private SmartTree constituentStructure;
	private PhiPlusMapping phiPlusMapping;

	protected SentencePair(int id, String[] sentence, String[] sentence2, String alignments,
			String context, String comment, SmartTree featureStructure,
			SmartTree constituentStructure, PhiPlusMapping phiPlusMapping, String fstLine,
			String cstLine, String myLine) throws ParseException {

		// assert id != -1;
		assert sentence != null;
		assert sentence2 != null;
		assert alignments != null;
		assert context != null;
		assert comment != null;

		this.id = id;
		eSentence = sentence;
		fSentence = sentence2;
		this.alignment = new Alignment(alignments, eSentence.length, fSentence.length);
		this.context = context;
		this.comment = comment;
		this.featureStructure = featureStructure;
		this.constituentStructure = constituentStructure;
		this.phiPlusMapping = phiPlusMapping;

		this.fstLine = fstLine;
		this.cstLine = cstLine;
		this.myLine = myLine;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		else if (obj instanceof SentencePair) {
			SentencePair other = (SentencePair) obj;
			if (this.id == -1 || other.id == -1)
				throw new RuntimeException("ID not assigned!");
			return (this.id == other.id);
		} else {
			return false;
		}
	}

	public int hashCode() {
		return id;
	}

	public boolean hasEqualTranslation(SentencePair other) {
		if (this.fSentence.length != other.fSentence.length) {
			return false;
		} else {
			for (int i = 0; i < fSentence.length; i++) {
				if (!this.fSentence[i].equals(other.fSentence[i]))
					return false;
			}

			return true;
		}
	}

	public String[] getSourceSentence() {
		return eSentence;
	}

	public void setESentence(String[] sentence) {
		eSentence = sentence;
	}

	public String[] getTargetSentence() {
		return fSentence;
	}

	public void setFSentence(String[] sentence) {
		fSentence = sentence;
	}

	public Alignment getAlignment() {
		return alignment;
	}

	public void setAlignments(Alignment alignment) {
		this.alignment = alignment;
	}

	public String getContext() {
		return context;
	}

	public void setContext(String context) {
		this.context = context;
	}

	public String getComment() {
		return comment;
	}

	public void setComment(String comment) {
		this.comment = comment;
	}

	public int getId() {
		return id;
	}

	public SmartTree getFeatureStructure() {
		return featureStructure;
	}

	public SmartTree getConstituentStructure() {
		return constituentStructure;
	}

	public PhiPlusMapping getPhiPlusMapping() {
		return phiPlusMapping;
	}

	public void setPhiPlusMapping(PhiPlusMapping mapping) {
		this.phiPlusMapping = mapping;
	}

	public String getMyLine() {
		return myLine;
	}

	@Override
	public int compareTo(SentencePair other) {
		assert id != -1 : "No ID has been assigned.";
		return this.id - other.id;
	}

	public String serialize() {

		StringBuilder builder = new StringBuilder();

		// NOTE: myLine is not printed since it will change next time the file
		// is read in

		builder.append("newpair\n");
		if (fstLine != null)
			builder.append("fstline: " + fstLine);
		if (eSentence != null)
			builder.append("srcsent: " + eSentence + "\n");
		if (fSentence != null)
			builder.append("tgtsent: " + fSentence + "\n");
		if (alignment != null)
			builder.append("aligned: " + alignment + "\n");
		if (context != null)
			builder.append("context: " + context + "\n");
		if (comment != null)
			builder.append("comment: " + comment + "\n");
		if (featureStructure != null)
			builder.append("fstruct: " + featureStructure + "\n");
		if (cstLine != null)
			builder.append("cstline: " + cstLine + "\n");
		if (constituentStructure != null)
			builder.append("cstruct: " + constituentStructure + "\n");
		if (phiPlusMapping != null)
			builder.append("phiplus: " + phiPlusMapping + "\n");

		return builder.toString();

	}
}
