package edu.cmu.cs.lti.letras.corpus;

import info.jonclark.util.StringUtils;
import edu.cmu.cs.lti.letras.trees.TreeNode;

public class PhiPlusMapping {

	// private static final String PHI = "\u03D5";
	// private static final String PHI_INVERSE = "\u03D5\u207B\u00B9";
	public static final String PHI = "phi";
	public static final String PHI_INVERSE = "phiInv";
	public static final String HEAD = "h";
	
	public static final int IS_HEAD = Integer.MAX_VALUE;

	private final int[][] phiInverse;
	private final int[] phiInverseTop;
	private final int[] phiMapping;
	private final int[] headMapping;

	// public PhiPlusMapping(int[] phiMapping, int[] headMapping) {
	//
	// this.phiMapping = phiMapping;
	// this.headMapping = headMapping;
	//
	// // get the dimensions for the phiInverse
	// int nMaxFeatureIndex = 0;
	// for (int i = 0; i < phiMapping.length; i++)
	// nMaxFeatureIndex = Math.max(phiMapping[i], nMaxFeatureIndex);
	//		
	// this.phiInverse = new int[nMaxFeatureIndex][];
	//		
	// // TODO: create arraylists of features indices and build up each list
	// }

	/**
	 * @param phiInverse
	 *            A two-dimensional array in which the first dimension is the
	 *            zero-based index of the feature node and the second dimension
	 *            is a list of zero-based indices of constituent nodes
	 *            associated with the feature node specified by the first
	 *            dimension.
	 */
	public PhiPlusMapping(int[][] phiInverse, int[] headMapping) {

		this.phiInverse = phiInverse;
		this.headMapping = headMapping;
		
		// construct phiInverseTop
		this.phiInverseTop = new int[phiInverse.length];
		for (int i = 0; i < phiInverse.length; i++)
			for (int j = 0; j < phiInverse[i].length; j++)
				phiInverseTop[i] = Math.min(phiInverse[i][j], phiInverseTop[i]);

		// get the dimensions for the phiMapping
		int nMaxConstituentIndex = 0;
		for (int i = 0; i < phiInverse.length; i++)
			for (int j = 0; j < phiInverse[i].length; j++)
				nMaxConstituentIndex = Math.max(phiInverse[i][j], nMaxConstituentIndex);

		this.phiMapping = new int[nMaxConstituentIndex + 1];

		// initialize all elements of phi mapping to -1
		for (int i = 0; i < phiMapping.length; i++)
			phiMapping[i] = -1;

		// now set elements of phiInverse
		for (int i = 0; i < phiInverse.length; i++)
			for (int j = 0; j < phiInverse[i].length; j++)
				phiMapping[phiInverse[i][j]] = i;

		// assert phiMapping.length == headMapping.length : "Phi mapping (" +
		// phiMapping.length
		// + ") and head mapping (" + headMapping.length + ") have different
		// lengths: "
		// + this.toString();

	}

	/**
	 * Get the feature structure node associated with the specified constituent
	 * structure node.
	 * 
	 * @param constituentNodeIndex
	 * @return
	 */
	public int getPhi(int constituentNodeIndex) {
		return phiMapping[constituentNodeIndex];
	}
	
	public TreeNode getPhi(SentencePair pair, TreeNode constituentNode) {
		int fIndex = getPhi(constituentNode.getTreeIndex());
		TreeNode fNode = pair.getFeatureStructure().getByIndex(fIndex);
		return fNode;
	}

	/**
	 * Returns the constituent structure node index corresponding to the lexical
	 * head of the specified constituent node index.
	 * 
	 * @param constituentNodeIndex
	 * @return
	 */
	public int getImmediateHead(int constituentNodeIndex) {
		return headMapping[constituentNodeIndex];
	}
	
	public TreeNode getImmediateHead(SentencePair pair, TreeNode constituentNode) {
		int cIndex = getImmediateHead(constituentNode.getTreeIndex());
		TreeNode cNode = pair.getConstituentStructure().getByIndex(cIndex);
		return cNode;
	}
	
	public int getUltimateHead(int constituentNodeIndex) {
		int headIndex = headMapping[constituentNodeIndex];
		while(headIndex != IS_HEAD) {
			if(headIndex == -1)
				throw new RuntimeException("No head for node.");
			else
				headIndex = headMapping[constituentNodeIndex];
		}
		return headIndex;
	}
	
	public TreeNode getUltimateHead(SentencePair pair, TreeNode constituentNode) {
		int cIndex = getUltimateHead(constituentNode.getTreeIndex());
		TreeNode cNode = pair.getConstituentStructure().getByIndex(cIndex);
		return cNode;
	}

	/**
	 * Get the the constituent structure nodes associated with the specified
	 * feature structure node.
	 * 
	 * @param featureNodeIndex
	 * @return
	 */
	public int[] getPhiInverse(int featureNodeIndex) {
		return phiInverse[featureNodeIndex];
	}
	
	public TreeNode[] getPhiInverse(SentencePair pair, TreeNode featureNode) {
		int[] cIndices = getPhiInverse(featureNode.getTreeIndex());
		TreeNode[] cNodes = new TreeNode[cIndices.length];
		for(int i=0; i<cIndices.length; i++)
			cNodes[i] = pair.getConstituentStructure().getByIndex(cIndices[i]);
		return cNodes;
	}

	/**
	 * Gets the constituent structure node with the minimum index (that is, the
	 * mother of all the c-structure nodes) associated with the specified
	 * feature node.
	 * 
	 * @param featureNodeIndex
	 * @return
	 */
	public int getPhiInverseTop(int featureNodeIndex) {
		return phiInverseTop[featureNodeIndex];
	}
	
	public TreeNode getPhiInverseTop(SentencePair pair, TreeNode featureNode) {
		int cIndex = getPhiInverseTop(featureNode.getTreeIndex());
		TreeNode cNode = pair.getConstituentStructure().getByIndex(cIndex);
		return cNode;
	}

	public String toString() {

		StringBuilder builder = new StringBuilder();
		builder.append("[ ");
		builder.append("len(" + PHI + ")=" + phiMapping.length + "; ");
		builder.append("len(" + PHI_INVERSE + ")=" + phiInverse.length + "; ");

		// first print phi mapping
		for (int i = 0; i < phiMapping.length; i++) {
			if (phiMapping[i] != -1)
				builder.append(PHI + "(n" + (i + 1) + ")=f" + (phiMapping[i] + 1) + "; ");
		}

		// now give inverse mapping
		for (int i = 0; i < phiInverse.length; i++) {

			if (phiInverse[i].length > 0) {
				builder.append(PHI_INVERSE + "(f" + (i + 1) + ")={");
				for (int j = 0; j < phiInverse[i].length; j++) {
					if (j == phiInverse[i].length - 1)
						builder.append("n" + (phiInverse[i][j] + 1));
					else
						builder.append("n" + (phiInverse[i][j] + 1) + ",");
				}
				builder.append("}; ");
			}

		}

		// finally show head relationships
		for (int i = 0; i < headMapping.length; i++) {
			if (headMapping[i] != -1) {
				builder.append(HEAD + "(n" + (i + 1) + ")=n" + (headMapping[i] + 1) + "; ");
			}
		}

		builder.append("]");

		return builder.toString();
	}

	public static PhiPlusMapping deserialize(String serialized) {

		String strPhiLength = StringUtils.substringBetween(serialized, "len(" + PHI + ")=", ";");
		String strPhiInvLength = StringUtils.substringBetween(serialized, "len(" + PHI_INVERSE
				+ ")=", ";");
		int phiLength = Integer.parseInt(strPhiLength);
		int phiInvLength = Integer.parseInt(strPhiInvLength);

		int[] headMapping = new int[phiLength];
		int[][] phiInverse = new int[phiInvLength][];

		// initialize phiInverse with 0-length arrays
		for (int i = 0; i < phiInverse.length; i++) {
			phiInverse[i] = new int[0];
		}

		// initialize head mapping with -1
		for (int i = 0; i < headMapping.length; i++) {
			headMapping[i] = -1;
		}

		String strNoBrackets = StringUtils.substringBetween(serialized, "[", "]");
		String[] tokens = StringUtils.tokenize(strNoBrackets, "; ");

		for (final String token : tokens) {

			if (token.startsWith(PHI_INVERSE)) {

				String strFNodeIndex = StringUtils.substringBetween(token, "(f", ")");
				int fNodeIndex = Integer.parseInt(strFNodeIndex);
				String strCNodeList = StringUtils.substringBetween(token, "={", "}");
				String[] strCNodes = StringUtils.tokenize(strCNodeList, ",n");
				int[] cNodes = StringUtils.toIntArray(strCNodes);

				// System.out.println("PI-Mapping " + fNodeIndex + " ==> ");

				for (int i = 0; i < cNodes.length; i++)
					cNodes[i]--;

				phiInverse[fNodeIndex - 1] = cNodes;

			} else if (token.startsWith(PHI)) {
				;
			} else if (token.startsWith(HEAD)) {

				String strKey = StringUtils.substringBetween(token, "(n", ")");
				int key = Integer.parseInt(strKey);
				String strValue = StringUtils.substringAfter(token, "=n");
				int value = Integer.parseInt(strValue);

				// System.out.println("H-Mapping " + key + " ==> " + value);

				headMapping[key - 1] = value - 1;
			}
		}

		PhiPlusMapping mapping = new PhiPlusMapping(phiInverse, headMapping);
		return mapping;
	}

	/**
	 * For testing only...
	 */
	public static void main(String[] args) throws Exception {

		int[] headMapping = new int[] { -1, 3, -1, -1 };
		int[][] phiInverse = new int[][] { {}, { 0, 1, 3 }, {} };

		PhiPlusMapping map = new PhiPlusMapping(phiInverse, headMapping);
		System.out.println(map.toString());
		System.out.println(deserialize(map.toString()).toString());
	}
}
