/**
 * The LETRAS Project
 * Language Technologies Institute
 * School of Computer Science
 * (c) 2007 Carnegie Mellon University
 */
package edu.cmu.cs.lti.letras.navigation;

import info.jonclark.clientserver.ConnectionException;
import info.jonclark.clientserver.SimpleClient;
import info.jonclark.clientserver.SimpleServer;
import info.jonclark.log.LogUtils;
import info.jonclark.properties.PropertyUtils;
import info.jonclark.properties.SmartProperties;
import info.jonclark.util.ArrayUtils;
import info.jonclark.util.StringUtils;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Properties;
import java.util.Queue;
import java.util.logging.Logger;

import edu.cmu.cs.lti.letras.corpus.SentencePair;
import edu.cmu.cs.lti.letras.corpus.SentencePairFactory;
import edu.cmu.cs.lti.letras.corpus.SentenceTest;
import edu.cmu.cs.lti.letras.features.Feature;
import edu.cmu.cs.lti.letras.features.FeatureFactory;
import edu.cmu.cs.lti.letras.features.FeatureGroup;
import edu.cmu.cs.lti.letras.filtering.ImplicationFinder;
import edu.cmu.cs.lti.letras.filtering.SentenceAnalyzer;
import edu.cmu.cs.lti.letras.filtering.SentenceSelector;
import edu.cmu.cs.lti.letras.filtering.TestFilter;

/**
 * A navigator that applies implicational universals to find the most relevant
 * examples from the elicitation corpus.
 */
public class ElicitationNavigator {

	// TODO: keep a list of sentences the user has already completed
	// and NEVER remove one of these sentences
	//
	// Talk to Erik about implementing this on the client side
	// so that we don't annoy our poor user

	private final String elicitationCorpusFile;
	private final String universalsFile;
	private final String testsFile;

	// the minimum probability an implicational universal must have to be used
	// in calculations
	private final double minProb;

	// the four major components of the navigation system
	private final SentenceAnalyzer sentenceAnalyzer;
	private final ImplicationFinder implicationFinder;
	private final TestFilter testFilter;
	private final SentenceSelector sentenceSelector;

	private ArrayList<SentenceTest> remainingTests = new ArrayList<SentenceTest>();

	private static final Logger log = LogUtils.getLogger();

	// our connection to the elicitation tool
	private final SimpleServer server;
	private final int port;

	public ElicitationNavigator(Properties props) throws NumberFormatException, IOException {

		log.info("Initializing...");

		// extract configuration from properties file
		SmartProperties smartProps = new SmartProperties(props);
		this.elicitationCorpusFile = props.getProperty("elicitationCorpusFile");
		this.universalsFile = props.getProperty("universalsFile");
		this.testsFile = props.getProperty("testsFile");

		this.minProb = Double.parseDouble(props.getProperty("minProb"));

		this.port = smartProps.getPropertyInt("port");
		int nMaxConnections = smartProps.getPropertyInt("maxConnections");
		String serverEncoding = props.getProperty("serverEncoding");

		// load data files
		HashSet<SentencePair> sentencePairs = loadSentencePairs();
		ArrayList<FeatureGroup<Feature>> implicationalUniversals = loadImplicationalUniversals();
		ArrayList<FeatureGroup<SentenceTest>> tests = loadSentenceTests(sentencePairs);

		// initialize remaining tests with all
		for (FeatureGroup<SentenceTest> group : tests) {
			remainingTests.add(group.getValue());
		}

		// instantiate major navigation components
		this.sentenceAnalyzer = new SentenceAnalyzer(tests);
		this.implicationFinder = new ImplicationFinder(implicationalUniversals);
		this.testFilter = new TestFilter(tests);
		this.sentenceSelector = new SentenceSelector();

		// start a "feature finder" server for the elicitation tool to connect
		// to
		this.server = new SimpleServer(port, nMaxConnections, serverEncoding) {
			@Override
			public void handleClientRequest(SimpleClient client) {
				try {
					provideElicitationSentences(client);
				} catch (ConnectionException e) {
					log.severe("ConnectionException while communicating with client: "
							+ StringUtils.getStackTrace(e));
				} catch (IOException e) {
					log.severe("IOException while communicating with client: "
							+ StringUtils.getStackTrace(e));
				}
			}
		};
	}

	/**
	 * Starts the ElicitationNavigator in server mode for use with the
	 * Elicitation Tool
	 * 
	 * @throws IOException
	 */
	public void runServer() throws IOException {
		server.runServer();
		log.info("Server is now running on port " + port);
	}

	/**
	 * Run the ElicitationNavigator in console mode (as opposed to server mode)
	 * 
	 * @throws IOException
	 */
	public void runConsoleDemo() throws IOException {

		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		SentencePair sentencePair;

		while ((sentencePair = getNextSentencePair()) != null) {
			System.err.println("Context: " + sentencePair.getContext());
			System.err.println("Please translate: "
					+ StringUtils.untokenize(sentencePair.getESentence()));
			System.err.print("Translation: ");

			String line = in.readLine();
			sentencePair.setFSentence(StringUtils.tokenize(line));
			submitElicitedSentencePair(sentencePair);
		}
	}

	/**
	 * A thin wrapper around the sentence selector that returns the next
	 * sentence (hopefully the current most valuable sentence) for elicitation.
	 * 
	 * @return
	 */
	public SentencePair getNextSentencePair() {
		log.info("There are now " + remainingTests.size() + " tests remaining.");

		SentencePair sentencePair = sentenceSelector.getNextSentencePair(remainingTests);
		return sentencePair;
	}

	/**
	 * Pass along a sentence elicited from the user to the various components of
	 * the system.
	 * 
	 * @param elicitedSentencePair
	 */
	public void submitElicitedSentencePair(SentencePair elicitedSentencePair) {

		// analyze sentence
		sentenceAnalyzer.addElicitedSentencePair(elicitedSentencePair);
		ArrayList<Feature> discoveredFeatures = sentenceAnalyzer.getDiscoveredFeatures();

		// debug output
		StringBuilder discoveredBuilder = new StringBuilder();
		int i = 0;
		for (Feature feature : discoveredFeatures)
			discoveredBuilder.append(++i + ". " + feature.getName() + "\n");
		log.info("The following features were discovered:\n" + discoveredBuilder.toString());

		// see if we can gather any knowledge from out IU's
		implicationFinder.addDiscoveredFeatures(discoveredFeatures);
		ArrayList<Feature> impliedFeatures = implicationFinder.getImplications();

		// debug output
		StringBuilder impliedBuilder = new StringBuilder();
		i = 0;
		for (Feature feature : impliedFeatures)
			impliedBuilder.append(++i + ". " + feature.getName() + "\n");
		log.info("The following features were found in the implicational universals:\n"
				+ impliedBuilder.toString());

		// now combine all of our known features
		ArrayList<Feature> allKnownFeatures = new ArrayList<Feature>();
		allKnownFeatures.addAll(discoveredFeatures);
		allKnownFeatures.addAll(impliedFeatures);

		// turn our eliminated features into the features that we still care
		// about
		FeatureGroup<Void> allKnownFeaturesGroup = new FeatureGroup<Void>(allKnownFeatures, null);
		FeatureGroup<Void> unknownFeaturesGroup = allKnownFeaturesGroup.complement();

		// update how many tests are remaining
		remainingTests = testFilter.getRemainingTests(unknownFeaturesGroup);
	}

	/**
	 * This method is the main point of integration between the server, client,
	 * and all navigational components.
	 * 
	 * @param client
	 * @throws ConnectionException
	 * @throws IOException
	 */
	public void provideElicitationSentences(SimpleClient client) throws ConnectionException,
			IOException {

		log.info("New incoming connection from " + client.getHost());

		BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
		SentencePair outgoingSentencePair;
		SentencePair incomingSentencePair = null;

		// the sentences that we have sent to the client, but not yet received
		// back.
		Queue<SentencePair> outstandingSentences = new LinkedList<SentencePair>();

		while ((outgoingSentencePair = getNextSentencePair()) != null
				|| incomingSentencePair != null) {

			if (outgoingSentencePair != null) {
				log.info("Requesting translation of sentence: '"
						+ StringUtils.untokenize(outgoingSentencePair.getESentence())
						+ "' With context: '" + outgoingSentencePair.getContext() + "'");

				client.sendMessage("newpair");
				client.sendMessage("srcsent: "
						+ StringUtils.untokenize(outgoingSentencePair.getESentence()));
				client.sendMessage("tgtsent: "
						+ StringUtils.untokenize(outgoingSentencePair.getFSentence()));
				client.sendMessage("aligned: " + outgoingSentencePair.getAlignments());
				client.sendMessage("context: " + outgoingSentencePair.getContext());
				client.sendMessage("comment: " + outgoingSentencePair.getComment());
				client.sendMessage("");
				client.sendMessage("");

				outstandingSentences.add(outgoingSentencePair);
			}

			// get the group back from the client, do some processing, and send
			// more
			incomingSentencePair = null;
			String clientLine;
			while ((clientLine = client.getMessage()) != null && !(clientLine.equals(""))) {

				String value = StringUtils.substringAfter(clientLine, ":");
				value = value.trim();

				if (clientLine.startsWith("newpair")) {
					incomingSentencePair = outstandingSentences.poll();
				} else if (clientLine.startsWith("srcsent")) {
					incomingSentencePair.setESentence(StringUtils.tokenize(value));
				} else if (clientLine.startsWith("tgtsent")) {
					incomingSentencePair.setFSentence(StringUtils.tokenize(value));
				} else if (clientLine.startsWith("aligned")) {
					incomingSentencePair.setAlignments(value);
				} else if (clientLine.startsWith("context")) {
					incomingSentencePair.setContext(value);
				} else if (clientLine.startsWith("comment")) {
					incomingSentencePair.setComment(value);
				} else {
					log.warning("Unknown tag in response from elicitation tool: " + clientLine);
				}

			} // end while
			
			// kill off extra newline
			String extraNewline = client.getMessage();
			assert extraNewline.equals("");

			if (incomingSentencePair != null) {
				log.info("For sentence '"
						+ StringUtils.untokenize(outgoingSentencePair.getESentence())
						+ "' got translation '"
						+ StringUtils.untokenize(outgoingSentencePair.getFSentence()) + "'");
				submitElicitedSentencePair(outgoingSentencePair);
			} else {
				log.info("No sentence received this iteration.");
			}
		}

		in.close();

		client.sendMessage("endcorpus");
		log.info("Done. Terminating ElicitationNavigator.");
		System.exit(0);
	}

	private HashSet<SentencePair> loadSentencePairs() throws IOException {
		HashSet<SentencePair> sentencePairs = new HashSet<SentencePair>();

		// TODO: Respect the encoding specified at the top of the file
		// TODO: Handle other possible tags besides newpair -- read header info

		BufferedReader in = new BufferedReader(new FileReader(elicitationCorpusFile));

		String corpusLine;
		while ((corpusLine = in.readLine()) != null && !(corpusLine.equals("newpair")))
			;

		// read file to exhaustion
		while (corpusLine != null) {

			int id = -1;
			String[] eSentence = null;
			String[] fSentence = null;
			String alignments = null;
			String context = null;
			String comment = null;
			while ((corpusLine = in.readLine()) != null && !(corpusLine.equals(""))) {

				String value = StringUtils.substringAfter(corpusLine, ":");
				value = value.trim();

				// System.out.println("reading: " + corpusLine);

				if (corpusLine.startsWith("newpair")) {
					continue;
				} else if (corpusLine.startsWith("sentid#")) {
					id = Integer.parseInt(value);
				} else if (corpusLine.startsWith("srcsent")) {
					eSentence = StringUtils.tokenize(value);
				} else if (corpusLine.startsWith("tgtsent")) {
					fSentence = StringUtils.tokenize(value);
				} else if (corpusLine.startsWith("aligned")) {
					alignments = value;
				} else if (corpusLine.startsWith("context")) {
					context = value;
				} else if (corpusLine.startsWith("comment")) {
					comment = value;
				} else {
					log.warning("Unknown tag in elicitation corpus: " + corpusLine);
				}
			}

			// any of these being null means we have an incomplete entry (or the
			// file ended in the middle of an entry)
			assert id != -1;
			assert eSentence != null;
			assert fSentence != null;
			assert alignments != null;
			assert context != null;
			assert comment != null;

			SentencePair sentencePair = SentencePairFactory.getInstance(id, eSentence, fSentence,
					alignments, context, comment);
			sentencePairs.add(sentencePair);
		} // end while corpusLine != null

		in.close();

		return sentencePairs;
	}

	private ArrayList<FeatureGroup<Feature>> loadImplicationalUniversals()
			throws NumberFormatException, IOException {

		ArrayList<FeatureGroup<Feature>> implicationalUniverals = new ArrayList<FeatureGroup<Feature>>();

		BufferedReader in = new BufferedReader(new FileReader(universalsFile));

		String line;
		while ((line = in.readLine()) != null) {

			String[] tokens = StringUtils.split(line, "\t", 11);

			double prob = Double.parseDouble(tokens[0]);
			if (prob > minProb) {
				String strTrigger = tokens[1] + ": " + tokens[3];
				String strImplication = tokens[2] + ": " + tokens[4];

				Feature trigger = FeatureFactory.getInstance(strTrigger);
				Feature implication = FeatureFactory.getInstance(strImplication);

				implicationalUniverals.add(new FeatureGroup<Feature>(trigger, implication));
			}
		}

		return implicationalUniverals;
	}

	/**
	 * Load feature groups into an array list, enclosing each in a feature group
	 * that represents all features that either trigger the test or are fired by
	 * the test.
	 * 
	 * @param sentencePairs
	 * @return
	 * @throws IOException
	 */
	private ArrayList<FeatureGroup<SentenceTest>> loadSentenceTests(
			HashSet<SentencePair> sentencePairs) throws IOException {
		ArrayList<FeatureGroup<SentenceTest>> sentenceTests = new ArrayList<FeatureGroup<SentenceTest>>();

		BufferedReader in = new BufferedReader(new FileReader(testsFile));

		// TODO: Handle other possible tags besides newtest
		String testLine;
		while ((testLine = in.readLine()) != null && !testLine.equals("newtest"))
			;

		while (testLine != null) {

			// send a group of sentences (1 for now)
			ArrayList<Feature> ifEqual = null;
			ArrayList<Feature> ifNotEqual = null;
			int[] required = null;
			while ((testLine = in.readLine()) != null && !(testLine.equals(""))) {

				String value = StringUtils.substringAfter(testLine, ":");
				value = value.trim();

				// System.err.println("reading: " + testLine);

				if (testLine.startsWith("newtest")) {
					continue;
				} else if (testLine.startsWith("ifequal")) {

					String[] features = StringUtils.tokenize(value, ",");
					ifEqual = new ArrayList<Feature>(features.length);
					for (String feature : features) {
						ifEqual.add(FeatureFactory.getInstance(feature.trim()));
					}

				} else if (testLine.startsWith("ifnoteq")) {

					String[] features = StringUtils.tokenize(value, ",");
					ifNotEqual = new ArrayList<Feature>(features.length);
					for (String feature : features) {
						ifNotEqual.add(FeatureFactory.getInstance(feature.trim()));
					}

				} else if (testLine.startsWith("require")) {
					required = StringUtils.toIntArray(StringUtils.tokenize(value));
				} else {
					log.warning("Unknown tag in elicitation corpus: " + testLine);
				}
			}

			// any of these being null means we have an incomplete entry (or the
			// file ended in the middle of an entry)
			assert ifEqual != null;
			assert ifNotEqual != null;
			assert required != null;

			SentenceTest test = new SentenceTest(required, ifEqual, ifNotEqual);

			ArrayList<Feature> allFeatures = new ArrayList<Feature>(ifEqual.size()
					+ ifNotEqual.size());
			allFeatures.addAll(ifEqual);
			allFeatures.addAll(ifNotEqual);

			FeatureGroup<SentenceTest> node = new FeatureGroup<SentenceTest>(allFeatures, test);
			sentenceTests.add(node);
		} // end while corpusLine != null

		in.close();

		return sentenceTests;
	}

	public static void main(String[] args) throws Exception {

		if (args.length != 1) {
			System.err.println("Usage: program <navigator_properties_file>");
			System.exit(1);
		}

		Properties props = PropertyUtils.getProperties(args[0]);
		ElicitationNavigator nav = new ElicitationNavigator(props);
		assert nav != null;

		if (ArrayUtils.unsortedArrayContains(args, "--console")) {
			nav.runConsoleDemo();
		} else {
			nav.runServer();
		}
		// wait until the user kills this process
		while (true)
			Thread.sleep(Long.MAX_VALUE);
	}
}
